import axios from 'axios'

const API_BASE_URL = import.meta.env.VITE_API_URL || '/api/v1'

const api = axios.create({
  baseURL: API_BASE_URL,
  timeout: 15000,
  headers: {
    'Content-Type': 'application/json',
    'Accept': 'application/json',
  },
})

// Add token to requests
api.interceptors.request.use((config) => {
  const token = localStorage.getItem('token')
  if (token) config.headers.Authorization = `Bearer ${token}`
  return config
})

// Handle 401 responses
api.interceptors.response.use(
  (response) => response,
  (error) => {
    if (error.response?.status === 401) {
      localStorage.removeItem('token')
      localStorage.removeItem('user')
      if (!window.location.pathname.includes('/login')) {
        window.location.href = '/login'
      }
    }
    return Promise.reject(error)
  }
)

// ─── Auth ───────────────────────────────────────────────────
export const login = (data) => api.post('/auth/login', data)
export const register = (data) => api.post('/auth/register', data)
export const logout = () => api.post('/auth/logout')
export const getProfile = () => api.get('/auth/profile')
export const updateProfile = (data) => api.put('/auth/profile', data)
export const changePassword = (data) => api.put('/auth/password', data)

// ─── Products ───────────────────────────────────────────────
export const getProducts = (params) => api.get('/products', { params })
export const getProduct = (slug) => api.get(`/products/${slug}`)
export const searchProducts = (query, params = {}) => api.get('/products', { params: { search: query, ...params } })

// ─── Categories ─────────────────────────────────────────────
export const getCategories = (params) => api.get('/categories', { params })
export const getCategory = (slug) => api.get(`/categories/${slug}`)

// ─── Cart ────────────────────────────────────────────────────
export const getCart = () => api.get('/cart')
export const addCartItem = (productId, quantity = 1) => api.post('/cart', { product_id: productId, quantity })
export const updateCartItem = (id, quantity) => api.put(`/cart/${id}`, { quantity })
export const removeCartItem = (id) => api.delete(`/cart/${id}`)
export const clearCartApi = () => api.delete('/cart')

// ─── Wishlist ────────────────────────────────────────────────
export const getWishlist = () => api.get('/wishlist')
export const addWishlistItem = (productId) => api.post('/wishlist', { product_id: productId })
export const removeWishlistItem = (productId) => api.delete(`/wishlist/${productId}`)

// ─── Compare ────────────────────────────────────────────────
export const getCompare = () => api.get('/compare')
export const addToCompare = (productId) => api.post('/compare', { product_id: productId })
export const removeFromCompare = (productId) => api.delete(`/compare/${productId}`)
export const clearCompare = () => api.delete('/compare')

// ─── Orders ─────────────────────────────────────────────────
export const getOrders = (params) => api.get('/orders', { params })
export const getOrder = (id) => api.get(`/orders/${id}`)
export const placeOrder = (data) => api.post('/checkout', data)
export const cancelOrder = (id) => api.put(`/orders/${id}/cancel`)

// ─── Reviews ────────────────────────────────────────────────
export const getReviews = (productId) => api.get(`/products/${productId}/reviews`)
export const addReview = (productId, data) => api.post(`/products/${productId}/reviews`, data)

// ─── Settings / CMS ─────────────────────────────────────────
export const getSettings = () => api.get('/settings')
export const getSliders = () => api.get('/sliders')
export const getBanners = () => api.get('/banners')

// ─── Vendors ────────────────────────────────────────────────
export const getVendors = (params) => api.get('/vendors', { params })
export const getVendor = (id) => api.get(`/vendors/${id}`)

// ─── Newsletter ──────────────────────────────────────────────
export const subscribeNewsletter = (email) => api.post('/newsletter/subscribe', { email })

// ─── Contact ────────────────────────────────────────────────
export const sendContact = (data) => api.post('/contact', data)

// ─── Wholesale ──────────────────────────────────────────────
export const applyWholesale = (data) => api.post('/wholesale/apply', data)
export const getWholesaleStatus = () => api.get('/wholesale/status')
export const getWholesaleOrders = (params) => api.get('/wholesale/orders', { params })

// ─── Coupon ──────────────────────────────────────────────────
export const applyCoupon = (code) => api.post('/coupons/apply', { code })
export const removeCoupon = () => api.delete('/coupons/remove')

// ─── Password Reset ──────────────────────────────────────────
export const forgotPassword = (data) => api.post('/auth/forgot-password', data)
export const resetPassword = (data) => api.post('/auth/reset-password', data)

// ─── Stripe Payment ──────────────────────────────────────────
export const createStripeSession = (data) => api.post('/payment/stripe-session', data)
export const verifyStripeSession = (sessionId) => api.post('/payment/stripe-verify', { session_id: sessionId })

export default api
