import { Link } from 'react-router-dom'
import { useWishlist } from '../context/WishlistContext'
import { useCart } from '../context/CartContext'
import Loading from '../components/ui/Loading'
import toast from 'react-hot-toast'

const Wishlist = () => {
  const { wishlist, wishlistLoading, removeFromWishlist } = useWishlist()
  const { addToCart } = useCart()

  const handleAddToCart = async (productId) => {
    try {
      await addToCart(productId, 1)
      toast.success('Added to cart!')
    } catch {
      toast.error('Failed to add to cart')
    }
  }

  const handleRemove = async (productId) => {
    try {
      await removeFromWishlist(productId)
      toast.success('Removed from wishlist')
    } catch {
      toast.error('Failed to remove from wishlist')
    }
  }

  return (
    <main className="main">
      {/* Breadcrumb */}
      <div className="section-box">
        <div className="breadcrumbs-div">
          <div className="container">
            <ul className="breadcrumb">
              <li>
                <Link className="font-xs color-gray-1000" to="/">Home</Link>
              </li>
              <li>
                <Link className="font-xs color-gray-500" to="/shop">Shop</Link>
              </li>
              <li>
                <span className="font-xs color-gray-500">Wishlist</span>
              </li>
            </ul>
          </div>
        </div>
      </div>

      {/* Wishlist Section */}
      <section className="section-box shop-template">
        <div className="container">
          {wishlistLoading ? (
            <Loading height={300} />
          ) : (
            <div className="box-wishlist">
              {/* Table Header */}
              <div className="head-wishlist">
                <div className="item-wishlist">
                  <div className="wishlist-cb">
                    <input className="cb-layout cb-all" type="checkbox" />
                  </div>
                  <div className="wishlist-product">
                    <span className="font-md-bold color-brand-3">Product</span>
                  </div>
                  <div className="wishlist-price">
                    <span className="font-md-bold color-brand-3">Price</span>
                  </div>
                  <div className="wishlist-status">
                    <span className="font-md-bold color-brand-3">Stock status</span>
                  </div>
                  <div className="wishlist-action">
                    <span className="font-md-bold color-brand-3">Action</span>
                  </div>
                  <div className="wishlist-remove">
                    <span className="font-md-bold color-brand-3">Remove</span>
                  </div>
                </div>
              </div>

              {/* Table Body */}
              <div className="content-wishlist">
                {wishlist.length === 0 ? (
                  <p className="text-center color-gray-500 font-sm py-30">
                    Your wishlist is empty.{' '}
                    <Link className="color-brand-3" to="/shop">Browse products</Link>
                  </p>
                ) : (
                  wishlist.map(item => {
                    const product = item.product || item
                    const productId = product.id || item.product_id
                    const price = parseFloat(product.sale_price || product.price || 0)
                    const image = product.primary_image?.url || product.image_url || '/assets/imgs/no-image.png'
                    const inStock = product.stock > 0

                    return (
                      <div className="item-wishlist" key={item.id || productId}>
                        {/* Checkbox */}
                        <div className="wishlist-cb">
                          <input className="cb-layout cb-select" type="checkbox" />
                        </div>

                        {/* Product */}
                        <div className="wishlist-product">
                          <div className="product-wishlist">
                            <div className="product-image">
                              <Link to={`/product/${product.slug}`}>
                                <img src={image} alt={product.name} />
                              </Link>
                            </div>
                            <div className="product-info">
                              <Link to={`/product/${product.slug}`}>
                                <h6 className="color-brand-3">{product.name}</h6>
                              </Link>
                              <div className="rating">
                                {[1, 2, 3, 4, 5].map(star => (
                                  <img
                                    key={star}
                                    src={
                                      star <= Math.round(product.rating || 0)
                                        ? '/assets/imgs/template/icons/star.svg'
                                        : '/assets/imgs/template/icons/star-gray.svg'
                                    }
                                    alt="star"
                                  />
                                ))}
                                {product.reviews_count != null && (
                                  <span className="font-xs color-gray-500"> ({product.reviews_count})</span>
                                )}
                              </div>
                            </div>
                          </div>
                        </div>

                        {/* Price */}
                        <div className="wishlist-price">
                          <h4 className="color-brand-3">${price.toFixed(2)}</h4>
                          {product.sale_price && product.price && (
                            <span
                              className="font-xs color-gray-500"
                              style={{ textDecoration: 'line-through', display: 'block' }}
                            >
                              ${parseFloat(product.price).toFixed(2)}
                            </span>
                          )}
                        </div>

                        {/* Stock Status */}
                        <div className="wishlist-status">
                          <span className="btn btn-gray font-md-bold color-brand-3">
                            {inStock ? 'In Stock' : 'Out of Stock'}
                          </span>
                        </div>

                        {/* Add to Cart */}
                        <div className="wishlist-action">
                          <button
                            className="btn btn-cart font-sm-bold"
                            onClick={() => handleAddToCart(productId)}
                            disabled={!inStock}
                          >
                            Add to Cart
                          </button>
                        </div>

                        {/* Remove */}
                        <div className="wishlist-remove">
                          <button
                            className="btn btn-delete"
                            onClick={() => handleRemove(productId)}
                            aria-label="Remove from wishlist"
                          ></button>
                        </div>
                      </div>
                    )
                  })
                )}
              </div>
            </div>
          )}
        </div>
      </section>
    </main>
  )
}

export default Wishlist
