import { useState, useEffect } from 'react'
import { Link } from 'react-router-dom'
import { useAuth } from '../context/AuthContext'
import { getWholesaleStatus, getWholesaleOrders } from '../services/api'
import Loading from '../components/ui/Loading'
import Pagination from '../components/ui/Pagination'

const STATUS_BADGE = {
  pending:    { cls: 'pending',    label: 'Pending' },
  processing: { cls: 'processing', label: 'Processing' },
  shipped:    { cls: 'shipped',    label: 'Shipped' },
  delivered:  { cls: 'delivered',  label: 'Delivered' },
  completed:  { cls: 'shipped',    label: 'Completed' },
  cancelled:  { cls: 'cancelled',  label: 'Cancelled' },
}

const formatCurrency = (amount) =>
  '$' + parseFloat(amount || 0).toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })

const WholesaleDashboard = () => {
  const { user, isLoggedIn } = useAuth()

  const [status, setStatus] = useState(null)
  const [orders, setOrders] = useState([])
  const [ordersMeta, setOrdersMeta] = useState({ current_page: 1, last_page: 1, total: 0 })
  const [loading, setLoading] = useState(true)
  const [ordersLoading, setOrdersLoading] = useState(false)
  const [tab, setTab] = useState('overview')
  const [ordersPage, setOrdersPage] = useState(1)

  // Load wholesale status on mount
  useEffect(() => {
    if (!isLoggedIn) {
      setLoading(false)
      return
    }
    setLoading(true)
    getWholesaleStatus()
      .then(res => {
        // API returns { status, application } directly
        const d = res.data
        if (d && d.status) setStatus(d)
      })
      .catch(() => {})
      .finally(() => setLoading(false))
  }, [isLoggedIn])

  // Load orders whenever tab is orders or page changes
  useEffect(() => {
    if (!isLoggedIn || !status || status.status !== 'approved') return
    setOrdersLoading(true)
    getWholesaleOrders({ page: ordersPage, per_page: 10 })
      .then(res => {
        // Wholesale orders are paginated: { data: [...], current_page, last_page, total }
        const d = res.data
        setOrders(d?.data || [])
        setOrdersMeta({
          current_page: d?.current_page || 1,
          last_page: d?.last_page || 1,
          total: d?.total || 0,
        })
      })
      .catch(() => {})
      .finally(() => setOrdersLoading(false))
  }, [isLoggedIn, status, ordersPage])

  if (loading) return <Loading height={400} />

  // Not logged in
  if (!isLoggedIn) {
    return (
      <section className="section-box shop-template mt-30">
        <div className="container">
          <div className="not-wholesale" style={{ textAlign: 'center', padding: '60px 20px' }}>
            <i className="fas fa-lock" style={{ fontSize: 80, color: '#ddd', marginBottom: 20, display: 'block' }}></i>
            <h3 className="color-gray-900 mb-15">Sign In Required</h3>
            <p className="font-md color-gray-500 mb-30">
              Please sign in to access your wholesale dashboard.
            </p>
            <Link to="/login" className="btn btn-buy mr-15">Sign In</Link>
            <Link to="/register" className="btn btn-brand-2">Create Account</Link>
          </div>
        </div>
      </section>
    )
  }

  // No application
  if (!status) {
    return (
      <section className="section-box shop-template mt-30">
        <div className="container">
          <div className="not-wholesale" style={{ textAlign: 'center', padding: '60px 20px' }}>
            <i className="fas fa-clipboard-list" style={{ fontSize: 80, color: '#ddd', marginBottom: 20, display: 'block' }}></i>
            <h3 className="color-gray-900 mb-15">No Wholesale Application</h3>
            <p className="font-md color-gray-500 mb-30">
              You haven't applied for a wholesale account yet. Apply today to unlock exclusive B2B pricing.
            </p>
            <Link to="/wholesale" className="btn btn-buy">Apply for Wholesale</Link>
          </div>
        </div>
      </section>
    )
  }

  // Application pending
  if (status.status === 'pending') {
    return (
      <section className="section-box shop-template mt-30">
        <div className="container" style={{ maxWidth: 700 }}>
          <div style={{ background: 'linear-gradient(135deg, #fff9e6, #fff3cd)', border: '1px solid #ffc107', borderRadius: 15, padding: 50, textAlign: 'center' }}>
            <div style={{ fontSize: 64, marginBottom: 20 }}>⏳</div>
            <h3 className="color-gray-900 mb-15">Application Under Review</h3>
            <p className="font-md color-gray-500 mb-15">
              Your wholesale application was submitted on{' '}
              <strong>{new Date(status.created_at).toLocaleDateString()}</strong>.
            </p>
            <p className="font-sm color-gray-500 mb-30">
              We typically review applications within 2&ndash;3 business days. You'll receive an email once a decision is made.
            </p>
            <Link to="/" className="btn btn-brand-2 mr-15">Continue Shopping</Link>
            <Link to="/contact" className="btn btn-border">Contact Support</Link>
          </div>
        </div>
      </section>
    )
  }

  // Application rejected
  if (status.status === 'rejected') {
    return (
      <section className="section-box shop-template mt-30">
        <div className="container" style={{ maxWidth: 700 }}>
          <div style={{ background: 'linear-gradient(135deg, #ffebee, #f8d7da)', border: '1px solid #dc3545', borderRadius: 15, padding: 50, textAlign: 'center' }}>
            <div style={{ fontSize: 64, marginBottom: 20 }}>✗</div>
            <h3 className="color-gray-900 mb-15">Application Not Approved</h3>
            {status.rejection_reason && (
              <p className="font-md color-gray-500 mb-15">
                <strong>Reason:</strong> {status.rejection_reason}
              </p>
            )}
            <p className="font-sm color-gray-500 mb-30">
              Please contact our support team for more information or to discuss your application.
            </p>
            <Link to="/contact" className="btn btn-buy mr-15">Contact Support</Link>
            <Link to="/wholesale" className="btn btn-brand-2">Reapply</Link>
          </div>
        </div>
      </section>
    )
  }

  // Approved: full dashboard
  const totalOrders = ordersMeta.total
  const totalSpent = orders.reduce((sum, o) => sum + parseFloat(o.total_amount || 0), 0)
  const pendingOrders = orders.filter(o => o.status === 'pending').length
  const completedOrders = orders.filter(o => ['delivered', 'completed'].includes(o.status)).length

  return (
    <>
      {/* Breadcrumb */}
      <div className="section-box">
        <div className="breadcrumbs-div">
          <div className="container">
            <ul className="breadcrumb">
              <li><Link className="font-xs color-gray-1000" to="/">Home</Link></li>
              <li><span className="font-xs color-gray-500">Wholesale Dashboard</span></li>
            </ul>
          </div>
        </div>
      </div>

      <section className="section-box shop-template mt-30">
        <div className="container box-account-template">

          {/* Welcome Banner */}
          <div style={{
            background: 'linear-gradient(135deg, #253D4E 0%, #3BB77E 100%)',
            borderRadius: 15,
            padding: '35px 40px',
            color: '#fff',
            marginBottom: 30,
          }}>
            <div className="row align-items-center">
              <div className="col-lg-8">
                <h2 style={{ color: '#fff', marginBottom: 5 }}>
                  Welcome back, {user?.name || status.contact_name}!
                </h2>
                <p style={{ color: 'rgba(255,255,255,0.8)', marginBottom: 10 }}>{user?.email}</p>
                <span style={{ background: 'rgba(255,255,255,0.2)', color: '#fff', padding: '4px 14px', borderRadius: 20, fontSize: 13, fontWeight: 600, marginRight: 10 }}>
                  Wholesale Account
                </span>
                {status.discount_percentage && (
                  <span style={{ background: '#3BB77E', border: '1px solid rgba(255,255,255,0.3)', color: '#fff', padding: '4px 14px', borderRadius: 20, fontSize: 13, fontWeight: 600 }}>
                    {status.discount_percentage}% Discount Active
                  </span>
                )}
              </div>
              <div className="col-lg-4 text-lg-end mt-20 mt-lg-0">
                <Link to="/shop" className="btn font-sm-bold" style={{ background: '#fff', color: '#3BB77E', padding: '10px 24px', borderRadius: 8 }}>
                  Browse Products
                </Link>
              </div>
            </div>
          </div>

          {/* Stats Cards */}
          <div className="row mb-30">
            {[
              { label: 'Total Orders', value: totalOrders, cls: 'primary', icon: 'fa-shopping-bag' },
              { label: 'Total Spent', value: formatCurrency(totalSpent), cls: 'success', icon: 'fa-dollar-sign' },
              { label: 'Pending Orders', value: pendingOrders, cls: 'warning', icon: 'fa-clock' },
              { label: 'Completed Orders', value: completedOrders, cls: 'info', icon: 'fa-check-circle' },
            ].map(stat => (
              <div key={stat.label} className="col-lg-3 col-md-6 mb-20">
                <div className={`stat-card ${stat.cls}`} style={{
                  background: '#fff',
                  border: '1px solid #eee',
                  borderRadius: 12,
                  padding: 24,
                  textAlign: 'center',
                  boxShadow: '0 2px 10px rgba(0,0,0,0.05)',
                }}>
                  <i className={`fas ${stat.icon}`} style={{ fontSize: 28, color: '#3BB77E', marginBottom: 10, display: 'block' }}></i>
                  <h2 style={{ fontSize: 28, fontWeight: 700, color: '#253D4E', marginBottom: 5 }}>{stat.value}</h2>
                  <p className="font-sm color-gray-500 mb-0">{stat.label}</p>
                </div>
              </div>
            ))}
          </div>

          {/* Tabs */}
          <div className="box-tabs mb-100">
            <ul className="nav nav-tabs nav-tabs-account" style={{ borderBottom: '2px solid #eee', listStyle: 'none', padding: 0, display: 'flex', gap: 0, margin: 0 }}>
              {[
                { key: 'overview', label: 'Overview' },
                { key: 'orders', label: 'Orders' },
                { key: 'account', label: 'Account Info' },
              ].map(t => (
                <li key={t.key} style={{ marginBottom: -2 }}>
                  <button
                    className={`font-sm-bold${tab === t.key ? ' color-brand-3' : ' color-gray-900'}`}
                    onClick={() => setTab(t.key)}
                    style={{
                      background: 'none',
                      border: 'none',
                      borderBottom: tab === t.key ? '2px solid #3BB77E' : '2px solid transparent',
                      padding: '10px 20px',
                      cursor: 'pointer',
                      fontWeight: tab === t.key ? 700 : 400,
                      transition: 'all 0.2s',
                    }}
                  >
                    {t.label}
                  </button>
                </li>
              ))}
            </ul>

            <div className="border-bottom mt-0 mb-40" style={{ borderTop: 'none' }}></div>

            <div className="tab-content mt-30">

              {/* Overview Tab */}
              {tab === 'overview' && (
                <div className="row">
                  {/* Company Info Card */}
                  <div className="col-lg-6 mb-30">
                    <div className="company-info-card h-100" style={{ background: '#fff', border: '1px solid #eee', borderRadius: 12, padding: 24 }}>
                      <h5 className="color-gray-900 mb-20">Company Information</h5>
                      <div id="company-info">
                        {[
                          { label: 'Company Name', value: status.business_name },
                          { label: 'Business Type', value: status.business_type },
                          { label: 'Tax ID', value: status.tax_id || '—' },
                          { label: 'Contact Person', value: status.contact_name },
                          { label: 'Contact Phone', value: status.phone || '—' },
                          { label: 'Contact Email', value: status.email || user?.email },
                          { label: 'Discount Rate', value: status.discount_percentage ? `${status.discount_percentage}% off retail` : '—', highlight: true },
                          { label: 'Approved On', value: status.approved_at ? new Date(status.approved_at).toLocaleDateString() : '—' },
                        ].map(row => (
                          <div key={row.label} className="info-row" style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', padding: '10px 0', borderBottom: '1px solid #f5f5f5' }}>
                            <span className="font-sm color-gray-500">{row.label}</span>
                            <span className={`font-sm${row.highlight ? ' color-brand-3' : ' color-gray-900'}`} style={{ fontWeight: row.highlight ? 700 : 400 }}>
                              {row.value}
                            </span>
                          </div>
                        ))}
                      </div>
                    </div>
                  </div>

                  {/* Quick Links + Account Details */}
                  <div className="col-lg-6 mb-30">
                    <div className="company-info-card h-100" style={{ background: '#fff', border: '1px solid #eee', borderRadius: 12, padding: 24 }}>
                      <h5 className="color-gray-900 mb-20">Quick Links</h5>
                      <div className="quick-links">
                        {[
                          { icon: 'fa-shopping-bag', label: 'Browse Products', to: '/shop' },
                          { icon: 'fa-shopping-cart', label: 'View Cart', to: '/cart' },
                          { icon: 'fa-user', label: 'My Account', to: '/account' },
                          { icon: 'fa-headset', label: 'Contact Support', to: '/contact' },
                        ].map(link => (
                          <Link
                            key={link.label}
                            to={link.to}
                            style={{ display: 'block', padding: '12px 16px', background: '#f8f9fa', borderRadius: 10, marginBottom: 10, color: '#333', transition: 'all 0.3s ease', textDecoration: 'none' }}
                            onMouseEnter={e => { e.currentTarget.style.background = '#3BB77E'; e.currentTarget.style.color = '#fff' }}
                            onMouseLeave={e => { e.currentTarget.style.background = '#f8f9fa'; e.currentTarget.style.color = '#333' }}
                          >
                            <i className={`fas ${link.icon} mr-10`}></i>
                            {link.label}
                          </Link>
                        ))}
                      </div>

                      <h5 className="color-gray-900 mt-25 mb-15">Account Details</h5>
                      <div className="info-row" style={{ display: 'flex', justifyContent: 'space-between', padding: '10px 0', borderBottom: '1px solid #f5f5f5' }}>
                        <span className="font-sm color-gray-500">Member Since</span>
                        <span className="font-sm color-gray-900">
                          {status.created_at ? new Date(status.created_at).toLocaleDateString() : '—'}
                        </span>
                      </div>
                      <div className="info-row" style={{ display: 'flex', justifyContent: 'space-between', padding: '10px 0' }}>
                        <span className="font-sm color-gray-500">Account Status</span>
                        <span className="order-status shipped font-xs" style={{ padding: '4px 12px', borderRadius: 20, fontSize: 12, fontWeight: 600, background: '#d4edda', color: '#155724' }}>
                          Active
                        </span>
                      </div>
                    </div>
                  </div>

                  {/* Recent Orders */}
                  <div className="col-lg-12">
                    <div className="company-info-card" style={{ background: '#fff', border: '1px solid #eee', borderRadius: 12, padding: 24 }}>
                      <div className="d-flex justify-content-between align-items-center mb-20">
                        <h5 className="color-gray-900 mb-0">Recent Orders</h5>
                        <button
                          className="btn btn-brand-2 font-xs"
                          onClick={() => setTab('orders')}
                          style={{ padding: '6px 16px' }}
                        >
                          View All
                        </button>
                      </div>
                      {ordersLoading ? (
                        <Loading height={120} />
                      ) : orders.length === 0 ? (
                        <p className="font-sm color-gray-500 text-center py-3">No orders yet. <Link to="/shop" className="color-brand-3">Start shopping</Link></p>
                      ) : (
                        <div className="table-responsive">
                          <table className="table table-hover mb-0">
                            <thead>
                              <tr>
                                <th className="font-sm color-gray-900">Order #</th>
                                <th className="font-sm color-gray-900">Date</th>
                                <th className="font-sm color-gray-900">Total</th>
                                <th className="font-sm color-gray-900">Status</th>
                              </tr>
                            </thead>
                            <tbody>
                              {orders.slice(0, 5).map(order => {
                                const badge = STATUS_BADGE[order.status] || { cls: 'pending', label: order.status }
                                return (
                                  <tr key={order.id}>
                                    <td className="font-sm color-gray-900">#{order.id}</td>
                                    <td className="font-sm color-gray-500">{new Date(order.created_at).toLocaleDateString()}</td>
                                    <td className="font-sm-bold color-brand-3">{formatCurrency(order.total_amount)}</td>
                                    <td>
                                      <span className={`order-status ${badge.cls} font-xs`} style={{ padding: '3px 10px', borderRadius: 20, fontSize: 12, fontWeight: 600 }}>
                                        {badge.label}
                                      </span>
                                    </td>
                                  </tr>
                                )
                              })}
                            </tbody>
                          </table>
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              )}

              {/* Orders Tab */}
              {tab === 'orders' && (
                <div className="box-orders">
                  <div className="d-flex justify-content-between align-items-center mb-25">
                    <h4 className="color-gray-900 mb-0">Order History</h4>
                    <span className="font-sm color-gray-500">{ordersMeta.total} total orders</span>
                  </div>

                  {ordersLoading ? (
                    <Loading height={300} />
                  ) : orders.length === 0 ? (
                    <div style={{ textAlign: 'center', padding: 60 }}>
                      <i className="fas fa-box-open" style={{ fontSize: 64, color: '#ddd', marginBottom: 20, display: 'block' }}></i>
                      <h5 className="color-gray-900 mb-10">No Orders Yet</h5>
                      <p className="font-md color-gray-500 mb-25">You haven't placed any wholesale orders yet.</p>
                      <Link to="/shop" className="btn btn-buy">Start Shopping</Link>
                    </div>
                  ) : (
                    <>
                      <div className="table-responsive">
                        <table className="table table-hover">
                          <thead style={{ background: '#f8f9fa' }}>
                            <tr>
                              <th className="font-sm color-gray-900">Order #</th>
                              <th className="font-sm color-gray-900">Date</th>
                              <th className="font-sm color-gray-900">Items</th>
                              <th className="font-sm color-gray-900">Total</th>
                              <th className="font-sm color-gray-900">Payment</th>
                              <th className="font-sm color-gray-900">Status</th>
                            </tr>
                          </thead>
                          <tbody>
                            {orders.map(order => {
                              const badge = STATUS_BADGE[order.status] || { cls: 'pending', label: order.status }
                              return (
                                <tr key={order.id}>
                                  <td className="font-sm-bold color-brand-3">#{order.id}</td>
                                  <td className="font-sm color-gray-500">{new Date(order.created_at).toLocaleDateString()}</td>
                                  <td className="font-sm color-gray-900">{order.items_count || order.items?.length || '—'}</td>
                                  <td className="font-sm-bold color-brand-3">{formatCurrency(order.total_amount)}</td>
                                  <td className="font-sm color-gray-500" style={{ textTransform: 'capitalize' }}>
                                    {order.payment_method?.replace(/_/g, ' ') || '—'}
                                  </td>
                                  <td>
                                    <span className={`order-status ${badge.cls} font-xs`} style={{ padding: '4px 12px', borderRadius: 20, fontSize: 12, fontWeight: 600 }}>
                                      {badge.label}
                                    </span>
                                  </td>
                                </tr>
                              )
                            })}
                          </tbody>
                        </table>
                      </div>
                      <Pagination
                        currentPage={ordersMeta.current_page}
                        lastPage={ordersMeta.last_page}
                        onPageChange={(p) => { setOrdersPage(p); window.scrollTo(0, 200) }}
                      />
                    </>
                  )}
                </div>
              )}

              {/* Account Info Tab */}
              {tab === 'account' && (
                <div className="row">
                  <div className="col-lg-8">
                    <div style={{ background: '#fff', border: '1px solid #eee', borderRadius: 12, padding: 30 }}>
                      <h5 className="color-gray-900 mb-25" style={{ paddingBottom: 15, borderBottom: '2px solid #3BB77E' }}>
                        Wholesale Account Details
                      </h5>
                      <div className="table-responsive">
                        <table className="table table-borderless mb-0">
                          <tbody>
                            {[
                              { label: 'Business Name', value: status.business_name },
                              { label: 'Business Type', value: status.business_type },
                              { label: 'Tax ID / EIN', value: status.tax_id || '—' },
                              { label: 'Contact Name', value: status.contact_name },
                              { label: 'Email', value: status.email || user?.email },
                              { label: 'Phone', value: status.phone || '—' },
                              { label: 'Address', value: [status.address, status.city, status.state, status.zip, status.country].filter(Boolean).join(', ') || '—' },
                              { label: 'Annual Revenue', value: status.annual_revenue || '—' },
                              { label: 'Account Status', value: 'Active', special: true },
                              { label: 'Discount Rate', value: status.discount_percentage ? `${status.discount_percentage}% off retail` : '—', special: true },
                              { label: 'Member Since', value: status.created_at ? new Date(status.created_at).toLocaleDateString() : '—' },
                              { label: 'Approved On', value: status.approved_at ? new Date(status.approved_at).toLocaleDateString() : '—' },
                            ].map(row => (
                              <tr key={row.label}>
                                <td className="font-sm color-gray-500" style={{ paddingLeft: 0, width: '35%', borderBottom: '1px solid #f5f5f5' }}>
                                  <strong>{row.label}</strong>
                                </td>
                                <td className={`font-sm${row.special ? ' color-brand-3 font-sm-bold' : ' color-gray-900'}`} style={{ borderBottom: '1px solid #f5f5f5' }}>
                                  {row.value}
                                </td>
                              </tr>
                            ))}
                          </tbody>
                        </table>
                      </div>
                      <div className="mt-25">
                        <Link to="/account" className="btn btn-brand-2 mr-15">Edit Account</Link>
                        <Link to="/contact" className="btn btn-border">Contact Support</Link>
                      </div>
                    </div>
                  </div>
                  <div className="col-lg-4">
                    <div style={{ background: '#f0fdf4', border: '1px solid #3BB77E', borderRadius: 12, padding: 24, textAlign: 'center' }}>
                      <i className="fas fa-award" style={{ fontSize: 48, color: '#3BB77E', marginBottom: 15, display: 'block' }}></i>
                      <h5 className="color-brand-3 mb-10">
                        {status.discount_percentage >= 40 ? 'Gold' : status.discount_percentage >= 30 ? 'Silver' : 'Bronze'} Partner
                      </h5>
                      <p className="font-sm color-gray-500 mb-15">
                        {status.discount_percentage
                          ? `You're saving ${status.discount_percentage}% on all purchases.`
                          : 'Your wholesale discount is active on all products.'}
                      </p>
                      <Link to="/shop" className="btn btn-buy font-sm">Shop Now</Link>
                    </div>
                  </div>
                </div>
              )}

            </div>
          </div>
        </div>
      </section>
    </>
  )
}

export default WholesaleDashboard
