import { useState, useEffect } from 'react'
import { Link, useNavigate } from 'react-router-dom'
import { useAuth } from '../context/AuthContext'
import { applyWholesale, getWholesaleStatus } from '../services/api'
import toast from 'react-hot-toast'
import Loading from '../components/ui/Loading'

const STATUS_STYLES = {
  pending: { bg: '#fff3cd', border: '#ffc107', icon: '⏳', label: 'Under Review' },
  approved: { bg: '#d4edda', border: '#3BB77E', icon: '✓', label: 'Approved' },
  rejected: { bg: '#f8d7da', border: '#dc3545', icon: '✗', label: 'Not Approved' },
}

const BENEFITS = [
  { icon: 'fa-percentage', title: 'Up to 50% Off Retail', desc: 'Access exclusive wholesale pricing with significant margins on our entire product catalog.' },
  { icon: 'fa-layer-group', title: 'Volume Discounts', desc: 'The more you buy, the more you save. Unlock additional discounts with tiered pricing.' },
  { icon: 'fa-user-tie', title: 'Dedicated Support', desc: 'Get your own account manager for personalized service and priority assistance.' },
  { icon: 'fa-shipping-fast', title: 'Priority Fulfillment', desc: 'Skip the queue with expedited processing and same-day shipping options.' },
  { icon: 'fa-undo-alt', title: 'Flexible Returns', desc: '30-day hassle-free returns on defective items with full credit to your account.' },
  { icon: 'fa-credit-card', title: 'Net 30 Terms', desc: 'Qualified partners can access flexible payment terms to manage cash flow.' },
]

const Wholesale = () => {
  const { isLoggedIn, user } = useAuth()
  const navigate = useNavigate()

  const [statusData, setStatusData] = useState(null)
  const [checkingStatus, setCheckingStatus] = useState(true)
  const [submitting, setSubmitting] = useState(false)
  const [submitted, setSubmitted] = useState(false)

  const [form, setForm] = useState({
    business_name: '',
    contact_name: '',
    email: '',
    phone: '',
    address: '',
    city: '',
    state: '',
    zip: '',
    country: 'United States',
    business_type: 'Retailer',
    tax_id: '',
    annual_revenue: '',
    notes: '',
  })

  // Check if the user has an existing application
  useEffect(() => {
    if (!isLoggedIn) {
      setCheckingStatus(false)
      return
    }
    getWholesaleStatus()
      .then(res => {
        // API returns { status, application } directly
        const d = res.data
        if (d && d.status) setStatusData(d)
      })
      .catch(() => {})
      .finally(() => setCheckingStatus(false))
  }, [isLoggedIn])

  // Pre-fill email from logged-in user
  useEffect(() => {
    if (user) {
      setForm(f => ({
        ...f,
        email: user.email || '',
        contact_name: user.name || '',
      }))
    }
  }, [user])

  const handleChange = (e) => {
    const { name, value } = e.target
    setForm(f => ({ ...f, [name]: value }))
  }

  const handleSubmit = async (e) => {
    e.preventDefault()
    if (!isLoggedIn) {
      toast.error('Please login first to submit an application.')
      navigate('/login')
      return
    }
    setSubmitting(true)
    try {
      await applyWholesale(form)
      setSubmitted(true)
      toast.success('Your wholesale application has been submitted!')
      // Refresh status
      getWholesaleStatus()
        .then(res => { const d = res.data; if (d?.status) setStatusData(d) })
        .catch(() => {})
    } catch (err) {
      const errors = err?.response?.data?.errors
      if (errors) {
        Object.values(errors).flat().forEach(msg => toast.error(msg))
      } else {
        toast.error(err?.response?.data?.message || 'Failed to submit application. Please try again.')
      }
    } finally {
      setSubmitting(false)
    }
  }

  if (checkingStatus) return <Loading height={400} />

  const statusInfo = statusData ? STATUS_STYLES[statusData.status] : null

  return (
    <>
      {/* Breadcrumb */}
      <div className="section-box">
        <div className="breadcrumbs-div mb-0">
          <div className="container">
            <ul className="breadcrumb">
              <li><Link className="font-xs color-gray-1000" to="/">Home</Link></li>
              <li><span className="font-xs color-gray-500">Wholesale</span></li>
            </ul>
          </div>
        </div>
      </div>

      {/* Hero */}
      <section style={{
        background: 'linear-gradient(135deg, #253D4E 0%, #3BB77E 100%)',
        padding: '60px 0 50px',
        color: '#fff',
        textAlign: 'center',
      }}>
        <div className="container">
          <span style={{ background: 'rgba(255,255,255,0.15)', color: '#fff', padding: '5px 18px', borderRadius: 20, fontSize: 13, fontWeight: 600, textTransform: 'uppercase', letterSpacing: 1, display: 'inline-block', marginBottom: 20 }}>
            B2B Wholesale Program
          </span>
          <h1 style={{ color: '#fff', fontSize: 40, fontWeight: 700, marginBottom: 16 }}>
            Partner With <span style={{ color: '#a8f0ce' }}>Kalses</span> for<br />Wholesale Success
          </h1>
          <p style={{ fontSize: 18, maxWidth: 600, margin: '0 auto 30px', opacity: 0.9 }}>
            Join thousands of retailers who trust Kalses for premium products at competitive wholesale prices.
          </p>
          <div style={{ display: 'flex', justifyContent: 'center', gap: 40, flexWrap: 'wrap', marginTop: 30 }}>
            {[
              { number: '5,000+', label: 'Active Partners' },
              { number: '50%', label: 'Avg. Savings' },
              { number: '10K+', label: 'Products' },
              { number: '24/7', label: 'B2B Support' },
            ].map(stat => (
              <div key={stat.label} style={{ textAlign: 'center' }}>
                <div style={{ fontSize: 32, fontWeight: 700, color: '#a8f0ce' }}>{stat.number}</div>
                <div style={{ fontSize: 14, opacity: 0.85 }}>{stat.label}</div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Trust Badges */}
      <div style={{ background: '#fff', padding: '20px 0', borderBottom: '1px solid #eee' }}>
        <div className="container">
          <div className="row">
            {[
              { icon: 'fa-shield-alt', text: 'Verified Business Only' },
              { icon: 'fa-lock', text: 'Secure Transactions' },
              { icon: 'fa-truck', text: 'Fast B2B Shipping' },
              { icon: 'fa-headset', text: 'Dedicated Account Manager' },
            ].map(badge => (
              <div key={badge.text} className="col-md-3 col-6 mb-15 mb-md-0">
                <div className="d-flex align-items-center justify-content-center" style={{ gap: 10, color: '#555', fontSize: 14 }}>
                  <i className={`fas ${badge.icon}`} style={{ color: '#3BB77E', fontSize: 20 }}></i>
                  <span>{badge.text}</span>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Benefits */}
      <section className="section-box pt-80 pb-50" style={{ background: '#f8f9fa' }}>
        <div className="container">
          <div className="text-center mb-50">
            <span className="font-sm color-brand-3 text-uppercase" style={{ fontWeight: 600, letterSpacing: 1 }}>Why Choose Us</span>
            <h2 className="mt-10 mb-15 color-gray-900">Powerful Benefits for Your Business</h2>
            <p className="font-md color-gray-500" style={{ maxWidth: 600, margin: '0 auto' }}>
              Everything you need to grow your retail business with competitive wholesale pricing and exceptional service.
            </p>
          </div>
          <div className="row">
            {BENEFITS.map(b => (
              <div key={b.title} className="col-lg-4 col-md-6 mb-30">
                <div className="benefits-card" style={{ background: '#fff', borderRadius: 15, padding: '35px 25px', textAlign: 'center', border: '1px solid #eee', transition: 'all 0.3s ease' }}>
                  <div style={{ width: 80, height: 80, background: '#e8f5f0', borderRadius: '50%', display: 'flex', alignItems: 'center', justifyContent: 'center', margin: '0 auto 25px' }}>
                    <i className={`fas ${b.icon}`} style={{ fontSize: '2rem', color: '#3BB77E' }}></i>
                  </div>
                  <h5 style={{ marginBottom: 10, color: '#253D4E', fontWeight: 600 }}>{b.title}</h5>
                  <p className="font-sm color-gray-500" style={{ margin: 0 }}>{b.desc}</p>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Application Section */}
      <section className="section-box shop-template mt-60 mb-60" id="application-section">
        <div className="container">
          <div className="row">
            {/* Form Column */}
            <div className="col-lg-7 mb-40">
              <h3 className="color-gray-900 mb-10">Wholesale (B2B) Application</h3>
              <p className="font-md color-gray-500 mb-30">
                Apply to become an authorized wholesale partner and unlock exclusive B2B pricing.
                Applications are reviewed within 2&ndash;3 business days.
              </p>

              {/* Application status display */}
              {statusData && statusInfo && (
                <div style={{
                  background: statusInfo.bg,
                  border: `1px solid ${statusInfo.border}`,
                  borderRadius: 12,
                  padding: 30,
                  marginBottom: 30,
                  textAlign: 'center',
                }}>
                  <div style={{ fontSize: 48, marginBottom: 12 }}>{statusInfo.icon}</div>
                  <h5 style={{ color: '#253D4E', marginBottom: 8 }}>
                    Application Status: <strong>{statusInfo.label}</strong>
                  </h5>
                  {statusData.status === 'pending' && (
                    <p className="font-sm color-gray-500">
                      Submitted on {new Date(statusData.created_at).toLocaleDateString()}. We typically respond within 2&ndash;3 business days.
                    </p>
                  )}
                  {statusData.status === 'approved' && (
                    <>
                      <p className="font-sm color-gray-500 mb-20">
                        Congratulations! Your wholesale account is active.
                        {statusData.discount_percentage && (
                          <> You have <strong>{statusData.discount_percentage}% discount</strong> on all products.</>
                        )}
                      </p>
                      <Link className="btn btn-buy" to="/wholesale-dashboard">
                        Go to Wholesale Dashboard
                      </Link>
                    </>
                  )}
                  {statusData.status === 'rejected' && (
                    <>
                      <p className="font-sm color-gray-500 mb-15">
                        {statusData.rejection_reason || 'Your application was not approved at this time.'}
                      </p>
                      <Link className="btn btn-brand-2" to="/contact">Contact Support</Link>
                    </>
                  )}
                </div>
              )}

              {/* Submitted confirmation */}
              {submitted && !statusData && (
                <div style={{ background: '#d4edda', border: '1px solid #3BB77E', borderRadius: 12, padding: 30, textAlign: 'center' }}>
                  <div style={{ fontSize: 48, marginBottom: 12 }}>✓</div>
                  <h5 className="color-brand-3">Application Submitted!</h5>
                  <p className="font-sm color-gray-500">Thank you! We'll review your application and contact you within 2&ndash;3 business days.</p>
                  <Link className="btn btn-buy mt-15" to="/">Return to Home</Link>
                </div>
              )}

              {/* Not logged in warning */}
              {!isLoggedIn && (
                <div style={{ background: '#fff3cd', border: '1px solid #ffc107', borderRadius: 8, padding: 16, marginBottom: 24 }}>
                  <p className="font-sm mb-0">
                    Please <Link to="/login" className="color-brand-3">login</Link> or{' '}
                    <Link to="/register" className="color-brand-3">create an account</Link> before applying.
                  </p>
                </div>
              )}

              {/* Application Form: only show if no active status and not just submitted */}
              {!statusData && !submitted && (
                <form className="form-register" onSubmit={handleSubmit}>
                  {/* Step: Business Info */}
                  <h6 className="color-gray-900 mb-20 mt-10" style={{ paddingBottom: 10, borderBottom: '2px solid #3BB77E' }}>
                    Business Information
                  </h6>
                  <div className="row">
                    <div className="col-lg-6">
                      <div className="form-group">
                        <label className="font-sm color-gray-900">Business Name <span style={{ color: '#dc3545' }}>*</span></label>
                        <input
                          className="form-control font-sm"
                          name="business_name"
                          value={form.business_name}
                          onChange={handleChange}
                          placeholder="Your company name"
                          required
                        />
                      </div>
                    </div>
                    <div className="col-lg-6">
                      <div className="form-group">
                        <label className="font-sm color-gray-900">Contact Name <span style={{ color: '#dc3545' }}>*</span></label>
                        <input
                          className="form-control font-sm"
                          name="contact_name"
                          value={form.contact_name}
                          onChange={handleChange}
                          placeholder="Your full name"
                          required
                        />
                      </div>
                    </div>
                    <div className="col-lg-6">
                      <div className="form-group">
                        <label className="font-sm color-gray-900">Email Address <span style={{ color: '#dc3545' }}>*</span></label>
                        <input
                          className="form-control font-sm"
                          type="email"
                          name="email"
                          value={form.email}
                          onChange={handleChange}
                          placeholder="business@email.com"
                          required
                        />
                      </div>
                    </div>
                    <div className="col-lg-6">
                      <div className="form-group">
                        <label className="font-sm color-gray-900">Phone Number <span style={{ color: '#dc3545' }}>*</span></label>
                        <input
                          className="form-control font-sm"
                          name="phone"
                          value={form.phone}
                          onChange={handleChange}
                          placeholder="+1 (555) 000-0000"
                          required
                        />
                      </div>
                    </div>
                    <div className="col-lg-6">
                      <div className="form-group">
                        <label className="font-sm color-gray-900">Business Type</label>
                        <select
                          className="form-control font-sm"
                          name="business_type"
                          value={form.business_type}
                          onChange={handleChange}
                        >
                          <option value="Retailer">Retailer</option>
                          <option value="Distributor">Distributor</option>
                          <option value="Reseller">Reseller</option>
                          <option value="Online Store">Online Store</option>
                          <option value="Corporation">Corporation</option>
                          <option value="Other">Other</option>
                        </select>
                      </div>
                    </div>
                    <div className="col-lg-6">
                      <div className="form-group">
                        <label className="font-sm color-gray-900">Tax ID / EIN</label>
                        <input
                          className="form-control font-sm"
                          name="tax_id"
                          value={form.tax_id}
                          onChange={handleChange}
                          placeholder="XX-XXXXXXX"
                        />
                      </div>
                    </div>
                  </div>

                  {/* Step: Address */}
                  <h6 className="color-gray-900 mb-20 mt-20" style={{ paddingBottom: 10, borderBottom: '2px solid #3BB77E' }}>
                    Business Address
                  </h6>
                  <div className="row">
                    <div className="col-lg-12">
                      <div className="form-group">
                        <label className="font-sm color-gray-900">Street Address <span style={{ color: '#dc3545' }}>*</span></label>
                        <input
                          className="form-control font-sm"
                          name="address"
                          value={form.address}
                          onChange={handleChange}
                          placeholder="123 Business Ave"
                          required
                        />
                      </div>
                    </div>
                    <div className="col-lg-4">
                      <div className="form-group">
                        <label className="font-sm color-gray-900">City</label>
                        <input
                          className="form-control font-sm"
                          name="city"
                          value={form.city}
                          onChange={handleChange}
                          placeholder="City"
                        />
                      </div>
                    </div>
                    <div className="col-lg-4">
                      <div className="form-group">
                        <label className="font-sm color-gray-900">State / Province</label>
                        <input
                          className="form-control font-sm"
                          name="state"
                          value={form.state}
                          onChange={handleChange}
                          placeholder="State"
                        />
                      </div>
                    </div>
                    <div className="col-lg-4">
                      <div className="form-group">
                        <label className="font-sm color-gray-900">ZIP / Postal Code</label>
                        <input
                          className="form-control font-sm"
                          name="zip"
                          value={form.zip}
                          onChange={handleChange}
                          placeholder="12345"
                        />
                      </div>
                    </div>
                    <div className="col-lg-12">
                      <div className="form-group">
                        <label className="font-sm color-gray-900">Country</label>
                        <input
                          className="form-control font-sm"
                          name="country"
                          value={form.country}
                          onChange={handleChange}
                          placeholder="United States"
                        />
                      </div>
                    </div>
                  </div>

                  {/* Step: Additional */}
                  <h6 className="color-gray-900 mb-20 mt-20" style={{ paddingBottom: 10, borderBottom: '2px solid #3BB77E' }}>
                    Additional Details
                  </h6>
                  <div className="row">
                    <div className="col-lg-12">
                      <div className="form-group">
                        <label className="font-sm color-gray-900">Annual Revenue</label>
                        <select
                          className="form-control font-sm"
                          name="annual_revenue"
                          value={form.annual_revenue}
                          onChange={handleChange}
                        >
                          <option value="">Select revenue range...</option>
                          <option value="under_50k">Under $50,000</option>
                          <option value="50k_200k">$50,000 – $200,000</option>
                          <option value="200k_1m">$200,000 – $1,000,000</option>
                          <option value="over_1m">Over $1,000,000</option>
                        </select>
                      </div>
                    </div>
                    <div className="col-lg-12">
                      <div className="form-group">
                        <label className="font-sm color-gray-900">Additional Notes</label>
                        <textarea
                          className="form-control font-sm"
                          name="notes"
                          value={form.notes}
                          onChange={handleChange}
                          rows={4}
                          placeholder="Tell us about your business and what products you're interested in..."
                        />
                      </div>
                    </div>
                    <div className="col-lg-12">
                      <div className="form-group">
                        <button
                          className="btn btn-buy font-md-bold"
                          type="submit"
                          disabled={submitting || !isLoggedIn}
                          style={{ padding: '12px 40px', minWidth: 200 }}
                        >
                          {submitting ? (
                            <>
                              <span className="spinner-border spinner-border-sm mr-10" role="status" aria-hidden="true"></span>
                              Submitting...
                            </>
                          ) : 'Submit Application'}
                        </button>
                      </div>
                    </div>
                  </div>
                </form>
              )}
            </div>

            {/* Benefits Sidebar */}
            <div className="col-lg-5">
              <div className="box-border p-30" style={{ background: '#f8f9fa', borderRadius: 12, border: '1px solid #eee' }}>
                <h5 className="font-md-bold color-gray-900 mb-20">Why Join Our B2B Program?</h5>
                <ul style={{ listStyle: 'none', padding: 0, margin: 0 }}>
                  {[
                    'Exclusive wholesale pricing up to 50% off',
                    'Dedicated account manager for your business',
                    'Priority shipping and fulfillment',
                    'Net 30 payment terms (qualified accounts)',
                    'Custom bulk ordering with no minimums',
                    'Early access to new product launches',
                    'Detailed sales analytics and reporting',
                    '30-day hassle-free return policy',
                  ].map(item => (
                    <li key={item} className="font-sm color-gray-500" style={{ padding: '8px 0', borderBottom: '1px solid #eee', display: 'flex', alignItems: 'flex-start', gap: 10 }}>
                      <span style={{ color: '#3BB77E', fontWeight: 700, flexShrink: 0 }}>✓</span>
                      {item}
                    </li>
                  ))}
                </ul>
              </div>

              {/* Pricing tiers preview */}
              <div className="mt-30" style={{ background: '#fff', borderRadius: 12, border: '1px solid #eee', padding: 24 }}>
                <h5 className="font-md-bold color-gray-900 mb-20">Partner Tiers</h5>
                {[
                  { tier: 'Bronze', discount: '20%', range: 'Under $1,000/mo' },
                  { tier: 'Silver', discount: '35%', range: '$1,000 – $5,000/mo', featured: true },
                  { tier: 'Gold', discount: '50%', range: 'Over $5,000/mo' },
                ].map(t => (
                  <div key={t.tier} style={{
                    padding: '12px 16px',
                    borderRadius: 8,
                    marginBottom: 10,
                    border: t.featured ? '2px solid #3BB77E' : '1px solid #eee',
                    background: t.featured ? '#f0fdf4' : '#f8f9fa',
                    display: 'flex',
                    justifyContent: 'space-between',
                    alignItems: 'center',
                  }}>
                    <div>
                      <strong className={t.featured ? 'color-brand-3' : 'color-gray-900'}>{t.tier}</strong>
                      <span className="font-xs color-gray-500 d-block">{t.range}</span>
                    </div>
                    <span style={{ fontSize: 22, fontWeight: 700, color: '#3BB77E' }}>{t.discount} OFF</span>
                  </div>
                ))}
              </div>

              {/* Already approved */}
              {statusData?.status === 'approved' && (
                <div className="mt-30" style={{ background: '#d4edda', border: '1px solid #3BB77E', borderRadius: 12, padding: 24, textAlign: 'center' }}>
                  <h5 className="color-brand-3 mb-10">You have wholesale access!</h5>
                  <p className="font-sm color-gray-500 mb-15">Visit your dashboard to manage your wholesale account.</p>
                  <Link className="btn btn-buy" to="/wholesale-dashboard">Go to Dashboard</Link>
                </div>
              )}
            </div>
          </div>
        </div>
      </section>
    </>
  )
}

export default Wholesale
