import { useState, useEffect, useCallback } from 'react'
import { Link } from 'react-router-dom'
import { getVendors } from '../services/api'
import Pagination from '../components/ui/Pagination'

const VendorList = () => {
  const [vendors, setVendors] = useState([])
  const [meta, setMeta] = useState({ current_page: 1, last_page: 1, total: 0 })
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState(null)
  const [page, setPage] = useState(1)
  const [search, setSearch] = useState('')
  const [searchInput, setSearchInput] = useState('')
  const [sortBy, setSortBy] = useState('latest')
  const [perPage, setPerPage] = useState(30)

  const fetchVendors = useCallback(() => {
    setLoading(true)
    setError(null)
    getVendors({ page, per_page: perPage, search, sort: sortBy })
      .then(r => {
        // Vendors API returns paginated: { data: [...], current_page, last_page, total }
        const d = r.data
        setVendors(Array.isArray(d) ? d : (d?.data || []))
        setMeta({
          current_page: d?.current_page || 1,
          last_page: d?.last_page || 1,
          total: d?.total || 0,
        })
      })
      .catch(() => setError('Failed to load vendors. Please try again.'))
      .finally(() => setLoading(false))
  }, [page, search, sortBy, perPage])

  useEffect(() => {
    fetchVendors()
  }, [fetchVendors])

  const handleSearchSubmit = (e) => {
    e.preventDefault()
    setSearch(searchInput)
    setPage(1)
  }

  const handleSortChange = (value) => {
    setSortBy(value)
    setPage(1)
  }

  const handlePerPageChange = (value) => {
    setPerPage(value)
    setPage(1)
  }

  const sortLabel = {
    latest: 'Latest added',
    oldest: 'Oldest added',
    name: 'Name (A-Z)',
  }

  return (
    <>
      {/* Breadcrumb */}
      <div className="section-box">
        <div className="breadcrumbs-div">
          <div className="container">
            <ul className="breadcrumb">
              <li><Link className="font-xs color-gray-1000" to="/">Home</Link></li>
              <li><span className="font-xs color-gray-500">Vendor Listing</span></li>
            </ul>
          </div>
        </div>
      </div>

      {/* Main Section */}
      <section className="section-box shop-template mt-0">
        <div className="container">
          <h2>Vendors Listing</h2>
          <div className="row align-items-center">
            <div className="col-lg-6 mb-30">
              <p className="font-md color-gray-500">
                We have
                <span className="font-md-bold color-brand-3"> {meta.total}</span>
                <span> vendors now</span>
              </p>
            </div>
            <div className="col-lg-6 mb-30 text-end">
              <Link className="btn btn-buy w-auto font-sm-bold" to="/register">Open a Shop</Link>
            </div>
          </div>
          <div className="border-bottom pt-0 mb-30"></div>

          <div className="row">
            {/* Vendors Grid */}
            <div className="col-lg-9 order-first order-lg-last">
              {/* Filters bar */}
              <div className="box-filters mt-0 pb-5 border-bottom">
                <div className="row">
                  <div className="col-xl-4 col-lg-4 mb-10 text-lg-start text-center">
                    <span className="font-sm color-gray-900 font-medium">
                      {loading ? 'Loading...' : `Showing ${vendors.length} of ${meta.total} vendors`}
                    </span>
                  </div>
                  <div className="col-xl-8 col-lg-8 mb-10 text-lg-end text-center">
                    <div className="d-inline-block">
                      <span className="font-sm color-gray-500 font-medium">Sort by:</span>
                      <div className="dropdown dropdown-sort border-1-right d-inline-block">
                        <button
                          className="btn dropdown-toggle font-sm color-gray-900 font-medium"
                          type="button"
                          data-bs-toggle="dropdown"
                          aria-expanded="false"
                        >
                          {sortLabel[sortBy] || 'Latest added'}
                        </button>
                        <ul className="dropdown-menu dropdown-menu-light" style={{ margin: 0 }}>
                          <li>
                            <button className={`dropdown-item${sortBy === 'latest' ? ' active' : ''}`} onClick={() => handleSortChange('latest')}>
                              Latest added
                            </button>
                          </li>
                          <li>
                            <button className={`dropdown-item${sortBy === 'oldest' ? ' active' : ''}`} onClick={() => handleSortChange('oldest')}>
                              Oldest added
                            </button>
                          </li>
                          <li>
                            <button className={`dropdown-item${sortBy === 'name' ? ' active' : ''}`} onClick={() => handleSortChange('name')}>
                              Name (A-Z)
                            </button>
                          </li>
                        </ul>
                      </div>
                    </div>
                    <div className="d-inline-block">
                      <span className="font-sm color-gray-500 font-medium">Show</span>
                      <div className="dropdown dropdown-sort border-1-right d-inline-block">
                        <button
                          className="btn dropdown-toggle font-sm color-gray-900 font-medium"
                          type="button"
                          data-bs-toggle="dropdown"
                          aria-expanded="false"
                          data-bs-display="static"
                        >
                          <span>{perPage} items</span>
                        </button>
                        <ul className="dropdown-menu dropdown-menu-light">
                          {[12, 24, 30, 48].map(n => (
                            <li key={n}>
                              <button className={`dropdown-item${perPage === n ? ' active' : ''}`} onClick={() => handlePerPageChange(n)}>
                                {n} items
                              </button>
                            </li>
                          ))}
                        </ul>
                      </div>
                    </div>
                  </div>
                </div>
              </div>

              {/* Loading spinner */}
              {loading && (
                <div className="row mt-20">
                  <div className="col-12 text-center py-5">
                    <div className="spinner-border text-primary" role="status">
                      <span className="visually-hidden">Loading...</span>
                    </div>
                    <p className="mt-2 font-sm color-gray-500">Loading vendors...</p>
                  </div>
                </div>
              )}

              {/* Error state */}
              {!loading && error && (
                <div className="row mt-20">
                  <div className="col-12 text-center py-5">
                    <p className="font-md color-gray-500">{error}</p>
                    <button className="btn btn-brand-2 mt-15" onClick={fetchVendors}>Try Again</button>
                  </div>
                </div>
              )}

              {/* Empty state */}
              {!loading && !error && vendors.length === 0 && (
                <div className="row mt-20">
                  <div className="col-12 text-center py-5">
                    <p className="font-md color-gray-500">
                      {search ? `No vendors found for "${search}"` : 'No vendors available yet.'}
                    </p>
                    {search && (
                      <button className="btn btn-brand-2 mt-15" onClick={() => { setSearch(''); setSearchInput(''); setPage(1) }}>
                        Clear Search
                      </button>
                    )}
                  </div>
                </div>
              )}

              {/* Vendor Grid */}
              {!loading && !error && vendors.length > 0 && (
                <div className="row mt-20" id="vendor-grid">
                  {vendors.map(vendor => (
                    <div className="col-xl-4 col-lg-4 col-md-6 mb-30" key={vendor.id}>
                      <div className="vendor-wrap hover-up">
                        <div className="vendor-img-action-main">
                          <div className="vendor-img">
                            <Link to={`/vendors/${vendor.id}`}>
                              {vendor.logo_url || vendor.logo ? (
                                <img
                                  src={vendor.logo_url || vendor.logo}
                                  alt={vendor.store_name}
                                  style={{ width: '100%', height: 160, objectFit: 'contain' }}
                                />
                              ) : (
                                <div style={{
                                  width: '100%', height: 160,
                                  display: 'flex', alignItems: 'center', justifyContent: 'center',
                                  background: '#f0fdf4',
                                }}>
                                  <span style={{ fontSize: 48, fontWeight: 700, color: '#3BB77E' }}>
                                    {vendor.store_name?.charAt(0)?.toUpperCase()}
                                  </span>
                                </div>
                              )}
                            </Link>
                          </div>
                          <div className="vendor-action">
                            <a className="btn btn-follow" href="#" onClick={e => e.preventDefault()}>Follow</a>
                          </div>
                        </div>
                        <div className="vendor-content">
                          <div className="d-flex justify-content-between mb-10">
                            <div>
                              <Link className="font-lg-bold color-brand-3" to={`/vendors/${vendor.id}`}>
                                {vendor.store_name}
                              </Link>
                            </div>
                            <div>
                              <div className="rating">
                                {[1, 2, 3, 4, 5].map(s => (
                                  <img
                                    key={s}
                                    src={s <= Math.round(vendor.rating || 5)
                                      ? '/assets/imgs/template/icons/star.svg'
                                      : '/assets/imgs/template/icons/star-gray.svg'}
                                    alt="star"
                                    style={{ width: 14, height: 14 }}
                                  />
                                ))}
                              </div>
                            </div>
                          </div>
                          <div className="font-sm color-gray-500 mb-10">
                            {vendor.description
                              ? vendor.description.length > 80
                                ? vendor.description.slice(0, 80) + '...'
                                : vendor.description
                              : 'Quality products from a trusted vendor.'}
                          </div>
                          {vendor.city && (
                            <p className="font-xs color-gray-500 mb-10">
                              <i className="fi-rs-marker mr-5"></i>
                              {vendor.city}{vendor.state ? `, ${vendor.state}` : ''}
                            </p>
                          )}
                          {vendor.products_count !== undefined && (
                            <p className="font-xs color-gray-500 mb-10">
                              <strong className="color-brand-3">{vendor.products_count}</strong> Products
                            </p>
                          )}
                          <div className="d-flex align-items-center mt-10">
                            <Link className="btn btn-brand-2 mr-10 font-xs" to={`/vendors/${vendor.id}`}>
                              Visit Store
                            </Link>
                          </div>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              )}

              {/* Pagination */}
              {!loading && meta.last_page > 1 && (
                <nav id="vendor-pagination" className="mt-20">
                  <Pagination
                    currentPage={meta.current_page}
                    lastPage={meta.last_page}
                    onPageChange={(p) => { setPage(p); window.scrollTo(0, 0) }}
                  />
                </nav>
              )}
            </div>

            {/* Sidebar */}
            <div className="col-lg-3 order-last order-lg-first">
              <div className="sidebar-border mb-40">
                <div className="sidebar-head">
                  <h6 className="color-gray-900">Filter Vendors</h6>
                </div>
                <div className="sidebar-content">
                  <h6 className="color-gray-900 mt-10 mb-10">Search</h6>
                  <form onSubmit={handleSearchSubmit}>
                    <div className="d-flex gap-2">
                      <input
                        className="form-control font-sm"
                        type="text"
                        placeholder="Search vendors..."
                        value={searchInput}
                        onChange={e => setSearchInput(e.target.value)}
                      />
                      <button type="submit" className="btn btn-brand-2 font-xs" style={{ whiteSpace: 'nowrap' }}>
                        Go
                      </button>
                    </div>
                  </form>
                  {search && (
                    <div className="mt-10">
                      <span className="font-xs color-gray-500">
                        Showing results for: <strong>"{search}"</strong>
                      </span>
                      <button
                        className="btn font-xs color-gray-500 d-block mt-5"
                        style={{ padding: 0, background: 'none', border: 'none', textDecoration: 'underline', cursor: 'pointer' }}
                        onClick={() => { setSearch(''); setSearchInput(''); setPage(1) }}
                      >
                        Clear
                      </button>
                    </div>
                  )}
                </div>
              </div>

              <div className="sidebar-border mb-40">
                <div className="sidebar-head">
                  <h6 className="color-gray-900">Vendor by Industry</h6>
                </div>
                <div className="sidebar-content">
                  <ul className="list-nav-arrow">
                    {['Beauty', 'Grocery', 'Baby', 'Toys', 'Pets', 'Health', 'Herbal'].map(cat => (
                      <li key={cat}>
                        <button
                          className="font-sm"
                          style={{ background: 'none', border: 'none', cursor: 'pointer', padding: '5px 0', display: 'block', width: '100%', textAlign: 'left', color: search === cat ? '#3BB77E' : 'inherit' }}
                          onClick={() => { setSearchInput(cat); setSearch(cat); setPage(1) }}
                        >
                          {cat}
                        </button>
                      </li>
                    ))}
                  </ul>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>
    </>
  )
}

export default VendorList
