import { useState, useEffect } from 'react'
import { useParams, Link } from 'react-router-dom'
import { getVendor, getProducts } from '../services/api'
import ProductCard from '../components/product/ProductCard'
import Pagination from '../components/ui/Pagination'
import Loading from '../components/ui/Loading'

const VendorDetail = () => {
  const { id } = useParams()
  const [vendor, setVendor] = useState(null)
  const [products, setProducts] = useState([])
  const [meta, setMeta] = useState({ current_page: 1, last_page: 1, total: 0 })
  const [loading, setLoading] = useState(true)
  const [productsLoading, setProductsLoading] = useState(false)
  const [error, setError] = useState(null)
  const [page, setPage] = useState(1)
  const [sortBy, setSortBy] = useState('latest')
  const [perPage, setPerPage] = useState(30)

  // Load vendor info on mount
  useEffect(() => {
    setLoading(true)
    setError(null)
    getVendor(id)
      .then(res => setVendor(res.data?.data || res.data))
      .catch(() => setError('Vendor not found.'))
      .finally(() => setLoading(false))
  }, [id])

  // Load products whenever page / sort / perPage changes
  useEffect(() => {
    setProductsLoading(true)
    getProducts({ vendor_id: id, page, per_page: perPage, sort: sortBy })
      .then(res => {
        const d = res.data  // paginated: { data: [...], current_page, last_page, total }
        setProducts(Array.isArray(d) ? d : (d?.data || []))
        setMeta({
          current_page: d?.current_page || 1,
          last_page: d?.last_page || 1,
          total: d?.total || 0,
        })
      })
      .catch(() => {})
      .finally(() => setProductsLoading(false))
  }, [id, page, sortBy, perPage])

  const handlePageChange = (p) => {
    setPage(p)
    window.scrollTo(0, 300)
  }

  const sortLabel = {
    latest: 'Latest products',
    oldest: 'Oldest products',
    price_asc: 'Price: Low to High',
    price_desc: 'Price: High to Low',
  }

  if (loading) return <Loading height={400} />

  if (error || !vendor) {
    return (
      <div className="container" style={{ padding: '80px 0', textAlign: 'center' }}>
        <h3 className="color-gray-900 mb-20">{error || 'Vendor not found'}</h3>
        <Link className="btn btn-brand-2 mr-10" to="/vendors">Back to Vendors</Link>
        <Link className="btn btn-buy" to="/">Go Home</Link>
      </div>
    )
  }

  const logoSrc = vendor.logo_url || vendor.logo

  return (
    <>
      {/* Breadcrumb */}
      <div className="section-box">
        <div className="breadcrumbs-div">
          <div className="container">
            <ul className="breadcrumb">
              <li><Link className="font-xs color-gray-1000" to="/">Home</Link></li>
              <li><Link className="font-xs color-gray-500" to="/vendors">Vendors</Link></li>
              <li><span className="font-xs color-gray-500">{vendor.store_name}</span></li>
            </ul>
          </div>
        </div>
      </div>

      {/* Vendor Profile Banner */}
      <section className="section-box shop-template mt-30">
        <div className="container">
          <div className="d-flex box-banner-vendor" style={{ flexWrap: 'wrap', gap: 20 }}>
            {/* Vendor left: banner + info */}
            <div className="vendor-left" style={{ flex: 1, minWidth: 280 }}>
              <div className="banner-vendor" style={{ position: 'relative', background: '#f0fdf4', borderRadius: 12, overflow: 'hidden', minHeight: 160 }}>
                {/* Banner background */}
                {vendor.banner_url || vendor.banner ? (
                  <img
                    src={vendor.banner_url || vendor.banner}
                    alt="Vendor Banner"
                    style={{ width: '100%', height: 160, objectFit: 'cover' }}
                  />
                ) : (
                  <div style={{
                    width: '100%', height: 160,
                    background: 'linear-gradient(135deg, #3BB77E 0%, #2eab68 100%)',
                  }} />
                )}

                <div className="d-flex box-info-vendor" style={{ padding: '15px 20px', background: '#fff', borderTop: '1px solid #eee', alignItems: 'center', gap: 20, flexWrap: 'wrap' }}>
                  <div className="avarta" style={{ textAlign: 'center', flexShrink: 0 }}>
                    {logoSrc ? (
                      <img
                        src={logoSrc}
                        alt={vendor.store_name}
                        style={{ width: 80, height: 80, objectFit: 'contain', borderRadius: '50%', border: '3px solid #fff', boxShadow: '0 2px 8px rgba(0,0,0,0.12)', marginBottom: 8 }}
                      />
                    ) : (
                      <div style={{
                        width: 80, height: 80, borderRadius: '50%',
                        background: '#f0fdf4', border: '3px solid #fff',
                        boxShadow: '0 2px 8px rgba(0,0,0,0.12)',
                        display: 'flex', alignItems: 'center', justifyContent: 'center',
                        fontSize: 28, fontWeight: 700, color: '#3BB77E', marginBottom: 8,
                      }}>
                        {vendor.store_name?.charAt(0)?.toUpperCase()}
                      </div>
                    )}
                    <span className="btn btn-buy font-xs">
                      {meta.total} Products
                    </span>
                  </div>

                  <div className="info-vendor" style={{ flex: 1 }}>
                    <h4 className="mb-5 color-brand-3">{vendor.store_name}</h4>
                    {vendor.created_at && (
                      <span className="font-xs color-gray-500 mr-20">
                        Member since {new Date(vendor.created_at).getFullYear()}
                      </span>
                    )}
                    <div className="rating d-inline-block">
                      {[1, 2, 3, 4, 5].map(s => (
                        <img
                          key={s}
                          src={s <= Math.round(vendor.rating || 5)
                            ? '/assets/imgs/template/icons/star.svg'
                            : '/assets/imgs/template/icons/star-gray.svg'}
                          alt="star"
                          style={{ width: 14, height: 14 }}
                        />
                      ))}
                      <span className="font-xs color-gray-500 ml-5">
                        ({vendor.reviews_count || 0})
                      </span>
                    </div>
                    {vendor.description && (
                      <p className="font-sm color-gray-500 mt-5 mb-0">{vendor.description}</p>
                    )}
                  </div>

                  <div className="vendor-contact">
                    <div className="row">
                      {(vendor.address || vendor.city) && (
                        <div className="col-xl-7 col-lg-12">
                          <div className="d-inline-block font-md color-gray-500 location mb-10">
                            {[vendor.address, vendor.city, vendor.state, vendor.country].filter(Boolean).join(', ')}
                          </div>
                        </div>
                      )}
                      {vendor.phone && (
                        <div className="col-xl-5 col-lg-12">
                          <div className="d-inline-block font-md color-gray-500 phone">
                            {vendor.phone}
                          </div>
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* Vendor right: feature badges */}
            <div className="vendor-right" style={{ minWidth: 200 }}>
              <div className="box-featured-product">
                {[
                  { icon: '/assets/imgs/page/product/delivery.svg', title: 'Free Delivery', desc: 'From all orders over $10' },
                  { icon: '/assets/imgs/page/product/support.svg', title: 'Support 24/7', desc: 'Shop with an expert' },
                  { icon: '/assets/imgs/page/product/return.svg', title: 'Return & Refund', desc: 'Free return over $200' },
                  { icon: '/assets/imgs/page/product/payment.svg', title: 'Secure payment', desc: '100% Protected' },
                ].map(feat => (
                  <div key={feat.title} className="item-featured">
                    <div className="featured-icon">
                      <img src={feat.icon} alt={feat.title} />
                    </div>
                    <div className="featured-info">
                      <span className="font-sm-bold color-gray-1000">{feat.title}</span>
                      <p className="font-sm color-gray-500 font-medium">{feat.desc}</p>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>

          <div className="border-bottom mb-20 border-vendor mt-20"></div>

          {/* Products section */}
          <div className="row">
            <div className="col-lg-12">
              {/* Filters bar */}
              <div className="box-filters mt-0 pb-5 border-bottom">
                <div className="row">
                  <div className="col-xl-4 col-lg-4 mb-10 text-lg-start text-center">
                    <span className="font-sm color-gray-900 font-medium">
                      {productsLoading
                        ? 'Loading...'
                        : `Showing ${products.length} of ${meta.total} products`}
                    </span>
                  </div>
                  <div className="col-xl-8 col-lg-8 mb-10 text-lg-end text-center">
                    <div className="d-inline-block">
                      <span className="font-sm color-gray-500 font-medium">Sort by:</span>
                      <div className="dropdown dropdown-sort border-1-right d-inline-block">
                        <button
                          className="btn dropdown-toggle font-sm color-gray-900 font-medium"
                          type="button"
                          data-bs-toggle="dropdown"
                          aria-expanded="false"
                        >
                          {sortLabel[sortBy] || 'Latest products'}
                        </button>
                        <ul className="dropdown-menu dropdown-menu-light" style={{ margin: 0 }}>
                          {Object.entries(sortLabel).map(([val, label]) => (
                            <li key={val}>
                              <button
                                className={`dropdown-item${sortBy === val ? ' active' : ''}`}
                                onClick={() => { setSortBy(val); setPage(1) }}
                              >
                                {label}
                              </button>
                            </li>
                          ))}
                        </ul>
                      </div>
                    </div>
                    <div className="d-inline-block">
                      <span className="font-sm color-gray-500 font-medium">Show</span>
                      <div className="dropdown dropdown-sort border-1-right d-inline-block">
                        <button
                          className="btn dropdown-toggle font-sm color-gray-900 font-medium"
                          type="button"
                          data-bs-toggle="dropdown"
                          data-bs-display="static"
                        >
                          <span>{perPage} items</span>
                        </button>
                        <ul className="dropdown-menu dropdown-menu-light">
                          {[12, 24, 30, 48].map(n => (
                            <li key={n}>
                              <button
                                className={`dropdown-item${perPage === n ? ' active' : ''}`}
                                onClick={() => { setPerPage(n); setPage(1) }}
                              >
                                {n} items
                              </button>
                            </li>
                          ))}
                        </ul>
                      </div>
                    </div>
                    <div className="d-inline-block">
                      <a className="view-type-grid mr-5 active" href="#products-grid" onClick={e => e.preventDefault()}></a>
                    </div>
                  </div>
                </div>
              </div>

              {/* Products loading */}
              {productsLoading && (
                <div className="text-center py-5 mt-20">
                  <div className="spinner-border text-primary" role="status">
                    <span className="visually-hidden">Loading...</span>
                  </div>
                  <p className="mt-2 font-sm color-gray-500">Loading products...</p>
                </div>
              )}

              {/* Empty products */}
              {!productsLoading && products.length === 0 && (
                <div className="text-center py-5 mt-20">
                  <p className="font-md color-gray-500">No products available from this vendor yet.</p>
                  <Link className="btn btn-brand-2 mt-15" to="/shop">Browse All Products</Link>
                </div>
              )}

              {/* Products grid */}
              {!productsLoading && products.length > 0 && (
                <div id="products-grid" className="row mt-20">
                  {products.map(product => (
                    <ProductCard key={product.id} product={product} />
                  ))}
                </div>
              )}

              {/* Pagination */}
              {!productsLoading && meta.last_page > 1 && (
                <div className="mt-30">
                  <Pagination
                    currentPage={meta.current_page}
                    lastPage={meta.last_page}
                    onPageChange={handlePageChange}
                  />
                </div>
              )}
            </div>
          </div>
        </div>
      </section>
    </>
  )
}

export default VendorDetail
