import { useState, useEffect } from 'react'
import { Link, useSearchParams, useNavigate } from 'react-router-dom'
import { getProducts, getCategories, getVendors } from '../services/api'
import ProductCard from '../components/product/ProductCard'

const PRICE_RANGES = [
  { label: '$0 – $100', min: 0, max: 100 },
  { label: '$100 – $250', min: 100, max: 250 },
  { label: '$250 – $500', min: 250, max: 500 },
  { label: '$500 – $1000', min: 500, max: 1000 },
  { label: 'Over $1000', min: 1000, max: null },
]

const SORT_OPTIONS = [
  { value: 'newest', label: 'Latest products' },
  { value: 'oldest', label: 'Oldest products' },
  { value: 'price_asc', label: 'Price Low to High' },
  { value: 'price_desc', label: 'Price High to Low' },
  { value: 'name_asc', label: 'Name A–Z' },
]

const PER_PAGE_OPTIONS = [12, 24, 48]

export default function Shop() {
  const [searchParams, setSearchParams] = useSearchParams()
  const navigate = useNavigate()

  // Read URL params
  const currentPage = parseInt(searchParams.get('page') || '1', 10)
  const currentSort = searchParams.get('sort') || 'newest'
  const currentCategory = searchParams.get('category') || ''
  const currentSearch = searchParams.get('search') || ''
  const currentPerPage = parseInt(searchParams.get('per_page') || '12', 10)
  const currentMinPrice = searchParams.get('min_price') || ''
  const currentMaxPrice = searchParams.get('max_price') || ''
  const currentVendor = searchParams.get('vendor_id') || ''

  // State
  const [products, setProducts] = useState([])
  const [categories, setCategories] = useState([])
  const [vendors, setVendors] = useState([])
  const [totalProducts, setTotalProducts] = useState(0)
  const [totalPages, setTotalPages] = useState(1)
  const [loading, setLoading] = useState(true)

  // Price range checkbox state
  const [selectedPriceRange, setSelectedPriceRange] = useState(null)

  // ─── Fetch sidebar data ────────────────────────────────────
  useEffect(() => {
    const loadSidebar = async () => {
      try {
        const [catRes, vendRes] = await Promise.allSettled([
          getCategories(),
          getVendors({ per_page: 20 }),
        ])
        if (catRes.status === 'fulfilled') {
          const d = catRes.value.data
          setCategories(d.data || d.categories || d || [])
        }
        if (vendRes.status === 'fulfilled') {
          const d = vendRes.value.data
          setVendors(d.data || d.vendors || d || [])
        }
      } catch {
        // ignore
      }
    }
    loadSidebar()
  }, [])

  useEffect(() => {
    let cancelled = false

    const run = async () => {
      setLoading(true)
      try {
        const params = {
          page: currentPage,
          per_page: currentPerPage,
          sort: currentSort,
        }
        if (currentCategory) params.category = currentCategory
        if (currentSearch) params.search = currentSearch
        if (currentMinPrice) params.min_price = currentMinPrice
        if (currentMaxPrice) params.max_price = currentMaxPrice
        if (currentVendor) params.vendor_id = currentVendor

        const res = await getProducts(params)
        if (cancelled) return
        const d = res.data
        const items = Array.isArray(d.data) ? d.data : (Array.isArray(d) ? d : [])
        const total = d.total ?? items.length
        const lastPage = d.last_page ?? Math.ceil(total / currentPerPage) ?? 1
        setProducts(items)
        setTotalProducts(total)
        setTotalPages(lastPage || 1)
      } catch (err) {
        if (cancelled) return
        console.error('Failed to load products:', err?.response?.data || err?.message || err)
        setProducts([])
        setTotalProducts(0)
        setTotalPages(1)
      } finally {
        if (!cancelled) setLoading(false)
      }
    }

    run()
    return () => { cancelled = true }
  }, [currentPage, currentPerPage, currentSort, currentCategory, currentSearch, currentMinPrice, currentMaxPrice, currentVendor])

  // ─── URL param helpers ─────────────────────────────────────
  const updateParam = (key, value) => {
    const next = new URLSearchParams(searchParams)
    if (value) {
      next.set(key, value)
    } else {
      next.delete(key)
    }
    next.set('page', '1') // reset page on filter change
    setSearchParams(next)
  }

  const handleCategoryClick = (slug) => {
    updateParam('category', slug === currentCategory ? '' : slug)
  }

  const handleSortChange = (value) => {
    updateParam('sort', value)
  }

  const handlePerPageChange = (value) => {
    const next = new URLSearchParams(searchParams)
    next.set('per_page', value)
    next.set('page', '1')
    setSearchParams(next)
  }

  const handlePriceRange = (range) => {
    if (selectedPriceRange === range.label) {
      // Deselect
      setSelectedPriceRange(null)
      const next = new URLSearchParams(searchParams)
      next.delete('min_price')
      next.delete('max_price')
      next.set('page', '1')
      setSearchParams(next)
    } else {
      setSelectedPriceRange(range.label)
      const next = new URLSearchParams(searchParams)
      next.set('min_price', range.min)
      if (range.max) next.set('max_price', range.max)
      else next.delete('max_price')
      next.set('page', '1')
      setSearchParams(next)
    }
  }

  const handleVendorFilter = (vendorId) => {
    updateParam('vendor_id', vendorId === currentVendor ? '' : vendorId)
  }

  const handlePageChange = (page) => {
    const next = new URLSearchParams(searchParams)
    next.set('page', page)
    setSearchParams(next)
    window.scrollTo({ top: 0, behavior: 'smooth' })
  }

  const handleViewGrid = (e) => {
    e.preventDefault()
    // Grid is the default view for this page
  }

  const handleViewList = (e) => {
    e.preventDefault()
    navigate('/shop/list?' + searchParams.toString())
  }

  // ─── Pagination range ──────────────────────────────────────
  const buildPageNumbers = () => {
    const pages = []
    const delta = 2
    const left = Math.max(1, currentPage - delta)
    const right = Math.min(totalPages, currentPage + delta)
    for (let i = left; i <= right; i++) pages.push(i)
    return pages
  }

  const currentSortLabel = SORT_OPTIONS.find((o) => o.value === currentSort)?.label || 'Latest products'
  const showingFrom = totalProducts === 0 ? 0 : (currentPage - 1) * currentPerPage + 1
  const showingTo = Math.min(currentPage * currentPerPage, totalProducts)

  return (
    <main className="main">

      {/* ── Breadcrumb ────────────────────────────────────────── */}
      <div className="section-box">
        <div className="breadcrumbs-div">
          <div className="container">
            <ul className="breadcrumb">
              <li>
                <Link className="font-xs color-gray-1000" to="/">Home</Link>
              </li>
              {currentCategory ? (
                <>
                  <li>
                    <Link className="font-xs color-gray-500" to="/shop">Shop</Link>
                  </li>
                  <li>
                    <a className="font-xs color-gray-500" href="#">
                      {categories.find((c) => c.slug === currentCategory)?.name || currentCategory}
                    </a>
                  </li>
                </>
              ) : (
                <li>
                  <a className="font-xs color-gray-500" href="#">Shop</a>
                </li>
              )}
              {currentSearch && (
                <li>
                  <a className="font-xs color-gray-500" href="#">
                    Search: {currentSearch}
                  </a>
                </li>
              )}
            </ul>
          </div>
        </div>
      </div>

      {/* ── Main Shop Layout ──────────────────────────────────── */}
      <div className="section-box shop-template mt-30">
        <div className="container">
          <div className="row">

            {/* ── Products Column ─────────────────────────────── */}
            <div className="col-lg-9 order-first order-lg-last">

              {/* Filter Bar */}
              <div className="box-filters mt-0 pb-5 border-bottom">
                <div className="row">
                  {/* Mobile filter trigger */}
                  <div className="col-xl-2 col-lg-3 mb-10 text-lg-start text-center">
                    <a
                      className="btn btn-filter font-sm color-brand-3 font-medium"
                      href="#ModalFiltersForm"
                      data-bs-toggle="modal"
                    >
                      All Filters
                    </a>
                  </div>

                  {/* Sort / Show / View */}
                  <div className="col-xl-10 col-lg-9 mb-10 text-lg-end text-center">
                    <span className="font-sm color-gray-900 font-medium border-1-right span">
                      {loading
                        ? 'Loading products...'
                        : `Showing ${showingFrom}–${showingTo} of ${totalProducts} results`}
                    </span>

                    {/* Sort dropdown */}
                    <div className="d-inline-block">
                      <span className="font-sm color-gray-500 font-medium">Sort by:</span>
                      <div className="dropdown dropdown-sort border-1-right">
                        <button
                          className="btn dropdown-toggle font-sm color-gray-900 font-medium"
                          id="dropdownSort"
                          type="button"
                          data-bs-toggle="dropdown"
                          aria-expanded="false"
                        >
                          {currentSortLabel}
                        </button>
                        <ul className="dropdown-menu dropdown-menu-light" aria-labelledby="dropdownSort">
                          {SORT_OPTIONS.map((opt) => (
                            <li key={opt.value}>
                              <a
                                className={`dropdown-item${currentSort === opt.value ? ' active' : ''}`}
                                href="#"
                                onClick={(e) => { e.preventDefault(); handleSortChange(opt.value) }}
                              >
                                {opt.label}
                              </a>
                            </li>
                          ))}
                        </ul>
                      </div>
                    </div>

                    {/* Per-page dropdown */}
                    <div className="d-inline-block">
                      <span className="font-sm color-gray-500 font-medium">Show</span>
                      <div className="dropdown dropdown-sort border-1-right">
                        <button
                          className="btn dropdown-toggle font-sm color-gray-900 font-medium"
                          id="dropdownSort2"
                          type="button"
                          data-bs-toggle="dropdown"
                          aria-expanded="false"
                          data-bs-display="static"
                        >
                          <span>{currentPerPage} items</span>
                        </button>
                        <ul className="dropdown-menu dropdown-menu-light" aria-labelledby="dropdownSort2">
                          {PER_PAGE_OPTIONS.map((n) => (
                            <li key={n}>
                              <a
                                className={`dropdown-item${currentPerPage === n ? ' active' : ''}`}
                                href="#"
                                onClick={(e) => { e.preventDefault(); handlePerPageChange(n) }}
                              >
                                {n} items
                              </a>
                            </li>
                          ))}
                        </ul>
                      </div>
                    </div>

                    {/* View type */}
                    <div className="d-inline-block">
                      <a className="view-type-grid mr-5 active" href="#" onClick={handleViewGrid}></a>
                      <a className="view-type-list" href="#" onClick={handleViewList}></a>
                    </div>
                  </div>
                </div>
              </div>

              {/* Products Grid */}
              <div id="products-container" className="row mt-20">
                {loading ? (
                  <div className="col-12 text-center py-5">
                    <div className="spinner-border text-primary" role="status">
                      <span className="visually-hidden">Loading...</span>
                    </div>
                  </div>
                ) : products.length > 0 ? (
                  products.map((product) => (
                    <ProductCard key={product.id} product={product} />
                  ))
                ) : (
                  <div className="col-12 text-center py-5">
                    <p className="color-gray-500 font-md">No products found matching your criteria.</p>
                    <Link className="btn btn-brand-2 mt-15" to="/shop">
                      Clear Filters
                    </Link>
                  </div>
                )}
              </div>

              {/* Pagination */}
              {!loading && totalPages > 1 && (
                <nav>
                  <ul className="pagination">
                    <li className={`page-item${currentPage === 1 ? ' disabled' : ''}`}>
                      <a
                        className="page-link page-prev"
                        href="#"
                        onClick={(e) => { e.preventDefault(); if (currentPage > 1) handlePageChange(currentPage - 1) }}
                      ></a>
                    </li>
                    {buildPageNumbers().map((page) => (
                      <li key={page} className="page-item">
                        <a
                          className={`page-link${page === currentPage ? ' active' : ''}`}
                          href="#"
                          onClick={(e) => { e.preventDefault(); handlePageChange(page) }}
                        >
                          {page}
                        </a>
                      </li>
                    ))}
                    <li className={`page-item${currentPage === totalPages ? ' disabled' : ''}`}>
                      <a
                        className="page-link page-next"
                        href="#"
                        onClick={(e) => { e.preventDefault(); if (currentPage < totalPages) handlePageChange(currentPage + 1) }}
                      ></a>
                    </li>
                  </ul>
                </nav>
              )}
            </div>

            {/* ── Sidebar ─────────────────────────────────────── */}
            <div className="col-lg-3 order-last order-lg-first">

              {/* Categories */}
              <div className="sidebar-border mb-0">
                <div className="sidebar-head">
                  <h6 className="color-gray-900">Product Categories</h6>
                </div>
                <div className="sidebar-content">
                  <ul className="list-nav-arrow">
                    <li>
                      <a
                        href="#"
                        className={!currentCategory ? 'active' : ''}
                        onClick={(e) => { e.preventDefault(); handleCategoryClick('') }}
                      >
                        All Categories
                      </a>
                    </li>
                    {categories.length === 0 ? (
                      <li><a href="#">Loading...</a></li>
                    ) : (
                      categories.map((cat) => (
                        <li key={cat.id}>
                          <a
                            href="#"
                            className={currentCategory === cat.slug ? 'active' : ''}
                            onClick={(e) => { e.preventDefault(); handleCategoryClick(cat.slug) }}
                          >
                            {cat.name}
                            {cat.products_count !== undefined && (
                              <span className="font-xs color-gray-500"> ({cat.products_count})</span>
                            )}
                          </a>
                        </li>
                      ))
                    )}
                  </ul>
                </div>
              </div>

              {/* Price Filter */}
              <div className="sidebar-border mb-40">
                <div className="sidebar-head">
                  <h6 className="color-gray-900">Products Filter</h6>
                </div>
                <div className="sidebar-content">
                  <h6 className="color-gray-900 mt-10 mb-10">Price</h6>
                  <div className="box-slider-range mt-20 mb-15">
                    <div className="row mb-20">
                      <div className="col-sm-12">
                        <div id="slider-range"></div>
                      </div>
                    </div>
                    <div className="row">
                      <div className="col-lg-12">
                        <label className="lb-slider font-sm color-gray-500">Price Range:</label>
                        <span className="min-value-money font-sm color-gray-1000">
                          ${currentMinPrice || '0'}
                        </span>
                        <label className="lb-slider font-sm font-medium color-gray-1000"> - </label>
                        <span className="max-value-money font-sm font-medium color-gray-1000">
                          ${currentMaxPrice || '1000+'}
                        </span>
                      </div>
                    </div>
                  </div>
                  <ul className="list-checkbox">
                    {PRICE_RANGES.map((range) => (
                      <li key={range.label}>
                        <label className="cb-container">
                          <input
                            type="checkbox"
                            checked={selectedPriceRange === range.label}
                            onChange={() => handlePriceRange(range)}
                          />
                          <span className="text-small">{range.label}</span>
                          <span className="checkmark"></span>
                        </label>
                      </li>
                    ))}
                  </ul>

                  {/* Brands / Vendors */}
                  {vendors.length > 0 && (
                    <>
                      <h6 className="color-gray-900 mt-20 mb-10">Brands</h6>
                      <ul className="list-checkbox">
                        {vendors.map((v) => (
                          <li key={v.id}>
                            <label className="cb-container">
                              <input
                                type="checkbox"
                                checked={currentVendor === String(v.id)}
                                onChange={() => handleVendorFilter(String(v.id))}
                              />
                              <span className="text-small">{v.store_name || v.name}</span>
                              <span className="checkmark"></span>
                            </label>
                          </li>
                        ))}
                      </ul>
                    </>
                  )}
                </div>
              </div>

              {/* Product Tags */}
              <div className="box-slider-item">
                <div className="head pb-15 border-brand-2">
                  <h5 className="color-gray-900">Product Tags</h5>
                </div>
                <div className="content-slider mb-50">
                  {['Electronics', 'Gaming', 'Cellphone', 'Accessories', 'VR', 'USB', 'Camera', 'Laptop', 'Keyboard'].map(
                    (tag) => (
                      <Link
                        key={tag}
                        className="btn btn-border mr-5"
                        to={`/shop?search=${tag}`}
                      >
                        {tag}
                      </Link>
                    )
                  )}
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* ── Advanced Filters Modal ────────────────────────────── */}
      <div
        className="modal fade"
        id="ModalFiltersForm"
        tabIndex="-1"
        aria-hidden="true"
        style={{ display: 'none' }}
      >
        <div className="modal-dialog modal-xl">
          <div className="modal-content apply-job-form">
            <div className="modal-header">
              <h5 className="modal-title color-gray-1000 filters-icon">Advanced Filters</h5>
              <button className="btn-close" type="button" data-bs-dismiss="modal" aria-Label="Close"></button>
            </div>
            <div className="modal-body p-30">
              <div className="row">
                {/* Brands column */}
                <div className="col-w-1">
                  <h6 className="color-gray-900 mb-0">Brands</h6>
                  <ul className="list-checkbox">
                    {vendors.length > 0 ? (
                      vendors.map((v) => (
                        <li key={v.id}>
                          <label className="cb-container">
                            <input
                              type="checkbox"
                              checked={currentVendor === String(v.id)}
                              onChange={() => handleVendorFilter(String(v.id))}
                            />
                            <span className="text-small">{v.store_name || v.name}</span>
                            <span className="checkmark"></span>
                          </label>
                        </li>
                      ))
                    ) : (
                      ['Apple', 'Samsung', 'Sony', 'HP', 'Dell', 'Logitech'].map((b) => (
                        <li key={b}>
                          <label className="cb-container">
                            <input type="checkbox" />
                            <span className="text-small">{b}</span>
                            <span className="checkmark"></span>
                          </label>
                        </li>
                      ))
                    )}
                  </ul>
                </div>

                {/* Special offers column */}
                <div className="col-w-1">
                  <h6 className="color-gray-900 mb-0">Special offers</h6>
                  <ul className="list-checkbox">
                    {['On sale', 'FREE shipping', 'Big deals', 'Shop Mall'].map((offer) => (
                      <li key={offer}>
                        <label className="cb-container">
                          <input type="checkbox" />
                          <span className="text-small">{offer}</span>
                          <span className="checkmark"></span>
                        </label>
                      </li>
                    ))}
                  </ul>
                  <h6 className="color-gray-900 mb-0 mt-40">Ready to ship in</h6>
                  <ul className="list-checkbox">
                    {['1 business day', '1–3 business days', 'in 1 week', 'Shipping now'].map((s) => (
                      <li key={s}>
                        <label className="cb-container">
                          <input type="checkbox" />
                          <span className="text-small">{s}</span>
                          <span className="checkmark"></span>
                        </label>
                      </li>
                    ))}
                  </ul>
                </div>

                {/* Ordering options column */}
                <div className="col-w-1">
                  <h6 className="color-gray-900 mb-0">Ordering options</h6>
                  <ul className="list-checkbox">
                    {['Accepts gift cards', 'Customizable', 'Installment 0%', 'Buyer protection'].map(
                      (opt) => (
                        <li key={opt}>
                          <label className="cb-container">
                            <input type="checkbox" />
                            <span className="text-small">{opt}</span>
                            <span className="checkmark"></span>
                          </label>
                        </li>
                      )
                    )}
                  </ul>
                </div>

                {/* Price range column */}
                <div className="col-w-1">
                  <h6 className="color-gray-900 mb-0">Price Range</h6>
                  <ul className="list-checkbox">
                    {PRICE_RANGES.map((range) => (
                      <li key={range.label}>
                        <label className="cb-container">
                          <input
                            type="checkbox"
                            checked={selectedPriceRange === range.label}
                            onChange={() => handlePriceRange(range)}
                          />
                          <span className="text-small">{range.label}</span>
                          <span className="checkmark"></span>
                        </label>
                      </li>
                    ))}
                  </ul>
                </div>
              </div>

              {/* Modal apply button */}
              <div className="row mt-30">
                <div className="col-12 text-end">
                  <button className="btn btn-buy" data-bs-dismiss="modal">
                    Apply Filters
                  </button>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

    </main>
  )
}
