import { useState, useEffect } from 'react'
import { Link, useNavigate, useSearchParams } from 'react-router-dom'
import { resetPassword } from '../services/api'
import toast from 'react-hot-toast'

const ResetPassword = () => {
  const [searchParams] = useSearchParams()
  const navigate = useNavigate()
  const [form, setForm] = useState({
    token: searchParams.get('token') || '',
    email: searchParams.get('email') || '',
    password: '',
    password_confirmation: '',
  })
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState('')

  useEffect(() => {
    if (!form.token || !form.email) {
      toast.error('Invalid reset link')
      navigate('/forgot-password')
    }
  }, [])

  const handleChange = (e) => {
    setForm(f => ({ ...f, [e.target.name]: e.target.value }))
  }

  const handleSubmit = async (e) => {
    e.preventDefault()
    setError('')
    if (form.password !== form.password_confirmation) {
      setError('Passwords do not match')
      return
    }
    if (form.password.length < 8) {
      setError('Password must be at least 8 characters')
      return
    }
    setLoading(true)
    try {
      await resetPassword(form)
      toast.success('Password reset successfully!')
      navigate('/login')
    } catch (err) {
      setError(err?.response?.data?.message || 'Failed to reset password. The link may have expired.')
    } finally {
      setLoading(false)
    }
  }

  return (
    <main className="main">
      <div className="section-box">
        <div className="breadcrumbs-div">
          <div className="container">
            <ul className="breadcrumb">
              <li><Link className="font-xs color-gray-1000" to="/">Home</Link></li>
              <li><a className="font-xs color-gray-500">Reset Password</a></li>
            </ul>
          </div>
        </div>
      </div>

      <section className="section-box shop-template">
        <div className="container">
          <div className="row justify-content-center">
            <div className="col-lg-5 col-md-7 col-sm-10">
              <div className="box-border p-40">
                <h4 className="font-xl-bold color-brand-3 mb-5 text-center">Reset Password</h4>
                <p className="font-sm color-gray-500 text-center mb-30">
                  Create a new password for your account.
                </p>

                <form onSubmit={handleSubmit}>
                  <div className="form-group">
                    <input
                      className="form-control font-sm"
                      type="email"
                      name="email"
                      placeholder="Email"
                      value={form.email}
                      onChange={handleChange}
                      required
                      readOnly
                    />
                  </div>
                  <div className="form-group">
                    <input
                      className="form-control font-sm"
                      type="password"
                      name="password"
                      placeholder="New Password*"
                      value={form.password}
                      onChange={handleChange}
                      required
                    />
                  </div>
                  <div className="form-group">
                    <input
                      className="form-control font-sm"
                      type="password"
                      name="password_confirmation"
                      placeholder="Confirm New Password*"
                      value={form.password_confirmation}
                      onChange={handleChange}
                      required
                    />
                  </div>

                  {error && (
                    <div className="form-group">
                      <p className="font-sm" style={{ color: '#e30000' }}>{error}</p>
                    </div>
                  )}

                  <div className="form-group">
                    <button
                      type="submit"
                      className="btn btn-buy w-100"
                      disabled={loading}
                    >
                      {loading ? 'Resetting...' : 'Reset Password'}
                    </button>
                  </div>
                  <div className="text-center">
                    <Link className="font-sm color-brand-1" to="/login">Back to Login</Link>
                  </div>
                </form>
              </div>
            </div>
          </div>
        </div>
      </section>
    </main>
  )
}

export default ResetPassword
