import { useState } from 'react'
import { Link, useNavigate } from 'react-router-dom'
import { useAuth } from '../context/AuthContext'
import toast from 'react-hot-toast'

const Register = () => {
  const [form, setForm] = useState({
    name: '',
    email: '',
    password: '',
    password_confirmation: '',
  })
  const [loading, setLoading] = useState(false)
  const [errors, setErrors] = useState({})
  const [generalError, setGeneralError] = useState('')
  const { register } = useAuth()
  const navigate = useNavigate()

  const handleChange = e => {
    const { name, value } = e.target
    setForm(f => ({ ...f, [name]: value }))
    // Clear field-specific error on change
    if (errors[name]) setErrors(prev => ({ ...prev, [name]: '' }))
    setGeneralError('')
  }

  const handleRegister = async (e) => {
    e.preventDefault()
    setErrors({})
    setGeneralError('')

    // Client-side validation
    if (form.password !== form.password_confirmation) {
      setErrors({ password_confirmation: 'Passwords do not match' })
      toast.error('Passwords do not match')
      return
    }
    if (form.password.length < 8) {
      setErrors({ password: 'Password must be at least 8 characters' })
      toast.error('Password must be at least 8 characters')
      return
    }

    setLoading(true)
    try {
      await register(form)
      toast.success('Account created successfully! Welcome!')
      navigate('/')
    } catch (err) {
      const apiErrors = err?.response?.data?.errors
      if (apiErrors) {
        // Map Laravel validation errors
        const fieldErrors = {}
        Object.entries(apiErrors).forEach(([field, messages]) => {
          fieldErrors[field] = Array.isArray(messages) ? messages[0] : messages
          toast.error(Array.isArray(messages) ? messages[0] : messages)
        })
        setErrors(fieldErrors)
      } else {
        const msg = err?.response?.data?.message || 'Registration failed. Please try again.'
        setGeneralError(msg)
        toast.error(msg)
      }
    } finally {
      setLoading(false)
    }
  }

  return (
    <main className="main">
      <section className="section-box shop-template mt-60">
        <div className="container">
          <div className="row mb-100">
            <div className="col-lg-1"></div>

            {/* Register Form Column */}
            <div className="col-lg-5">
              <h3>Create an account</h3>
              <p className="font-md color-gray-500">
                Already have an account?{' '}
                <Link className="color-brand-3" to="/login">Sign in</Link>
              </p>

              <form className="form-register mt-30 mb-30" onSubmit={handleRegister}>
                {/* Full Name */}
                <div className="form-group">
                  <label className="mb-5 font-sm color-gray-700" htmlFor="register-name">
                    Full Name *
                  </label>
                  <input
                    className="form-control"
                    type="text"
                    id="register-name"
                    name="name"
                    placeholder="Enter your full name"
                    value={form.name}
                    onChange={handleChange}
                    required
                    autoComplete="name"
                  />
                  {errors.name && (
                    <span className="font-xs" style={{ color: '#e30000' }}>{errors.name}</span>
                  )}
                </div>

                {/* Email */}
                <div className="form-group">
                  <label className="mb-5 font-sm color-gray-700" htmlFor="register-email">
                    Email *
                  </label>
                  <input
                    className="form-control"
                    type="email"
                    id="register-email"
                    name="email"
                    placeholder="your@email.com"
                    value={form.email}
                    onChange={handleChange}
                    required
                    autoComplete="email"
                  />
                  {errors.email && (
                    <span className="font-xs" style={{ color: '#e30000' }}>{errors.email}</span>
                  )}
                </div>

                {/* Password */}
                <div className="form-group">
                  <label className="mb-5 font-sm color-gray-700" htmlFor="register-password">
                    Password *
                  </label>
                  <input
                    className="form-control"
                    type="password"
                    id="register-password"
                    name="password"
                    placeholder="Minimum 8 characters"
                    value={form.password}
                    onChange={handleChange}
                    required
                    minLength={8}
                    autoComplete="new-password"
                  />
                  {errors.password && (
                    <span className="font-xs" style={{ color: '#e30000' }}>{errors.password}</span>
                  )}
                </div>

                {/* Confirm Password */}
                <div className="form-group">
                  <label className="mb-5 font-sm color-gray-700" htmlFor="register-password-confirm">
                    Confirm Password *
                  </label>
                  <input
                    className="form-control"
                    type="password"
                    id="register-password-confirm"
                    name="password_confirmation"
                    placeholder="Re-enter your password"
                    value={form.password_confirmation}
                    onChange={handleChange}
                    required
                    autoComplete="new-password"
                  />
                  {errors.password_confirmation && (
                    <span className="font-xs" style={{ color: '#e30000' }}>
                      {errors.password_confirmation}
                    </span>
                  )}
                </div>

                {/* Terms Agreement */}
                <div className="form-group">
                  <label>
                    <input
                      className="checkagree"
                      type="checkbox"
                      id="register-terms"
                      required
                    />
                    {' '}By clicking Register button, you agree our{' '}
                    <Link to="/terms">terms and policy</Link>.
                  </label>
                </div>

                {/* General error */}
                {generalError && (
                  <div className="form-group">
                    <p className="font-sm" style={{ color: '#e30000', marginBottom: 0 }}>
                      {generalError}
                    </p>
                  </div>
                )}

                {/* Submit */}
                <div className="form-group">
                  <input
                    className="font-md-bold btn btn-buy"
                    type="submit"
                    value={loading ? 'Creating Account...' : 'Create Account'}
                    disabled={loading}
                  />
                </div>

                {/* Sign In link */}
                <div className="mt-20">
                  <span className="font-xs color-gray-500 font-medium">
                    Already have an account?
                  </span>
                  <Link className="font-xs color-brand-3 font-medium" to="/login">
                    {' '}Sign In
                  </Link>
                </div>
              </form>
            </div>

            {/* Right: Social Login Column */}
            <div className="col-lg-5">
              <div className="box-login-social pt-65 pl-50">
                <h5 className="text-center">Use Social Network Account</h5>
                <div className="box-button-login mt-25">
                  <a className="btn btn-login font-md-bold color-brand-3 mb-15">
                    Sign up with
                    <img
                      src="/assets/imgs/page/account/google.svg"
                      alt="Google"
                      onError={e => e.target.style.display = 'none'}
                    />
                  </a>
                  <a className="btn btn-login font-md-bold color-brand-3 mb-15">
                    Sign up with
                    <span className="color-blue font-md-bold">Facebook</span>
                  </a>
                  <a className="btn btn-login font-md-bold color-brand-3 mb-15">
                    Sign up with
                    <img
                      src="/assets/imgs/page/account/amazon.svg"
                      alt="Amazon"
                      onError={e => e.target.style.display = 'none'}
                    />
                  </a>
                </div>
                <div className="mt-10 text-center">
                  <span className="font-xs color-gray-900">Buying for work?</span>
                  <a className="color-brand-1 font-xs" href="#">Create a free business account</a>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>
    </main>
  )
}

export default Register
