import { useState, useEffect, useRef, useCallback } from 'react'
import { createPortal } from 'react-dom'
import { useParams, Link, useNavigate } from 'react-router-dom'
import { getProduct, getProducts, getReviews, addReview } from '../services/api'
import { useCart } from '../context/CartContext'
import { useWishlist } from '../context/WishlistContext'
import { useAuth } from '../context/AuthContext'
import { addToCompare } from '../services/api'
import ProductCard from '../components/product/ProductCard'
import toast from 'react-hot-toast'

// ─── Star rating display ───────────────────────────────────────
const Stars = ({ rating = 0, size = 16 }) => {
  const r = parseFloat(rating) || 0
  return (
    <span style={{ display: 'inline-flex', gap: 1 }}>
      {[1, 2, 3, 4, 5].map(s => {
        const full = s <= Math.floor(r)
        const half = !full && s - 0.5 <= r
        return (
          <svg key={s} width={size} height={size} viewBox="0 0 20 20" style={{ flexShrink: 0 }}>
            <defs>
              <linearGradient id={`hg-${s}`}>
                <stop offset="50%" stopColor={half ? '#f90' : full ? '#f90' : '#ddd'} />
                <stop offset="50%" stopColor={full ? '#f90' : '#ddd'} />
              </linearGradient>
            </defs>
            <polygon
              points="10,1 12.9,7 19.5,7.6 14.5,12 16.2,18.5 10,15 3.8,18.5 5.5,12 0.5,7.6 7.1,7"
              fill={full ? '#f90' : half ? `url(#hg-${s})` : '#ddd'}
            />
          </svg>
        )
      })}
    </span>
  )
}

// ─── Professional magnifier zoom gallery ──────────────────────
const ZOOM_FACTOR = 3   // 3× magnification
const PANEL_SIZE  = 380 // zoom result panel px

const Gallery = ({ images, discount }) => {
  const [active, setActive]   = useState(0)
  const [lens, setLens]       = useState(null) // { x, y, bgX, bgY, rect }
  const containerRef          = useRef()
  const imgs = images && images.length > 0 ? images : []
  const src  = imgs[active] ? (imgs[active].url || imgs[active]) : '/assets/imgs/no-image.png'

  // lens size as % of image container (1 / ZOOM_FACTOR)
  const LENS_PCT = 100 / ZOOM_FACTOR // ~33.3%

  const handleMouseMove = useCallback((e) => {
    const rect = containerRef.current?.getBoundingClientRect()
    if (!rect) return

    // cursor position as % of the container, clamped so lens never goes outside
    const half = LENS_PCT / 2
    const xPct = Math.max(half, Math.min(100 - half, ((e.clientX - rect.left)  / rect.width)  * 100))
    const yPct = Math.max(half, Math.min(100 - half, ((e.clientY - rect.top)   / rect.height) * 100))

    // map lens position to background-position of the zoomed panel
    const bgX = ((xPct - half) / (100 - LENS_PCT)) * 100
    const bgY = ((yPct - half) / (100 - LENS_PCT)) * 100

    setLens({ x: xPct, y: yPct, bgX, bgY, rect })
  }, [LENS_PCT])

  const handleMouseLeave = useCallback(() => setLens(null), [])

  // decide where to show the panel (right if room, else left)
  const panelRect = lens?.rect
  const hasRoomRight = panelRect && (window.innerWidth - panelRect.right) >= PANEL_SIZE + 20
  const panelLeft = panelRect
    ? (hasRoomRight ? panelRect.right + 12 : panelRect.left - PANEL_SIZE - 12)
    : 0
  const panelTop = panelRect
    ? Math.min(panelRect.top, window.innerHeight - PANEL_SIZE - 8)
    : 0

  return (
    <div style={{ display: 'flex', gap: 10 }}>

      {/* Vertical thumbnails */}
      {imgs.length > 1 && (
        <div style={{ display: 'flex', flexDirection: 'column', gap: 6, width: 62, flexShrink: 0 }}>
          {imgs.map((img, i) => {
            const thumb = img?.url || img || '/assets/imgs/no-image.png'
            return (
              <div key={i} onClick={() => { setActive(i); setLens(null) }} style={{
                width: 60, height: 60,
                border: active === i ? '2px solid #f90' : '1px solid #ddd',
                borderRadius: 6, overflow: 'hidden', cursor: 'pointer', background: '#fff',
                display: 'flex', alignItems: 'center', justifyContent: 'center',
                transition: 'border-color .15s',
              }}>
                <img src={thumb} alt="" style={{ width: '100%', height: '100%', objectFit: 'contain', padding: 4 }}
                  onError={e => { e.target.src = '/assets/imgs/no-image.png' }} />
              </div>
            )
          })}
        </div>
      )}

      {/* Main image */}
      <div style={{ flex: 1 }}>
        {/* Image container */}
        <div
          ref={containerRef}
          onMouseMove={handleMouseMove}
          onMouseLeave={handleMouseLeave}
          style={{
            border: '1px solid #e8e8e8', borderRadius: 10, background: '#fff',
            aspectRatio: '1 / 1', position: 'relative', overflow: 'hidden',
            cursor: lens ? 'crosshair' : 'zoom-in', userSelect: 'none',
          }}
        >
          {/* Discount badge */}
          {discount && (
            <span style={{
              position: 'absolute', top: 12, left: 12, zIndex: 2,
              background: '#cc0c39', color: '#fff', borderRadius: 4,
              padding: '3px 10px', fontSize: 13, fontWeight: 700,
            }}>-{discount}%</span>
          )}

          {/* Product image */}
          <img
            src={src} alt="product"
            draggable={false}
            style={{ width: '100%', height: '100%', objectFit: 'contain', padding: 16, display: 'block' }}
            onError={e => { e.target.src = '/assets/imgs/no-image.png' }}
          />

          {/* Magnifier lens overlay */}
          {lens && (
            <div style={{
              position: 'absolute',
              width:  `${LENS_PCT}%`,
              height: `${LENS_PCT}%`,
              left:   `${lens.x - LENS_PCT / 2}%`,
              top:    `${lens.y - LENS_PCT / 2}%`,
              border: '2px solid #f90',
              background: 'rgba(255, 214, 20, 0.18)',
              pointerEvents: 'none',
              borderRadius: 4,
              boxShadow: '0 0 0 1px rgba(0,0,0,0.08)',
            }} />
          )}
        </div>

        {/* Dot indicators */}
        {imgs.length > 1 && (
          <div style={{ display: 'flex', justifyContent: 'center', gap: 6, marginTop: 10 }}>
            {imgs.map((_, i) => (
              <span key={i} onClick={() => setActive(i)} style={{
                width: 8, height: 8, borderRadius: '50%', cursor: 'pointer', display: 'inline-block',
                background: active === i ? '#f90' : '#ddd', transition: 'background .2s',
              }} />
            ))}
          </div>
        )}

        <p style={{ fontSize: 11, color: '#aaa', textAlign: 'center', marginTop: 6 }}>
          Hover to zoom · {imgs.length > 1 ? 'Click thumbnail to switch image' : ''}
        </p>
      </div>

      {/* Zoom result panel — rendered via portal so it's never clipped */}
      {lens && createPortal(
        <div style={{
          position: 'fixed',
          left: panelLeft,
          top: panelTop,
          width: PANEL_SIZE,
          height: PANEL_SIZE,
          backgroundImage: `url(${src})`,
          backgroundRepeat: 'no-repeat',
          backgroundSize: `${ZOOM_FACTOR * 100}%`,
          backgroundPosition: `${lens.bgX}% ${lens.bgY}%`,
          border: '1px solid #ddd',
          borderRadius: 10,
          boxShadow: '0 8px 40px rgba(0,0,0,0.18)',
          zIndex: 9999,
          pointerEvents: 'none',
          overflow: 'hidden',
          background: `#fff url(${src}) no-repeat ${lens.bgX}% ${lens.bgY}% / ${ZOOM_FACTOR * 100}%`,
        }}/>,
        document.body
      )}
    </div>
  )
}

// ─── Review form ──────────────────────────────────────────────
const ReviewForm = ({ productId, onSuccess }) => {
  const { isLoggedIn } = useAuth()
  const [rating, setRating] = useState(5)
  const [hover, setHover] = useState(0)
  const [title, setTitle] = useState('')
  const [body, setBody] = useState('')
  const [submitting, setSubmitting] = useState(false)
  const labels = ['', 'Poor', 'Fair', 'Good', 'Very Good', 'Excellent']

  const handleSubmit = async (e) => {
    e.preventDefault()
    if (!body.trim()) { toast.error('Please write a review'); return }
    setSubmitting(true)
    try {
      await addReview(productId, { rating, title: title.trim() || undefined, body })
      toast.success('Review submitted! It will appear after approval.')
      setBody(''); setTitle(''); setRating(5)
      if (onSuccess) onSuccess()
    } catch { toast.error('Failed to submit review') }
    finally { setSubmitting(false) }
  }

  if (!isLoggedIn) return (
    <div style={{ background: '#fef9e7', border: '1px solid #f0e68c', borderRadius: 8, padding: 16 }}>
      <p style={{ margin: 0, fontSize: 14 }}>
        Please <Link to="/login" style={{ color: '#007185', fontWeight: 600 }}>sign in</Link> to write a review.
      </p>
    </div>
  )

  return (
    <form onSubmit={handleSubmit}>
      <div style={{ marginBottom: 16 }}>
        <div style={{ display: 'flex', alignItems: 'center', gap: 8, marginBottom: 4 }}>
          {[1, 2, 3, 4, 5].map(s => (
            <svg key={s} width={28} height={28} viewBox="0 0 20 20" style={{ cursor: 'pointer' }}
              onMouseEnter={() => setHover(s)} onMouseLeave={() => setHover(0)}
              onClick={() => setRating(s)}>
              <polygon points="10,1 12.9,7 19.5,7.6 14.5,12 16.2,18.5 10,15 3.8,18.5 5.5,12 0.5,7.6 7.1,7"
                fill={(hover || rating) >= s ? '#f90' : '#ddd'} />
            </svg>
          ))}
          <span style={{ fontSize: 14, color: '#555', marginLeft: 4 }}>{labels[hover || rating]}</span>
        </div>
      </div>
      <input className="form-control" style={{ marginBottom: 10, fontSize: 14 }}
        placeholder="Add a headline (optional)" value={title} onChange={e => setTitle(e.target.value)} />
      <textarea className="form-control" rows={4} style={{ fontSize: 14, marginBottom: 12 }}
        placeholder="Share your experience with this product..."
        value={body} onChange={e => setBody(e.target.value)} />
      <button type="submit" disabled={submitting}
        style={{
          background: '#ffd814', border: '1px solid #f0c14b', borderRadius: 20,
          padding: '8px 24px', fontWeight: 600, fontSize: 14, cursor: 'pointer',
        }}>
        {submitting ? 'Submitting...' : 'Submit Review'}
      </button>
    </form>
  )
}

// ─── Main component ───────────────────────────────────────────
export default function ProductDetail() {
  const { slug } = useParams()
  const navigate = useNavigate()
  const { addToCart } = useCart()
  const { addToWishlist, removeFromWishlist, isInWishlist } = useWishlist()
  const { isLoggedIn } = useAuth()

  const [product, setProduct] = useState(null)
  const [loading, setLoading] = useState(true)
  const [notFound, setNotFound] = useState(false)
  const [quantity, setQuantity] = useState(1)
  const [reviews, setReviews] = useState([])
  const [relatedProducts, setRelatedProducts] = useState([])
  const [activeTab, setActiveTab] = useState('description')
  const [addingCart, setAddingCart] = useState(false)
  const [buyingNow, setBuyingNow] = useState(false)

  const inWishlist = product ? isInWishlist(product.id) : false

  useEffect(() => {
    if (!slug) return
    setLoading(true); setNotFound(false); setProduct(null); setQuantity(1)
    const load = async () => {
      try {
        const res = await getProduct(slug)
        const p = res.data.data || res.data.product || res.data
        setProduct(p)
        try {
          const relRes = await getProducts({ per_page: 6, category: p?.category?.slug })
          const items = relRes.data?.data?.data || relRes.data?.data || relRes.data?.products || []
          setRelatedProducts(items.filter(r => r.slug !== slug).slice(0, 5))
        } catch { setRelatedProducts([]) }
        try {
          const revRes = await getReviews(p.id)
          const revData = revRes.data?.data || revRes.data?.reviews || revRes.data || []
          setReviews(Array.isArray(revData) ? revData : [])
        } catch { setReviews([]) }
      } catch (err) {
        if (err.response?.status === 404) setNotFound(true)
      } finally { setLoading(false) }
    }
    load()
  }, [slug])

  const handleAddToCart = async () => {
    if (!product) return
    setAddingCart(true)
    try { await addToCart(product.id, quantity); toast.success('Added to cart!') }
    catch { toast.error('Failed to add to cart') }
    finally { setAddingCart(false) }
  }

  const handleBuyNow = async () => {
    if (!product) return
    setBuyingNow(true)
    try { await addToCart(product.id, quantity); navigate('/checkout') }
    catch { toast.error('Failed to add to cart'); setBuyingNow(false) }
  }

  const handleWishlist = async (e) => {
    e.preventDefault()
    if (!product) return
    try {
      if (inWishlist) { await removeFromWishlist(product.id); toast.success('Removed from wishlist') }
      else { await addToWishlist(product.id); toast.success('Added to wishlist!') }
    } catch { toast.error('Action failed') }
  }

  const handleCompare = async (e) => {
    e.preventDefault()
    try { await addToCompare(product.id); toast.success('Added to compare!') }
    catch { toast.error('Failed') }
  }

  const refreshReviews = async () => {
    try {
      const revRes = await getReviews(product.id)
      const revData = revRes.data?.data || revRes.data?.reviews || revRes.data || []
      setReviews(Array.isArray(revData) ? revData : [])
    } catch {}
  }

  if (loading) return (
    <main className="main">
      <div className="container py-5 text-center">
        <div style={{ display: 'flex', flexDirection: 'column', alignItems: 'center', gap: 16, padding: 60 }}>
          <div className="spinner-border text-primary"></div>
          <span style={{ color: '#888' }}>Loading product...</span>
        </div>
      </div>
    </main>
  )

  if (notFound || !product) return (
    <main className="main">
      <div className="container py-5 text-center">
        <h3 style={{ color: '#333', marginBottom: 12 }}>Product Not Found</h3>
        <p style={{ color: '#888', marginBottom: 24 }}>The product you're looking for doesn't exist.</p>
        <Link className="btn btn-brand-2" to="/shop">Back to Shop</Link>
      </div>
    </main>
  )

  // ─── Derived ──────────────────────────────────────────────
  const price = parseFloat(product.price || 0)
  const salePrice = product.sale_price ? parseFloat(product.sale_price) : null
  const displayPrice = salePrice || price
  const discount = salePrice ? Math.round((1 - salePrice / price) * 100) : null
  const rating = parseFloat(product.rating || 0)
  const reviewCount = product.reviews_count || reviews.length || 0

  const images = product.images?.length ? product.images
    : product.primary_image ? [product.primary_image] : []

  const attrs = product.attributes
    ? (typeof product.attributes === 'string' ? JSON.parse(product.attributes) : product.attributes)
    : null

  const bullets = product.short_description
    ? [product.short_description]
    : product.description
      ? product.description.replace(/<[^>]*>/g, '').split(/\.\s+/).filter(s => s.trim().length > 15).slice(0, 5)
      : []

  // ─── Styles ───────────────────────────────────────────────
  const S = {
    page: { background: '#fff', minHeight: '100vh' },
    breadcrumb: { background: '#f8f9fa', padding: '8px 0', borderBottom: '1px solid #eee', fontSize: 13 },
    section: { padding: '24px 0' },
    title: { fontSize: 22, fontWeight: 400, color: '#0F1111', lineHeight: 1.4, marginBottom: 8 },
    brand: { fontSize: 14, color: '#007185', textDecoration: 'none' },
    ratingRow: { display: 'flex', alignItems: 'center', gap: 8, marginBottom: 12, paddingBottom: 12, borderBottom: '1px solid #eee' },
    ratingLink: { fontSize: 14, color: '#007185', cursor: 'pointer', textDecoration: 'none' },
    priceBox: { marginBottom: 12, paddingBottom: 12, borderBottom: '1px solid #eee' },
    salePrice: { fontSize: 28, fontWeight: 700, color: '#B12704' },
    originalPrice: { fontSize: 15, color: '#888', textDecoration: 'line-through', marginLeft: 8 },
    saveBadge: { fontSize: 13, color: '#007600', fontWeight: 600, marginLeft: 8 },
    bullet: { fontSize: 14, color: '#333', marginBottom: 4, display: 'flex', alignItems: 'flex-start', gap: 8 },
    buyBox: {
      border: '1px solid #ddd', borderRadius: 8, padding: 20,
      position: 'sticky', top: 80, background: '#fff',
    },
    buyPrice: { fontSize: 26, fontWeight: 700, color: '#0F1111', marginBottom: 4 },
    deliveryRow: { fontSize: 14, marginBottom: 8 },
    stockBadge: { fontSize: 16, color: '#007600', fontWeight: 600, marginBottom: 12 },
    qtySelect: {
      border: '1px solid #888', borderRadius: 6, padding: '6px 10px',
      fontSize: 14, background: '#f0f2f2', cursor: 'pointer', marginBottom: 12, width: '100%',
    },
    btnCart: {
      width: '100%', background: '#ffd814', border: '1px solid #f0c14b',
      borderRadius: 20, padding: '10px 0', fontSize: 15, fontWeight: 600,
      cursor: 'pointer', marginBottom: 8, color: '#111',
    },
    btnBuy: {
      width: '100%', background: '#ffa41c', border: '1px solid #fa8900',
      borderRadius: 20, padding: '10px 0', fontSize: 15, fontWeight: 600,
      cursor: 'pointer', marginBottom: 12, color: '#111',
    },
    btnDisabled: { opacity: 0.6, cursor: 'not-allowed' },
    tabBtn: (active) => ({
      padding: '10px 20px', fontSize: 14, fontWeight: active ? 700 : 400,
      color: active ? '#C45500' : '#555', borderBottom: active ? '3px solid #C45500' : '3px solid transparent',
      background: 'none', border: 'none', cursor: 'pointer', whiteSpace: 'nowrap',
    }),
    reviewCard: {
      borderBottom: '1px solid #eee', paddingBottom: 20, marginBottom: 20,
    },
  }

  return (
    <main className="main" style={S.page}>

      {/* Breadcrumb */}
      <div style={S.breadcrumb}>
        <div className="container">
          <span style={{ color: '#007185' }}>
            <Link to="/" style={{ color: '#007185' }}>Home</Link>
            {product.category && <> &rsaquo; <Link to={`/shop?category=${product.category.slug}`} style={{ color: '#007185' }}>{product.category.name}</Link></>}
            {product.brand && <> &rsaquo; <span style={{ color: '#555' }}>{product.brand}</span></>}
            &rsaquo; <span style={{ color: '#555' }}>{product.name?.slice(0, 50)}{product.name?.length > 50 ? '…' : ''}</span>
          </span>
        </div>
      </div>

      {/* ── Main Product Section ───────────────────────────────── */}
      <div style={S.section}>
        <div className="container">
          <div className="row g-4">

            {/* Gallery */}
            <div className="col-lg-5 col-md-6">
              <Gallery images={images} discount={discount} />
            </div>

            {/* Product Info */}
            <div className="col-lg-4 col-md-6">
              {/* Brand */}
              {product.brand && (
                <div style={{ fontSize: 13, color: '#555', marginBottom: 4 }}>
                  Brand: <Link to={`/shop?brand=${product.brand}`} style={S.brand}>{product.brand}</Link>
                </div>
              )}

              {/* Title */}
              <h1 style={S.title}>{product.name}</h1>

              {/* Vendor */}
              {product.vendor && (
                <div style={{ fontSize: 13, color: '#555', marginBottom: 8 }}>
                  Sold by <Link to={`/vendors/${product.vendor.id}`} style={S.brand}>
                    {product.vendor.store_name || product.vendor.name}
                  </Link>
                </div>
              )}

              {/* Rating */}
              <div style={S.ratingRow}>
                <Stars rating={rating} size={18} />
                <a href="#reviews" style={S.ratingLink} onClick={e => { e.preventDefault(); setActiveTab('reviews') }}>
                  {rating.toFixed(1)}
                </a>
                <a href="#reviews" style={{ ...S.ratingLink, borderLeft: '1px solid #ddd', paddingLeft: 8 }}
                  onClick={e => { e.preventDefault(); setActiveTab('reviews') }}>
                  {reviewCount.toLocaleString()} {reviewCount === 1 ? 'rating' : 'ratings'}
                </a>
                {product.is_best_seller && (
                  <span style={{ background: '#c45500', color: '#fff', fontSize: 11, fontWeight: 700, padding: '2px 8px', borderRadius: 3 }}>
                    #1 Best Seller
                  </span>
                )}
              </div>

              {/* Price */}
              <div style={S.priceBox}>
                {salePrice ? (
                  <>
                    <div style={{ fontSize: 13, color: '#555' }}>List price: <span style={{ textDecoration: 'line-through' }}>${price.toFixed(2)}</span></div>
                    <div style={{ display: 'flex', alignItems: 'baseline', gap: 8, flexWrap: 'wrap' }}>
                      <span style={{ fontSize: 13, color: '#B12704' }}>Price:</span>
                      <span style={S.salePrice}>${salePrice.toFixed(2)}</span>
                      <span style={S.saveBadge}>Save {discount}% (${(price - salePrice).toFixed(2)})</span>
                    </div>
                  </>
                ) : (
                  <div style={{ display: 'flex', alignItems: 'baseline', gap: 6 }}>
                    <span style={{ fontSize: 13, color: '#555' }}>Price:</span>
                    <span style={{ ...S.salePrice, color: '#0F1111' }}>${price.toFixed(2)}</span>
                  </div>
                )}
              </div>

              {/* Bullet features */}
              {bullets.length > 0 && (
                <div style={{ marginBottom: 16 }}>
                  <div style={{ fontSize: 14, fontWeight: 600, marginBottom: 8, color: '#333' }}>About this item</div>
                  <ul style={{ paddingLeft: 0, listStyle: 'none', margin: 0 }}>
                    {bullets.map((b, i) => (
                      <li key={i} style={S.bullet}>
                        <span style={{ color: '#f90', fontSize: 18, lineHeight: 1, flexShrink: 0 }}>›</span>
                        <span>{b.trim()}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              )}

              {/* Attributes */}
              {attrs && Object.keys(attrs).length > 0 && (
                <div style={{ marginBottom: 16 }}>
                  <div style={{ fontSize: 14, fontWeight: 600, marginBottom: 8, color: '#333' }}>Technical Details</div>
                  <table style={{ width: '100%', fontSize: 13, borderCollapse: 'collapse' }}>
                    <tbody>
                      {Object.entries(attrs).slice(0, 6).map(([k, v]) => (
                        <tr key={k}>
                          <td style={{ padding: '4px 8px 4px 0', color: '#555', width: '40%', verticalAlign: 'top' }}>{k}</td>
                          <td style={{ padding: '4px 0', color: '#333', fontWeight: 500 }}>{String(v)}</td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              )}

              {/* SKU / Category */}
              <div style={{ fontSize: 13, color: '#555', lineHeight: 2 }}>
                {product.sku && <div>SKU: <span style={{ color: '#333' }}>{product.sku}</span></div>}
                {product.category && (
                  <div>Category: <Link to={`/shop?category=${product.category.slug}`} style={S.brand}>{product.category.name}</Link></div>
                )}
              </div>

              {/* Share / Wishlist / Compare */}
              <div style={{ display: 'flex', gap: 16, marginTop: 16, paddingTop: 16, borderTop: '1px solid #eee', flexWrap: 'wrap' }}>
                <a href="#" onClick={handleWishlist}
                  style={{ fontSize: 13, color: '#007185', display: 'flex', alignItems: 'center', gap: 4 }}>
                  <svg width={16} height={16} fill={inWishlist ? '#e31c3d' : 'none'} stroke={inWishlist ? '#e31c3d' : '#007185'} strokeWidth={2} viewBox="0 0 24 24">
                    <path d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z"/>
                  </svg>
                  {inWishlist ? 'In Wishlist' : 'Add to Wishlist'}
                </a>
                <a href="#" onClick={handleCompare}
                  style={{ fontSize: 13, color: '#007185', display: 'flex', alignItems: 'center', gap: 4 }}>
                  <svg width={16} height={16} fill="none" stroke="#007185" strokeWidth={2} viewBox="0 0 24 24">
                    <path d="M18 20V10M12 20V4M6 20v-6"/>
                  </svg>
                  Compare
                </a>
                <a href={`https://facebook.com/sharer/sharer.php?u=${encodeURIComponent(window.location.href)}`}
                  target="_blank" rel="noreferrer"
                  style={{ fontSize: 13, color: '#007185', display: 'flex', alignItems: 'center', gap: 4 }}>
                  <svg width={16} height={16} fill="#007185" viewBox="0 0 24 24">
                    <path d="M18 2h-3a5 5 0 0 0-5 5v3H7v4h3v8h4v-8h3l1-4h-4V7a1 1 0 0 1 1-1h3z"/>
                  </svg>
                  Share
                </a>
              </div>
            </div>

            {/* ── Buy Box ────────────────────────────────────────── */}
            <div className="col-lg-3 col-md-12">
              <div style={S.buyBox}>
                {/* Price */}
                <div style={{ marginBottom: 12 }}>
                  <span style={S.buyPrice}>${displayPrice.toFixed(2)}</span>
                  {salePrice && (
                    <span style={{ fontSize: 12, color: '#888', textDecoration: 'line-through', marginLeft: 6 }}>
                      ${price.toFixed(2)}
                    </span>
                  )}
                </div>

                {/* Delivery */}
                <div style={S.deliveryRow}>
                  <span style={{ color: '#007600', fontWeight: 600 }}>FREE Delivery</span>
                  <span style={{ color: '#555' }}> on all orders</span>
                </div>

                {/* Ships to */}
                <div style={{ fontSize: 13, color: '#555', marginBottom: 12 }}>
                  Ships to your location
                </div>

                {/* Stock */}
                <div style={S.stockBadge}>
                  {product.stock > 0
                    ? <>In Stock <span style={{ fontSize: 13, color: '#555', fontWeight: 400 }}>({product.stock} left)</span></>
                    : <span style={{ color: '#B12704' }}>Out of Stock</span>
                  }
                </div>

                {/* Seller */}
                {product.vendor && (
                  <div style={{ fontSize: 13, color: '#555', marginBottom: 12 }}>
                    Sold by <Link to={`/vendors/${product.vendor.id}`} style={S.brand}>
                      {product.vendor.store_name || product.vendor.name}
                    </Link>
                  </div>
                )}

                {/* Quantity */}
                <div style={{ fontSize: 13, marginBottom: 8 }}>
                  <label style={{ fontWeight: 600, marginBottom: 4, display: 'block' }}>Quantity:</label>
                  <div style={{ display: 'flex', alignItems: 'center', gap: 10, marginBottom: 12 }}>
                    <button onClick={() => setQuantity(q => Math.max(1, q - 1))}
                      style={{ width: 32, height: 32, borderRadius: '50%', border: '1px solid #888', background: '#f0f2f2', fontSize: 18, cursor: 'pointer', lineHeight: 1 }}>
                      −
                    </button>
                    <span style={{ fontSize: 16, fontWeight: 700, minWidth: 24, textAlign: 'center' }}>{quantity}</span>
                    <button onClick={() => setQuantity(q => q + 1)}
                      style={{ width: 32, height: 32, borderRadius: '50%', border: '1px solid #888', background: '#f0f2f2', fontSize: 18, cursor: 'pointer', lineHeight: 1 }}>
                      +
                    </button>
                  </div>
                </div>

                {/* Add to Cart */}
                <button
                  onClick={handleAddToCart}
                  disabled={addingCart || product.stock === 0}
                  style={{ ...S.btnCart, ...(product.stock === 0 ? S.btnDisabled : {}) }}
                >
                  {addingCart ? 'Adding...' : product.stock === 0 ? 'Out of Stock' : 'Add to Cart'}
                </button>

                {/* Buy Now */}
                <button
                  onClick={handleBuyNow}
                  disabled={buyingNow || product.stock === 0}
                  style={{ ...S.btnBuy, ...(product.stock === 0 ? S.btnDisabled : {}) }}
                >
                  {buyingNow ? 'Processing...' : 'Buy Now'}
                </button>

                {/* Trust badges */}
                <div style={{ fontSize: 12, color: '#555', borderTop: '1px solid #eee', paddingTop: 12 }}>
                  <div style={{ display: 'flex', alignItems: 'center', gap: 6, marginBottom: 6 }}>
                    <svg width={14} height={14} fill="none" stroke="#007600" strokeWidth={2} viewBox="0 0 24 24">
                      <path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/>
                    </svg>
                    Secure transaction
                  </div>
                  <div style={{ display: 'flex', alignItems: 'center', gap: 6, marginBottom: 6 }}>
                    <svg width={14} height={14} fill="none" stroke="#007600" strokeWidth={2} viewBox="0 0 24 24">
                      <polyline points="20 6 9 17 4 12"/>
                    </svg>
                    Returns accepted
                  </div>
                  <div style={{ display: 'flex', alignItems: 'center', gap: 6 }}>
                    <svg width={14} height={14} fill="none" stroke="#007600" strokeWidth={2} viewBox="0 0 24 24">
                      <rect x="1" y="3" width="15" height="13"/><polygon points="16 8 20 8 23 11 23 16 16 16 16 8"/>
                      <circle cx="5.5" cy="18.5" r="2.5"/><circle cx="18.5" cy="18.5" r="2.5"/>
                    </svg>
                    Free shipping
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* ── Tabs ──────────────────────────────────────────────── */}
      <div style={{ borderTop: '1px solid #eee', background: '#fff' }}>
        <div className="container">
          {/* Tab nav */}
          <div style={{ display: 'flex', gap: 0, overflowX: 'auto', borderBottom: '1px solid #eee' }}>
            {[
              ['description', 'Product Description'],
              ['specification', 'Technical Details'],
              ['reviews', `Customer Reviews (${reviewCount})`],
              ...(product.vendor ? [['vendor', 'Seller Info']] : []),
            ].map(([key, label]) => (
              <button key={key} style={S.tabBtn(activeTab === key)} onClick={() => setActiveTab(key)}>
                {label}
              </button>
            ))}
          </div>

          {/* Tab content */}
          <div style={{ padding: '32px 0' }}>

            {/* Description */}
            {activeTab === 'description' && (
              <div className="row">
                <div className="col-lg-9">
                  <h4 style={{ fontSize: 18, fontWeight: 700, marginBottom: 20, color: '#0F1111' }}>
                    Product Description
                  </h4>
                  {product.description ? (
                    <div style={{ fontSize: 14, lineHeight: 1.8, color: '#333' }}
                      dangerouslySetInnerHTML={{ __html: product.description }} />
                  ) : (
                    <p style={{ color: '#888', fontSize: 14 }}>No description available.</p>
                  )}
                </div>
              </div>
            )}

            {/* Technical Details */}
            {activeTab === 'specification' && (
              <div className="row">
                <div className="col-lg-8">
                  <h4 style={{ fontSize: 18, fontWeight: 700, marginBottom: 20, color: '#0F1111' }}>
                    Technical Details
                  </h4>
                  <table style={{ width: '100%', borderCollapse: 'collapse', fontSize: 14 }}>
                    <tbody>
                      {[
                        product.brand && ['Brand', product.brand],
                        product.sku && ['Model Number', product.sku],
                        product.category && ['Category', product.category.name],
                        product.vendor && ['Seller', product.vendor.store_name || product.vendor.name],
                        product.stock !== undefined && ['Availability', product.stock > 0 ? `${product.stock} in stock` : 'Out of Stock'],
                        product.weight && ['Weight', product.weight],
                        product.dimensions && ['Dimensions', product.dimensions],
                        ...(attrs ? Object.entries(attrs) : []),
                      ].filter(Boolean).map(([k, v], i) => (
                        <tr key={i} style={{ background: i % 2 === 0 ? '#f8f9fa' : '#fff' }}>
                          <td style={{ padding: '10px 16px', width: '35%', color: '#555', fontWeight: 600, borderBottom: '1px solid #eee' }}>{k}</td>
                          <td style={{ padding: '10px 16px', color: '#333', borderBottom: '1px solid #eee' }}>{String(v)}</td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              </div>
            )}

            {/* Reviews */}
            {activeTab === 'reviews' && (
              <div id="reviews">
                <div className="row g-5">
                  {/* Summary sidebar */}
                  <div className="col-lg-3">
                    <h4 style={{ fontSize: 18, fontWeight: 700, marginBottom: 16, color: '#0F1111' }}>
                      Customer Reviews
                    </h4>
                    <div style={{ display: 'flex', alignItems: 'center', gap: 10, marginBottom: 16 }}>
                      <Stars rating={rating} size={22} />
                      <span style={{ fontSize: 22, fontWeight: 700 }}>{rating.toFixed(1)}</span>
                      <span style={{ fontSize: 13, color: '#555' }}>out of 5</span>
                    </div>
                    <p style={{ fontSize: 13, color: '#555', marginBottom: 16 }}>
                      {reviewCount} global ratings
                    </p>
                    {[5, 4, 3, 2, 1].map(star => {
                      const count = reviews.filter(r => r.rating === star).length
                      const pct = reviews.length > 0 ? Math.round((count / reviews.length) * 100) : 0
                      return (
                        <div key={star} style={{ display: 'flex', alignItems: 'center', gap: 8, marginBottom: 6, fontSize: 13 }}>
                          <span style={{ color: '#007185', width: 40, whiteSpace: 'nowrap' }}>{star} star</span>
                          <div style={{ flex: 1, background: '#eee', height: 16, borderRadius: 3, overflow: 'hidden' }}>
                            <div style={{ width: `${pct}%`, background: '#f90', height: '100%', borderRadius: 3, transition: 'width .3s' }} />
                          </div>
                          <span style={{ width: 32, color: '#007185' }}>{pct}%</span>
                        </div>
                      )
                    })}

                    <div style={{ marginTop: 24, paddingTop: 16, borderTop: '1px solid #eee' }}>
                      <h6 style={{ fontSize: 15, fontWeight: 700, marginBottom: 8 }}>Review this product</h6>
                      <p style={{ fontSize: 13, color: '#555', marginBottom: 12 }}>Share your thoughts with other customers</p>
                      <ReviewForm productId={product.id} onSuccess={refreshReviews} />
                    </div>
                  </div>

                  {/* Review list */}
                  <div className="col-lg-9">
                    <h4 style={{ fontSize: 18, fontWeight: 700, marginBottom: 24, color: '#0F1111' }}>
                      Top Reviews
                    </h4>
                    {reviews.length === 0 ? (
                      <div style={{ background: '#fef9e7', border: '1px solid #f0e68c', borderRadius: 8, padding: 24, textAlign: 'center' }}>
                        <p style={{ fontSize: 15, color: '#555', margin: 0 }}>No reviews yet. Be the first to review this product!</p>
                      </div>
                    ) : (
                      reviews.map(review => (
                        <div key={review.id} style={S.reviewCard}>
                          <div style={{ display: 'flex', alignItems: 'center', gap: 10, marginBottom: 8 }}>
                            <div style={{
                              width: 36, height: 36, borderRadius: '50%', background: '#e8e8e8',
                              display: 'flex', alignItems: 'center', justifyContent: 'center',
                              fontWeight: 700, fontSize: 15, color: '#555', overflow: 'hidden', flexShrink: 0,
                            }}>
                              {review.user?.avatar
                                ? <img src={review.user.avatar} alt="" style={{ width: '100%', height: '100%', objectFit: 'cover' }} />
                                : (review.user?.name?.[0] || 'C').toUpperCase()
                              }
                            </div>
                            <span style={{ fontSize: 14, fontWeight: 600, color: '#0F1111' }}>
                              {review.user?.name || 'Customer'}
                            </span>
                          </div>
                          <div style={{ display: 'flex', alignItems: 'center', gap: 10, marginBottom: 6 }}>
                            <Stars rating={review.rating} size={16} />
                            {review.title && (
                              <span style={{ fontSize: 15, fontWeight: 700, color: '#0F1111' }}>{review.title}</span>
                            )}
                          </div>
                          <div style={{ fontSize: 12, color: '#888', marginBottom: 10 }}>
                            Reviewed on {new Date(review.created_at).toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' })}
                            <span style={{
                              marginLeft: 12, background: '#f0fdf4', color: '#007600',
                              fontSize: 11, padding: '2px 6px', borderRadius: 3, fontWeight: 600,
                            }}>Verified Purchase</span>
                          </div>
                          <p style={{ fontSize: 14, color: '#333', lineHeight: 1.7, margin: 0 }}>
                            {review.body || review.comment}
                          </p>
                        </div>
                      ))
                    )}
                  </div>
                </div>
              </div>
            )}

            {/* Vendor */}
            {activeTab === 'vendor' && product.vendor && (
              <div className="row">
                <div className="col-lg-8">
                  <div style={{ display: 'flex', alignItems: 'center', gap: 16, marginBottom: 24 }}>
                    {product.vendor.logo && (
                      <img src={product.vendor.logo} alt="" style={{ width: 80, height: 80, objectFit: 'contain', border: '1px solid #eee', borderRadius: 8, padding: 8 }} />
                    )}
                    <div>
                      <h4 style={{ fontSize: 20, fontWeight: 700, margin: 0 }}>
                        <Link to={`/vendors/${product.vendor.id}`} style={{ color: '#007185' }}>
                          {product.vendor.store_name || product.vendor.name}
                        </Link>
                      </h4>
                      <div style={{ fontSize: 13, color: '#555', marginTop: 4 }}>
                        {product.vendor.reviews_count || 0} ratings
                      </div>
                    </div>
                  </div>
                  <div style={{ fontSize: 14, lineHeight: 2, color: '#555' }}>
                    {product.vendor.address && <div>📍 {product.vendor.address}</div>}
                    {product.vendor.phone && <div>📞 {product.vendor.phone}</div>}
                    {product.vendor.email && <div>✉️ {product.vendor.email}</div>}
                  </div>
                  {product.vendor.description && (
                    <p style={{ fontSize: 14, color: '#555', marginTop: 16, lineHeight: 1.8 }}>{product.vendor.description}</p>
                  )}
                  <Link to={`/vendors/${product.vendor.id}`}
                    style={{ display: 'inline-block', marginTop: 16, fontSize: 14, color: '#007185', fontWeight: 600 }}>
                    See all products from this seller →
                  </Link>
                </div>
              </div>
            )}

          </div>
        </div>
      </div>

      {/* ── Related Products ──────────────────────────────────── */}
      {relatedProducts.length > 0 && (
        <div style={{ background: '#f8f9fa', padding: '32px 0', borderTop: '1px solid #eee' }}>
          <div className="container">
            <h4 style={{ fontSize: 20, fontWeight: 700, marginBottom: 24, color: '#0F1111' }}>
              Customers also viewed
            </h4>
            <div className="row g-3">
              {relatedProducts.map(p => <ProductCard key={p.id} product={p} />)}
            </div>
          </div>
        </div>
      )}

    </main>
  )
}
