import { useState } from 'react'
import { Link, useNavigate, useLocation } from 'react-router-dom'
import { useAuth } from '../context/AuthContext'
import toast from 'react-hot-toast'

const Login = () => {
  const [form, setForm] = useState({ email: '', password: '' })
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState('')
  const { login } = useAuth()
  const navigate = useNavigate()
  const location = useLocation()
  const from = location.state?.from?.pathname || '/'

  const handleChange = e => {
    setError('')
    setForm(f => ({ ...f, [e.target.name]: e.target.value }))
  }

  const handleLogin = async (e) => {
    e.preventDefault()
    setError('')
    setLoading(true)
    try {
      await login(form.email, form.password)
      toast.success('Welcome back!')
      navigate(from, { replace: true })
    } catch (err) {
      const msg = err?.response?.data?.message || 'Invalid email or password'
      setError(msg)
      toast.error(msg)
    } finally {
      setLoading(false)
    }
  }

  return (
    <main className="main">
      <section className="section-box shop-template mt-60">
        <div className="container">
          <div className="row mb-100">
            <div className="col-lg-1"></div>

            {/* Login Form Column */}
            <div className="col-lg-5">
              <h3>Member Login</h3>
              <p className="font-md color-gray-500">Welcome back!</p>

              <form className="form-register mt-30 mb-30" onSubmit={handleLogin}>
                {/* Email */}
                <div className="form-group">
                  <label className="mb-5 font-sm color-gray-700" htmlFor="login-email">
                    Email *
                  </label>
                  <input
                    className="form-control"
                    type="email"
                    id="login-email"
                    name="email"
                    placeholder="your@email.com"
                    value={form.email}
                    onChange={handleChange}
                    required
                    autoComplete="email"
                  />
                </div>

                {/* Password */}
                <div className="form-group">
                  <label className="mb-5 font-sm color-gray-700" htmlFor="login-password">
                    Password *
                  </label>
                  <input
                    className="form-control"
                    type="password"
                    id="login-password"
                    name="password"
                    placeholder="Enter your password"
                    value={form.password}
                    onChange={handleChange}
                    required
                    autoComplete="current-password"
                  />
                </div>

                {/* Remember me + Forgot password */}
                <div className="row">
                  <div className="col-lg-6">
                    <div className="form-group">
                      <label className="color-gray-500 font-xs">
                        <input
                          className="checkagree"
                          type="checkbox"
                          id="login-remember"
                        />
                        {' '}Remember me
                      </label>
                    </div>
                  </div>
                  <div className="col-lg-6 text-end">
                    <div className="form-group">
                      <Link className="font-xs color-gray-500" to="/forgot-password">
                        Forgot your password?
                      </Link>
                    </div>
                  </div>
                </div>

                {/* Error message */}
                {error && (
                  <div className="form-group">
                    <p className="font-sm" style={{ color: '#e30000', marginBottom: 0 }}>
                      {error}
                    </p>
                  </div>
                )}

                {/* Submit */}
                <div className="form-group">
                  <input
                    className="font-md-bold btn btn-buy"
                    type="submit"
                    value={loading ? 'Signing in...' : 'Sign In'}
                    disabled={loading}
                  />
                </div>

                {/* Sign Up link */}
                <div className="mt-20">
                  <span className="font-xs color-gray-500 font-medium">
                    Don't have an account?
                  </span>
                  <Link className="font-xs color-brand-3 font-medium" to="/register">
                    {' '}Sign Up
                  </Link>
                </div>
              </form>
            </div>

            {/* Right empty column (template matches col-lg-5 empty) */}
            <div className="col-lg-5"></div>
          </div>
        </div>
      </section>
    </main>
  )
}

export default Login
