import { useState, useEffect, useRef } from 'react'
import { Link } from 'react-router-dom'
import { getProducts, getSliders, getBanners, getSettings } from '../services/api'
import ProductCard from '../components/product/ProductCard'

// ─── Countdown to midnight ────────────────────────────────────
function useCountdown() {
  const [time, setTime] = useState({ hours: '00', mins: '00', secs: '00' })

  useEffect(() => {
    const tick = () => {
      const now = new Date()
      const midnight = new Date()
      midnight.setHours(24, 0, 0, 0)
      const diff = Math.max(0, Math.floor((midnight - now) / 1000))
      const h = Math.floor(diff / 3600)
      const m = Math.floor((diff % 3600) / 60)
      const s = diff % 60
      setTime({
        hours: String(h).padStart(2, '0'),
        mins: String(m).padStart(2, '0'),
        secs: String(s).padStart(2, '0'),
      })
    }
    tick()
    const id = setInterval(tick, 1000)
    return () => clearInterval(id)
  }, [])

  return time
}

// ─── Fallback static data (used when API has no data) ─────────
const DEFAULT_SLIDER = {
  title: 'Shop Now',
  subtitle: 'Quality Products',
  label: 'Welcome to Kalses',
  image: '/assets/imgs/page/homepage1/banner.png',
  button_text: 'Shop now',
  link: '/shop',
}

const DEFAULT_HOME_TOP_BANNERS = [
  {
    title: 'Up to 50% Off',
    subtitle: 'On selected items',
    label: 'Hot Deals',
    image: '/assets/imgs/page/homepage1/banner-small-1.png',
    link: '/shop',
    button_text: 'Shop Now',
    btn_class: 'btn-brand-3',
  },
  {
    title: 'Latest Products',
    subtitle: "Check out what's new",
    label: 'New Arrivals',
    image: '/assets/imgs/page/homepage1/banner-small-2.png',
    link: '/shop?sort=newest',
    button_text: 'Explore',
    btn_class: 'btn-brand-2',
  },
]

const TRUST_BADGES = [
  { icon: '/assets/imgs/template/icons/icon-delivery.svg', title: 'Free Shipping', desc: 'On orders over $50' },
  { icon: '/assets/imgs/template/icons/icon-return.svg',   title: 'Easy Returns',  desc: '30-day return policy' },
  { icon: '/assets/imgs/template/icons/icon-secure.svg',   title: 'Secure Payment', desc: '100% secure checkout' },
  { icon: '/assets/imgs/template/icons/icon-support.svg',  title: '24/7 Support',  desc: 'Dedicated support team' },
]

const TESTIMONIALS = [
  {
    avatar: '/assets/imgs/template/ava_1.png',
    name: 'Sarah Johnson', stars: 5,
    text: '"Amazing quality products and super fast shipping! I\'ve been shopping here for months and never been disappointed. Highly recommend!"',
  },
  {
    avatar: '/assets/imgs/template/ava_2.png',
    name: 'Michael Brown', stars: 5,
    text: '"Great customer service! Had an issue with my order and they resolved it within hours. The team really cares about their customers."',
  },
  {
    avatar: '/assets/imgs/template/ava_3.png',
    name: 'Emily Davis', stars: 5,
    text: '"Best prices I\'ve found online! Plus the website is so easy to use. Found exactly what I needed in minutes. Will definitely shop here again!"',
  },
]

// Gallery defaults — CSS class handles bg image/position/size/color/height
const GALLERY_DEFAULT = [
  { bgClass: 'bg-xbox',       wrapClass: 'banner-2',        title: 'Gaming',      subtitle: 'Collection',          label: 'Hot Deal', link: '/shop' },
  { bgClass: 'bg-controller', wrapClass: 'banner-img-left', title: 'Controllers', subtitle: 'For all platforms',              link: '/shop' },
  { bgClass: 'bg-metaverse',  wrapClass: '',                title: 'VR Headsets', subtitle: 'Experience the future',          link: '/shop' },
  { bgClass: 'bg-electronic', wrapClass: '',                title: 'Accessories', subtitle: 'Must-have gadgets',              link: '/shop' },
  { bgClass: 'bg-phone',      wrapClass: '',                title: 'Smartphones', subtitle: 'Latest models available',        link: '/shop' },
]

const BRANDS = ['apple', 'samsung', 'sony', 'dell', 'hp', 'logitech']

// ─── Sub-components ───────────────────────────────────────────
// ─── Hero Slider (manages own Swiper instance via ref) ────────
const HeroSlider = ({ slides }) => {
  const containerRef = useRef(null)
  const swiperRef    = useRef(null)

  useEffect(() => {
    if (!containerRef.current) return

    // Destroy previous instance
    if (swiperRef.current) {
      try { swiperRef.current.destroy(true, true) } catch {}
      swiperRef.current = null
    }

    // Wait for React to finish painting slides into the DOM
    const t = setTimeout(() => {
      if (!containerRef.current || !window.Swiper) return
      swiperRef.current = new window.Swiper(containerRef.current, {
        slidesPerView: 1,
        loop: slides.length > 1,
        navigation: {
          nextEl: '.swiper-button-next-group-1',
          prevEl: '.swiper-button-prev-group-1',
        },
        pagination: { el: '.swiper-pagination-1', clickable: true },
        autoplay: slides.length > 1
          ? { delay: 5000, disableOnInteraction: false }
          : false,
      })
    }, 150)

    return () => {
      clearTimeout(t)
      if (swiperRef.current) {
        try { swiperRef.current.destroy(true, true) } catch {}
        swiperRef.current = null
      }
    }
  }, [slides]) // re-init whenever slides array reference changes

  return (
    <div className="box-swiper" style={{ position: 'relative' }}>
      <div className="swiper-container swiper-group-1" ref={containerRef}>
        <div className="swiper-wrapper">
          {slides.map((slide, i) => (
            <div className="swiper-slide" key={i}>
              <div
                className="banner-big bg-11"
                style={{ backgroundImage: `url(${slide.image})` }}
              >
                {(slide.label || slide.subtitle) && (
                  <span className="font-sm text-uppercase">
                    {slide.label || slide.subtitle}
                  </span>
                )}
                {slide.subtitle && slide.label && (
                  <h2 className="mt-10">{slide.subtitle}</h2>
                )}
                <h1>{slide.title}</h1>
                {slide.description && (
                  <div className="row">
                    <div className="col-lg-5 col-md-7 col-sm-12">
                      <p className="font-sm color-brand-3">{slide.description}</p>
                    </div>
                  </div>
                )}
                <div className="mt-30">
                  <Link className="btn btn-brand-2" to={slide.link || '/shop'}>
                    {slide.button_text || 'Shop now'}
                  </Link>
                </div>
              </div>
            </div>
          ))}
        </div>
        <div className="swiper-pagination swiper-pagination-1"></div>
      </div>
      <div className="swiper-button-next swiper-button-next-group-1"></div>
      <div className="swiper-button-prev swiper-button-prev-group-1"></div>
    </div>
  )
}

const StarRow = ({ count }) => (
  <div className="rating">
    {Array.from({ length: count }).map((_, i) => (
      <img key={i} src="/assets/imgs/template/icons/star.svg" alt="star" />
    ))}
  </div>
)

const ProductGrid = ({ products, loading }) => {
  if (loading) {
    return (
      <div className="col-12 text-center py-5">
        <div className="spinner-border text-primary" role="status">
          <span className="visually-hidden">Loading...</span>
        </div>
      </div>
    )
  }
  if (!products.length) {
    return (
      <div className="col-12 text-center py-4">
        <p className="color-gray-500">No products found.</p>
      </div>
    )
  }
  return products.map((product) => <ProductCard key={product.id} product={product} />)
}

const ProductSection = ({ title, subtitle, products, loading, sortParam }) => (
  <section className="section-box mt-50">
    <div className="container">
      <div className="head-main">
        <div className="row">
          <div className="col-lg-6 col-md-6">
            <h3>{title}</h3>
            <p className="font-base color-gray-500">{subtitle}</p>
          </div>
          <div className="col-lg-6 col-md-6 text-end">
            <Link className="btn btn-brand-2 btn-arrow-right" to={`/shop${sortParam ? '?sort=' + sortParam : ''}`}>
              View All
            </Link>
          </div>
        </div>
      </div>
      <div className="row mt-30">
        <ProductGrid products={products} loading={loading} />
      </div>
    </div>
  </section>
)

// ─── Main component ───────────────────────────────────────────
export default function Home() {
  const countdown = useCountdown()

  const [featuredProducts, setFeaturedProducts] = useState([])
  const [newArrivals, setNewArrivals] = useState([])
  const [bestSellers, setBestSellers] = useState([])
  const [dealProducts, setDealProducts] = useState([])
  const [loading, setLoading] = useState(true)

  // CMS data from admin
  const [sliders, setSliders] = useState([])
  const [homeTopBanners, setHomeTopBanners] = useState([])
  const [homeMidBanners, setHomeMidBanners] = useState([])
  const [homeBottomBanners, setHomeBottomBanners] = useState([])
  const [apiGallery, setApiGallery]           = useState(null)
  const [apiTrustBadges, setApiTrustBadges]   = useState(null)
  const [apiTestimonials, setApiTestimonials] = useState(null)
  const [apiBrands, setApiBrands]             = useState(null)

  useEffect(() => {
    const loadAll = async () => {
      setLoading(true)

      // Load CMS data (sliders + banners + settings) — fail silently
      Promise.allSettled([getSliders(), getBanners(), getSettings()]).then(([slidersRes, bannersRes, settingsRes]) => {
        if (slidersRes.status === 'fulfilled') {
          const data = slidersRes.value.data
          const list = Array.isArray(data) ? data : (data?.data || [])
          const active = list.filter(s => s.is_active)
          if (active.length) setSliders(active)
        }
        if (bannersRes.status === 'fulfilled') {
          const data = bannersRes.value.data
          const list = Array.isArray(data) ? data : (data?.data || [])
          const active = list.filter(b => b.is_active)
          const top = active.filter(b => b.position === 'home_top')
          const mid = active.filter(b => b.position === 'home_middle')
          const bot = active.filter(b => b.position === 'home_bottom')
          if (top.length) setHomeTopBanners(top)
          if (mid.length) setHomeMidBanners(mid)
          if (bot.length) setHomeBottomBanners(bot)
        }
        if (settingsRes.status === 'fulfilled') {
          const s = settingsRes.value.data
          const parse = (val) => { if (!val) return null; if (Array.isArray(val)) return val; try { return JSON.parse(val) } catch { return null } }
          const gp = parse(s.gallery_panels); if (gp?.length) setApiGallery(gp)
          const tb = parse(s.trust_badges); if (tb?.length) setApiTrustBadges(tb)
          const tm = parse(s.testimonials); if (tm?.length) setApiTestimonials(tm)
          const br = parse(s.brands);       if (br?.length) setApiBrands(br)
        }
      })

      // Load products
      try {
        const [featRes, newRes, bestRes, dealRes] = await Promise.allSettled([
          getProducts({ per_page: 8, sort: 'featured' }),
          getProducts({ per_page: 8, sort: 'newest' }),
          getProducts({ per_page: 8, sort: 'best_selling' }),
          getProducts({ per_page: 4, sort: 'price_asc' }),
        ])

        const extract = (res) => {
          if (res.status !== 'fulfilled') return []
          const d = res.value.data
          if (Array.isArray(d)) return d
          if (Array.isArray(d?.data)) return d.data
          if (Array.isArray(d?.products)) return d.products
          return []
        }

        setFeaturedProducts(extract(featRes))
        setNewArrivals(extract(newRes))
        setBestSellers(extract(bestRes))
        setDealProducts(extract(dealRes))
      } catch {
        // fail silently
      } finally {
        setLoading(false)
      }
    }
    loadAll()
  }, [])

  // Use API sliders or fallback
  const activeSliders = sliders.length ? sliders : [DEFAULT_SLIDER]
  // Use API home_top banners or fallback (max 2)
  const topBanners = homeTopBanners.length ? homeTopBanners : DEFAULT_HOME_TOP_BANNERS
  // Merge gallery data: settings gallery_panels > home_middle banners > defaults
  const gallerySource = apiGallery || (homeMidBanners.length ? homeMidBanners : null)
  const gallerySlots = GALLERY_DEFAULT.map((def, i) => {
    const override = gallerySource?.[i]
    return override ? { ...def, ...override, isCustom: !!(override.image) } : { ...def, isCustom: false }
  })


  return (
    <main className="main">

      {/* ── Hero Banner ──────────────────────────────────────── */}
      <section className="section-box">
        <div className="banner-hero banner-1">
          <div className="container">
            <div className="row">

              {/* Main Swiper Slider */}
              <div className="col-lg-8">
                <HeroSlider slides={activeSliders} />
              </div>

              {/* Small Banners (home_top) */}
              <div className="col-lg-4">
                <div className="row">
                  {topBanners.map((b, i) => (
                    <div key={i} className="col-lg-12 col-md-6 col-sm-12">
                      <div
                        className={`banner-small banner-small-${i + 1} bg-1${i + 3}`}
                        style={{ backgroundImage: `url(${b.image})` }}
                      >
                        {b.label && <span className="color-danger text-uppercase font-sm-lh32">{b.label}</span>}
                        <h4 className="mb-10">{b.title}</h4>
                        <p className="color-brand-3 font-desc">{b.subtitle}</p>
                        <div className="mt-20">
                          <Link className={`btn ${b.btn_class || 'btn-brand-2'} btn-arrow-right`} to={b.link || '/shop'}>
                            {b.button_text || 'Shop Now'}
                          </Link>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* ── Gallery Promo (home_middle) — always 5 panels, API overrides per slot ── */}
      <section className="section-box mt-50">
        <div className="container">
          <div className="row">
            {/* Panel 0 — large left */}
            <div className="col-xl-4 col-lg-12">
              <div
                className={`banner-2${!gallerySlots[0].isCustom ? ' bg-xbox' : ''}`}
                style={gallerySlots[0].isCustom ? { backgroundImage: `url(${gallerySlots[0].image})`, backgroundSize: 'cover', backgroundPosition: 'center' } : {}}
              >
                {gallerySlots[0].label && <span className="color-danger text-uppercase">{gallerySlots[0].label}</span>}
                <h3 className="font-30">{gallerySlots[0].title}</h3>
                {gallerySlots[0].subtitle && <h5 className="font-55 mb-15">{gallerySlots[0].subtitle}</h5>}
                <div className="mt-25">
                  <Link className="btn btn-brand-2 btn-arrow-right" to={gallerySlots[0].link || '/shop'}>Shop Now</Link>
                </div>
              </div>
            </div>

            {/* Panels 1–4 — right grid */}
            <div className="col-xl-8 col-lg-12">
              {/* Row 1: big (slot 1) + small (slot 2) */}
              <div className="image-gallery">
                <div className="image-big">
                  <div
                    className={`banner-img-left${!gallerySlots[1].isCustom ? ' bg-controller' : ''}`}
                    style={gallerySlots[1].isCustom ? { backgroundImage: `url(${gallerySlots[1].image})`, backgroundSize: 'cover', backgroundPosition: 'center', backgroundRepeat: 'no-repeat', padding: '20px' } : {}}
                  >
                    <h3 className="font-33 mb-10">{gallerySlots[1].title}</h3>
                    {gallerySlots[1].subtitle && <p className="font-18">{gallerySlots[1].subtitle}</p>}
                    <div className="mt-25">
                      <Link className="btn btn-info btn-arrow-right" to={gallerySlots[1].link || '/shop'}>View All</Link>
                    </div>
                  </div>
                </div>
                <div className="image-small">
                  <div
                    className="bg-metaverse"
                    style={gallerySlots[2].isCustom ? { backgroundImage: `url(${gallerySlots[2].image})`, backgroundSize: 'cover', backgroundPosition: 'center' } : {}}
                  >
                    <h3 className="mb-10 font-32">{gallerySlots[2].title}</h3>
                    {gallerySlots[2].subtitle && <p className="font-16">{gallerySlots[2].subtitle}</p>}
                    <div className="mt-15">
                      <Link className="btn btn-link-brand-2 btn-arrow-brand-2" to={gallerySlots[2].link || '/shop'}>Explore</Link>
                    </div>
                  </div>
                </div>
              </div>

              {/* Row 2: small (slot 3) + big (slot 4) */}
              <div className="image-gallery">
                <div className="image-small">
                  <div
                    className="bg-electronic"
                    style={gallerySlots[3].isCustom ? { backgroundImage: `url(${gallerySlots[3].image})`, backgroundSize: 'cover', backgroundPosition: 'center' } : {}}
                  >
                    <h3 className="font-32">{gallerySlots[3].title}</h3>
                    {gallerySlots[3].subtitle && <p className="font-16 color-brand-3">{gallerySlots[3].subtitle}</p>}
                  </div>
                </div>
                <div className="image-big">
                  <div
                    className="bg-phone"
                    style={gallerySlots[4].isCustom ? { backgroundImage: `url(${gallerySlots[4].image})`, backgroundSize: 'cover', backgroundPosition: 'center' } : {}}
                  >
                    <h3 className="font-33 mb-15">{gallerySlots[4].title}</h3>
                    {gallerySlots[4].subtitle && <p className="font-18">{gallerySlots[4].subtitle}</p>}
                    <div className="mt-25">
                      <Link className="btn btn-brand-2 btn-arrow-right" to={gallerySlots[4].link || '/shop'}>Shop Now</Link>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* ── Trust Badges ──────────────────────────────────────── */}
      <section className="section-box mt-50">
        <div className="container">
          <div className="row">
            {(apiTrustBadges || TRUST_BADGES).map((badge, i) => (
              <div key={i} className="col-lg-3 col-md-6 col-sm-6 mb-30">
                <div className="card-offer hover-up">
                  <div className="card-image">
                    <img src={badge.icon || `/assets/imgs/template/icons/icon-${['delivery','return','secure','support'][i] || 'delivery'}.svg`} alt={badge.title} style={{ width: 60, height: 60 }} />
                  </div>
                  <div className="card-info">
                    <h5 className="color-brand-3">{badge.title}</h5>
                    <p className="font-sm color-gray-700">{badge.desc}</p>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* ── Deal of the Day ───────────────────────────────────── */}
      <section className="section-box mt-50">
        <div className="container">
          <div className="head-main">
            <div className="row">
              <div className="col-lg-6 col-md-6">
                <h3>Deal of the Day</h3>
                <p className="font-base color-gray-500">Don't miss these amazing deals!</p>
              </div>
              <div className="col-lg-6 col-md-6 text-end">
                <div className="box-countdown">
                  <span className="font-sm color-gray-700 mr-15">Ends in:</span>
                  <div className="deals-countdown">
                    <span className="countdown-section">
                      <span className="countdown-amount font-xl-bold">{countdown.hours}</span>
                      <span className="countdown-period font-xs">Hours</span>
                    </span>
                    <span className="countdown-section">
                      <span className="countdown-amount font-xl-bold">{countdown.mins}</span>
                      <span className="countdown-period font-xs">Mins</span>
                    </span>
                    <span className="countdown-section">
                      <span className="countdown-amount font-xl-bold">{countdown.secs}</span>
                      <span className="countdown-period font-xs">Secs</span>
                    </span>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div className="row mt-30">
            <ProductGrid products={dealProducts} loading={loading} />
          </div>
        </div>
      </section>

      {/* ── Featured Products ─────────────────────────────────── */}
      <ProductSection
        title="Featured Products"
        subtitle="Handpicked products just for you"
        products={featuredProducts}
        loading={loading}
        sortParam="featured"
      />

      {/* ── New Arrivals ──────────────────────────────────────── */}
      <ProductSection
        title="New Arrivals"
        subtitle="Check out our latest products"
        products={newArrivals}
        loading={loading}
        sortParam="newest"
      />

      {/* ── Best Sellers ──────────────────────────────────────── */}
      <ProductSection
        title="Best Sellers"
        subtitle="Our most popular products"
        products={bestSellers}
        loading={loading}
        sortParam="best-selling"
      />

      {/* ── Bottom Banners (home_bottom) ──────────────────────── */}
      {homeBottomBanners.length > 0 && (
        <section className="section-box mt-50">
          <div className="container">
            <div className="row g-3">
              {homeBottomBanners.map((b, i) => (
                <div key={i} className={homeBottomBanners.length === 1 ? 'col-12' : homeBottomBanners.length === 2 ? 'col-md-6' : 'col-md-4'}>
                  <a href={b.link || '/shop'} style={{ display: 'block', borderRadius: 12, overflow: 'hidden', position: 'relative', minHeight: 180 }}>
                    <img src={b.image} alt={b.title || ''} style={{ width: '100%', height: 180, objectFit: 'cover', display: 'block' }} />
                    {(b.title || b.subtitle) && (
                      <div style={{ position: 'absolute', inset: 0, background: 'linear-gradient(to top, rgba(0,0,0,.65) 0%, transparent 55%)', display: 'flex', flexDirection: 'column', justifyContent: 'flex-end', padding: '20px 24px' }}>
                        {b.title && <h4 style={{ color: '#fff', margin: 0, fontWeight: 700 }}>{b.title}</h4>}
                        {b.subtitle && <p style={{ color: 'rgba(255,255,255,.8)', margin: '4px 0 0', fontSize: 14 }}>{b.subtitle}</p>}
                      </div>
                    )}
                  </a>
                </div>
              ))}
            </div>
          </div>
        </section>
      )}

      {/* ── Brand Logos ───────────────────────────────────────── */}
      <section className="section-box mt-50">
        <div className="container">
          <div className="text-center mb-30">
            <h3>Our Trusted Brands</h3>
            <p className="font-base color-gray-500">Shop from top brands you know and love</p>
          </div>
          <div className="row">
            {(apiBrands || BRANDS.map((name, i) => ({
              name: name.charAt(0).toUpperCase() + name.slice(1),
              image: `/assets/imgs/page/homepage1/brand-${i + 1}.png`,
              link: `/shop?brand=${name}`,
            }))).map((brand, i) => (
              <div key={i} className="col-lg-2 col-md-3 col-sm-4 col-6 mb-20 text-center">
                <div className="card-brand hover-up p-20">
                  <Link to={brand.link || '/shop'}>
                    <img
                      src={brand.image || `/assets/imgs/page/homepage1/brand-${i + 1}.png`}
                      alt={brand.name}
                      style={{ maxHeight: 60, opacity: 0.7 }}
                    />
                  </Link>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* ── Testimonials ──────────────────────────────────────── */}
      <section className="section-box mt-50">
        <div className="container">
          <div className="text-center mb-30">
            <h3>What Our Customers Say</h3>
            <p className="font-base color-gray-500">Real reviews from real customers</p>
          </div>
          <div className="row">
            {(apiTestimonials || TESTIMONIALS).map((t, i) => (
              <div key={i} className="col-lg-4 col-md-6 mb-30">
                <div className="card-testimonial hover-up p-30 border-radius-10" style={{ background: '#f8f9fa' }}>
                  <div className="d-flex align-items-center mb-15">
                    <img src={t.avatar || `/assets/imgs/template/ava_${(i % 3) + 1}.png`} alt="Customer" className="rounded-circle mr-15" style={{ width: 50, height: 50, objectFit: 'cover' }} />
                    <div>
                      <h6 className="mb-0">{t.name}</h6>
                      <StarRow count={t.stars} />
                    </div>
                  </div>
                  <p className="font-sm color-gray-700">{t.text}</p>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

    </main>
  )
}
