import { useState } from 'react'
import { Link } from 'react-router-dom'
import { forgotPassword } from '../services/api'
import toast from 'react-hot-toast'

const ForgotPassword = () => {
  const [email, setEmail] = useState('')
  const [loading, setLoading] = useState(false)
  const [sent, setSent] = useState(false)
  const [devUrl, setDevUrl] = useState(null)

  const handleSubmit = async (e) => {
    e.preventDefault()
    setLoading(true)
    try {
      const res = await forgotPassword({ email })
      setSent(true)
      toast.success('Reset link sent!')
      // Dev mode: backend returns the reset URL for easy testing
      if (res.data?.dev_reset_url) {
        setDevUrl(res.data.dev_reset_url)
      }
    } catch (err) {
      toast.error(err?.response?.data?.message || 'Failed to send reset email')
    } finally {
      setLoading(false)
    }
  }

  return (
    <main className="main">
      <div className="section-box">
        <div className="breadcrumbs-div">
          <div className="container">
            <ul className="breadcrumb">
              <li><Link className="font-xs color-gray-1000" to="/">Home</Link></li>
              <li><a className="font-xs color-gray-500">Forgot Password</a></li>
            </ul>
          </div>
        </div>
      </div>

      <section className="section-box shop-template">
        <div className="container">
          <div className="row justify-content-center">
            <div className="col-lg-5 col-md-7 col-sm-10">
              <div className="box-border p-40">
                <h4 className="font-xl-bold color-brand-3 mb-5 text-center">Forgot Password</h4>
                <p className="font-sm color-gray-500 text-center mb-30">
                  Enter your email address and we'll send you a link to reset your password.
                </p>

                {sent ? (
                  <div>
                    <div className="text-center mb-20">
                      <div style={{ fontSize: 48, marginBottom: 10 }}>&#x2709;&#xFE0F;</div>
                      <h5 className="color-brand-3 mb-10">Check your email</h5>
                      <p className="font-sm color-gray-500">
                        If an account exists for <strong>{email}</strong>, a reset link has been sent.
                      </p>
                    </div>
                    {devUrl && (
                      <div className="mt-15 p-15 border rounded" style={{ background: '#fff3cd' }}>
                        <p className="font-xs color-gray-700 mb-5"><strong>Dev mode – reset link:</strong></p>
                        <a href={devUrl} className="font-xs" style={{ wordBreak: 'break-all' }}>{devUrl}</a>
                      </div>
                    )}
                    <div className="text-center mt-20">
                      <Link className="btn btn-buy w-100" to="/login">Back to Login</Link>
                    </div>
                  </div>
                ) : (
                  <form onSubmit={handleSubmit}>
                    <div className="form-group">
                      <input
                        className="form-control font-sm"
                        type="email"
                        placeholder="Email Address*"
                        value={email}
                        onChange={e => setEmail(e.target.value)}
                        required
                      />
                    </div>
                    <div className="form-group">
                      <button
                        type="submit"
                        className="btn btn-buy w-100"
                        disabled={loading}
                      >
                        {loading ? 'Sending...' : 'Send Reset Link'}
                      </button>
                    </div>
                    <div className="text-center">
                      <Link className="font-sm color-brand-1" to="/login">Back to Login</Link>
                    </div>
                  </form>
                )}
              </div>
            </div>
          </div>
        </div>
      </section>
    </main>
  )
}

export default ForgotPassword
