import { useState } from 'react'
import { Link } from 'react-router-dom'
import { sendContact } from '../services/api'

const Contact = () => {
  const [form, setForm] = useState({
    firstName: '',
    lastName: '',
    email: '',
    phone: '',
    subject: '',
    message: '',
  })
  const [sending, setSending] = useState(false)
  const [submitted, setSubmitted] = useState(false)
  const [error, setError] = useState('')

  const handleChange = (e) => {
    setForm((f) => ({ ...f, [e.target.name]: e.target.value }))
    setError('')
  }

  const handleSubmit = async (e) => {
    e.preventDefault()
    if (!form.firstName || !form.email || !form.message) {
      setError('Please fill in all required fields.')
      return
    }
    setSending(true)
    setError('')
    try {
      await sendContact({
        name: `${form.firstName} ${form.lastName}`.trim(),
        email: form.email,
        subject: form.subject || 'General Inquiry',
        message: form.message,
      })
      setSubmitted(true)
      setForm({ firstName: '', lastName: '', email: '', phone: '', subject: '', message: '' })
    } catch {
      setError('Failed to send message. Please try again or email us directly at Info@kalses.com')
    } finally {
      setSending(false)
    }
  }

  return (
    <main className="main">
      {/* Breadcrumb */}
      <div className="section-box">
        <div className="breadcrumbs-div mb-0">
          <div className="container">
            <ul className="breadcrumb">
              <li><Link className="font-xs color-gray-1000" to="/">Home</Link></li>
              <li><span className="font-xs color-gray-500">Page</span></li>
              <li><span className="font-xs color-gray-500">Contact</span></li>
            </ul>
          </div>
        </div>
      </div>

      <section className="section-box shop-template mt-0">
        <div className="container">

          {/* Contact Form + Map */}
          <div className="box-contact">
            <div className="row">
              {/* Contact Form */}
              <div className="col-lg-6">
                <div className="contact-form">
                  <h3 className="color-brand-3 mt-60">Contact Us</h3>
                  <p className="font-sm color-gray-700 mb-30">Our team would love to hear from you!</p>

                  {submitted && (
                    <div className="alert" style={{ background: '#f0fdf4', border: '1px solid #3BB77E', borderRadius: 8, padding: '16px 20px', marginBottom: 20, color: '#166534' }}>
                      <strong>Message sent!</strong> Thank you for reaching out. We'll get back to you within 24 hours.
                    </div>
                  )}

                  {error && (
                    <div className="alert" style={{ background: '#fef2f2', border: '1px solid #fca5a5', borderRadius: 8, padding: '16px 20px', marginBottom: 20, color: '#991b1b' }}>
                      {error}
                    </div>
                  )}

                  <form onSubmit={handleSubmit} noValidate>
                    <div className="row">
                      <div className="col-lg-6 col-md-6">
                        <div className="form-group">
                          <input
                            className="form-control"
                            type="text"
                            name="firstName"
                            placeholder="First name *"
                            value={form.firstName}
                            onChange={handleChange}
                            required
                          />
                        </div>
                      </div>
                      <div className="col-lg-6 col-md-6">
                        <div className="form-group">
                          <input
                            className="form-control"
                            type="text"
                            name="lastName"
                            placeholder="Last name"
                            value={form.lastName}
                            onChange={handleChange}
                          />
                        </div>
                      </div>
                      <div className="col-lg-12">
                        <div className="form-group">
                          <input
                            className="form-control"
                            type="email"
                            name="email"
                            placeholder="Email *"
                            value={form.email}
                            onChange={handleChange}
                            required
                          />
                        </div>
                      </div>
                      <div className="col-lg-12">
                        <div className="form-group">
                          <input
                            className="form-control"
                            type="tel"
                            name="phone"
                            placeholder="Phone number"
                            value={form.phone}
                            onChange={handleChange}
                          />
                        </div>
                      </div>
                      <div className="col-lg-12">
                        <div className="form-group">
                          <select
                            className="form-control"
                            name="subject"
                            value={form.subject}
                            onChange={handleChange}
                          >
                            <option value="">Select a subject</option>
                            <option value="General Inquiry">General Inquiry</option>
                            <option value="Order Support">Order Support</option>
                            <option value="Return / Refund">Return / Refund</option>
                            <option value="Wholesale Inquiry">Wholesale Inquiry</option>
                            <option value="Become a Vendor">Become a Vendor</option>
                            <option value="Product Question">Product Question</option>
                            <option value="Other">Other</option>
                          </select>
                        </div>
                      </div>
                      <div className="col-lg-12">
                        <div className="form-group">
                          <textarea
                            className="form-control"
                            name="message"
                            placeholder="Message *"
                            rows="5"
                            value={form.message}
                            onChange={handleChange}
                            required
                          ></textarea>
                        </div>
                      </div>
                      <div className="col-lg-12">
                        <div className="form-group">
                          <button
                            type="submit"
                            className="btn btn-buy w-auto"
                            disabled={sending}
                            style={{ cursor: sending ? 'not-allowed' : 'pointer', opacity: sending ? 0.7 : 1 }}
                          >
                            {sending ? 'Sending...' : 'Send message'}
                          </button>
                        </div>
                      </div>
                    </div>
                  </form>
                </div>
              </div>

              {/* Map */}
              <div className="col-lg-6">
                <div className="map">
                  <iframe
                    src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d99370.95830820498!2d-94.6274636!3d39.0558235!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x87c0f75eafe99997%3A0x558525e66aaa51a2!2sKansas%20City%2C%20MO%2C%20USA!5e0!3m2!1sen!2s!4v1704373110059!5m2!1sen!2s"
                    height="550"
                    style={{ border: 0, width: '100%' }}
                    allowFullScreen=""
                    loading="lazy"
                    referrerPolicy="no-referrer-when-downgrade"
                    title="Kalses Office Location - Kansas City, Missouri"
                  ></iframe>
                </div>
              </div>
            </div>
          </div>

          {/* Contact Address Info */}
          <div className="box-contact-address pt-80 pb-50">
            <div className="row">
              <div className="col-lg-4 mb-30">
                <h3 className="mb-5">Visit Our Office</h3>
                <p className="font-sm color-gray-700 mb-30">We'd love to meet you</p>
                <a
                  className="btn btn-buy w-auto"
                  href="https://maps.google.com/?q=8301+State+Line+Rd+Kansas+City+MO"
                  target="_blank"
                  rel="noopener noreferrer"
                >
                  View map
                </a>
              </div>
              <div className="col-lg-4">
                <div className="mb-30">
                  <h4>Kansas City Office</h4>
                  <p className="font-sm color-gray-700">
                    8301 State Line Rd. Ste 220 #2547<br />
                    Kansas City, Missouri 64114, USA
                  </p>
                </div>
                <div className="mb-30">
                  <h4>Business Hours</h4>
                  <p className="font-sm color-gray-700">
                    Monday - Friday: 24/7<br />
                    Saturday: 9am - 5pm
                  </p>
                </div>
              </div>
              <div className="col-lg-4">
                <div className="mb-30">
                  <h4>Contact Information</h4>
                  <p className="font-sm color-gray-700">
                    <strong>Phone:</strong> (913) 667-0804<br />
                    <strong>Email:</strong> Info@kalses.com
                  </p>
                </div>
                <div className="mb-30">
                  <h4>Customer Service</h4>
                  <p className="font-sm color-gray-700">
                    Our team is available to assist you with any questions or concerns about your orders,
                    products, returns, or wholesale inquiries.
                  </p>
                </div>
                {/* Social Links */}
                <div className="mb-30">
                  <h4 className="mb-15">Follow Us</h4>
                  <div>
                    <a className="icon-socials icon-facebook" href="#" aria-label="Facebook"></a>
                    <a className="icon-socials icon-instagram" href="#" aria-label="Instagram"></a>
                    <a className="icon-socials icon-twitter" href="#" aria-label="Twitter"></a>
                    <a className="icon-socials icon-linkedin" href="#" aria-label="LinkedIn"></a>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Support Contact Strip */}
        <div className="box-contact-support pt-80 pb-50 background-gray-50">
          <div className="container">
            <div className="row">
              <div className="col-lg-3 mb-30 text-center text-lg-start">
                <h3 className="mb-5">We'd love to hear from you</h3>
                <p className="font-sm color-gray-700">Chat with our friendly team</p>
              </div>
              <div className="col-lg-3 text-center mb-30">
                <div className="box-image mb-20">
                  <img src="/assets/imgs/page/contact/chat.svg" alt="Chat to sales" onError={e => { e.target.style.display = 'none' }} />
                </div>
                <h4 className="mb-5">Chat to sales</h4>
                <p className="font-sm color-gray-700 mb-5">Speak to our team.</p>
                <a className="font-sm color-gray-900" href="mailto:Info@kalses.com">Info@kalses.com</a>
              </div>
              <div className="col-lg-3 text-center mb-30">
                <div className="box-image mb-20">
                  <img src="/assets/imgs/page/contact/call.svg" alt="Call us" onError={e => { e.target.style.display = 'none' }} />
                </div>
                <h4 className="mb-5">Call us</h4>
                <p className="font-sm color-gray-700 mb-5">Mon-Fri 24/7, Sat 9am-5pm</p>
                <a className="font-sm color-gray-900" href="tel:+19136670804">(913) 667-0804</a>
              </div>
              <div className="col-lg-3 text-center mb-30">
                <div className="box-image mb-20">
                  <img src="/assets/imgs/page/contact/map.svg" alt="Visit us" onError={e => { e.target.style.display = 'none' }} />
                </div>
                <h4 className="mb-5">Visit us</h4>
                <p className="font-sm color-gray-700 mb-5">Visit our office</p>
                <span className="font-sm color-gray-900">
                  8301 State Line Rd. Ste 220 #2547<br />
                  Kansas City, Missouri 64114, USA
                </span>
              </div>
            </div>
          </div>
        </div>
      </section>
    </main>
  )
}

export default Contact
