import { useState, useEffect } from 'react'
import { Link } from 'react-router-dom'
import { getCompare, removeFromCompare, clearCompare } from '../services/api'
import { useCart } from '../context/CartContext'
import Loading from '../components/ui/Loading'
import toast from 'react-hot-toast'

const Compare = () => {
  const [products, setProducts] = useState([])
  const [loading, setLoading] = useState(true)
  const { addToCart } = useCart()

  const loadCompare = () => {
    setLoading(true)
    getCompare()
      .then(r => {
        // Compare API returns array directly
        const data = Array.isArray(r.data) ? r.data : (r.data?.data || [])
        setProducts(data)
      })
      .catch(() => setProducts([]))
      .finally(() => setLoading(false))
  }

  useEffect(() => {
    loadCompare()
  }, [])

  const handleRemove = async (productId) => {
    try {
      await removeFromCompare(productId)
      setProducts(prev => prev.filter(p => p.id !== productId))
      toast.success('Removed from compare')
    } catch {
      toast.error('Failed to remove from compare')
    }
  }

  const handleClearAll = async () => {
    try {
      await clearCompare()
      setProducts([])
      toast.success('Compare list cleared')
    } catch {
      toast.error('Failed to clear compare list')
    }
  }

  const handleAddToCart = async (productId) => {
    try {
      await addToCart(productId, 1)
      toast.success('Added to cart!')
    } catch {
      toast.error('Failed to add to cart')
    }
  }

  const renderStars = (rating) => {
    const rounded = Math.round(rating || 0)
    return [1, 2, 3, 4, 5].map(star => (
      <img
        key={star}
        src={
          star <= rounded
            ? '/assets/imgs/template/icons/star.svg'
            : '/assets/imgs/template/icons/star-gray.svg'
        }
        alt="star"
      />
    ))
  }

  if (loading) return <Loading height={300} />

  return (
    <main className="main">
      {/* Breadcrumb */}
      <div className="section-box">
        <div className="breadcrumbs-div">
          <div className="container">
            <ul className="breadcrumb">
              <li>
                <Link className="font-xs color-gray-1000" to="/">Home</Link>
              </li>
              <li>
                <Link className="font-xs color-gray-500" to="/shop">Shop</Link>
              </li>
              <li>
                <span className="font-xs color-gray-500">Compare</span>
              </li>
            </ul>
          </div>
        </div>
      </div>

      {/* Compare Section */}
      <section className="section-box shop-template">
        <div className="container">
          {products.length === 0 ? (
            <div className="row mb-80">
              <div className="col-lg-1"></div>
              <div className="col-lg-10">
                <div className="box-compare-products text-center" style={{ padding: '60px 0' }}>
                  <p className="font-md color-gray-500 mb-20">
                    No products to compare. Add products to compare their features.
                  </p>
                  <Link to="/shop" className="btn btn-buy w-auto">Browse Products</Link>
                </div>
              </div>
            </div>
          ) : (
            <div className="row mb-80">
              <div className="col-lg-1"></div>
              <div className="col-lg-10">
                <div className="table-responsive">
                  <table className="table-compare">
                    <thead>
                      <tr>
                        <th>Compare</th>
                        {products.map(p => (
                          <th key={p.id}>
                            <div style={{ position: 'relative', textAlign: 'center' }}>
                              <button
                                className="btn btn-delete"
                                onClick={() => handleRemove(p.id)}
                                aria-label={`Remove ${p.name} from compare`}
                                style={{ position: 'absolute', top: 0, right: 0 }}
                              ></button>
                              <Link to={`/product/${p.slug}`}>
                                <img
                                  src={p.primary_image?.url || p.image_url || '/assets/imgs/no-image.png'}
                                  alt={p.name}
                                  style={{ width: 100, height: 100, objectFit: 'cover', borderRadius: 8, display: 'block', margin: '0 auto 10px' }}
                                />
                                <span className="font-sm-bold color-brand-3">{p.name}</span>
                              </Link>
                            </div>
                          </th>
                        ))}
                      </tr>
                    </thead>

                    <tbody>
                      {/* Price Row */}
                      <tr>
                        <td className="font-md-bold color-brand-3">Price</td>
                        {products.map(p => (
                          <td key={p.id} className="text-center">
                            {p.sale_price ? (
                              <>
                                <span className="font-md-bold color-brand-3">
                                  ${parseFloat(p.sale_price).toFixed(2)}
                                </span>
                                <span
                                  className="font-xs color-gray-500 ml-5"
                                  style={{ textDecoration: 'line-through' }}
                                >
                                  ${parseFloat(p.price).toFixed(2)}
                                </span>
                              </>
                            ) : (
                              <span className="font-md-bold color-brand-3">
                                ${parseFloat(p.price || 0).toFixed(2)}
                              </span>
                            )}
                          </td>
                        ))}
                      </tr>

                      {/* Category Row */}
                      <tr>
                        <td className="font-md-bold color-brand-3">Category</td>
                        {products.map(p => (
                          <td key={p.id} className="text-center font-md color-gray-500">
                            {p.category?.name || '-'}
                          </td>
                        ))}
                      </tr>

                      {/* Vendor Row */}
                      <tr>
                        <td className="font-md-bold color-brand-3">Vendor</td>
                        {products.map(p => (
                          <td key={p.id} className="text-center font-md color-gray-500">
                            {p.vendor?.store_name || p.vendor?.name || '-'}
                          </td>
                        ))}
                      </tr>

                      {/* Stock Row */}
                      <tr>
                        <td className="font-md-bold color-brand-3">Stock</td>
                        {products.map(p => (
                          <td key={p.id} className="text-center">
                            <span className={`btn btn-gray font-sm-bold ${p.stock > 0 ? 'color-brand-3' : 'color-gray-500'}`}>
                              {p.stock > 0 ? 'In Stock' : 'Out of Stock'}
                            </span>
                          </td>
                        ))}
                      </tr>

                      {/* Rating Row */}
                      <tr>
                        <td className="font-md-bold color-brand-3">Rating</td>
                        {products.map(p => (
                          <td key={p.id} className="text-center">
                            <div className="rating" style={{ justifyContent: 'center', display: 'flex', gap: 2 }}>
                              {renderStars(p.rating)}
                            </div>
                            <span className="font-xs color-gray-500">
                              {parseFloat(p.rating || 0).toFixed(1)}/5
                              {p.reviews_count != null && ` (${p.reviews_count})`}
                            </span>
                          </td>
                        ))}
                      </tr>

                      {/* Action Row */}
                      <tr>
                        <td className="font-md-bold color-brand-3">Action</td>
                        {products.map(p => (
                          <td key={p.id} className="text-center">
                            <button
                              className="btn btn-cart font-sm-bold"
                              onClick={() => handleAddToCart(p.id)}
                              disabled={p.stock === 0}
                            >
                              Add to Cart
                            </button>
                          </td>
                        ))}
                      </tr>

                      {/* Remove Row */}
                      <tr>
                        <td className="font-md-bold color-brand-3">Remove</td>
                        {products.map(p => (
                          <td key={p.id} className="text-center">
                            <button
                              className="btn btn-delete"
                              onClick={() => handleRemove(p.id)}
                              aria-label={`Remove ${p.name}`}
                            ></button>
                          </td>
                        ))}
                      </tr>
                    </tbody>
                  </table>
                </div>

                {/* Clear All Button */}
                <div className="text-center mt-30">
                  <button className="btn btn-buy w-auto" onClick={handleClearAll}>
                    Clear All
                  </button>
                  <Link to="/shop" className="btn btn-gray font-sm-bold ml-15">
                    Continue Shopping
                  </Link>
                </div>
              </div>
            </div>
          )}
        </div>
      </section>
    </main>
  )
}

export default Compare
