import { useState, useEffect } from 'react'
import { Link, useSearchParams, useNavigate } from 'react-router-dom'
import { verifyStripeSession } from '../services/api'
import Loading from '../components/ui/Loading'
import toast from 'react-hot-toast'

const CheckoutSuccess = () => {
  const [searchParams] = useSearchParams()
  const navigate = useNavigate()
  const [order, setOrder] = useState(null)
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState('')
  const sessionId = searchParams.get('session_id')

  useEffect(() => {
    if (!sessionId) {
      navigate('/cart')
      return
    }
    verifyStripeSession(sessionId)
      .then(res => {
        setOrder(res.data)
        toast.success('Payment successful! Order confirmed.')
      })
      .catch(err => {
        setError(err?.response?.data?.message || 'Could not verify payment. Please contact support.')
      })
      .finally(() => setLoading(false))
  }, [sessionId])

  if (loading) return <Loading height={400} />

  if (error) {
    return (
      <main className="main">
        <section className="section-box shop-template">
          <div className="container text-center py-5">
            <div style={{ fontSize: 64 }}>&#x274C;</div>
            <h3 className="color-brand-3 mt-20">Payment Verification Failed</h3>
            <p className="color-gray-500">{error}</p>
            <Link to="/cart" className="btn btn-buy mt-20">Return to Cart</Link>
          </div>
        </section>
      </main>
    )
  }

  return (
    <main className="main">
      <section className="section-box shop-template">
        <div className="container">
          <div className="row justify-content-center">
            <div className="col-lg-7">
              <div className="box-border p-40 text-center">
                <div style={{ fontSize: 64 }}>&#x1F389;</div>
                <h3 className="color-brand-3 mt-15 mb-10">Order Confirmed!</h3>
                <p className="font-md color-gray-500 mb-20">
                  Thank you for your purchase. Your payment was successful.
                </p>

                {order && (
                  <div className="text-start mt-20">
                    <div className="d-flex justify-content-between mb-10">
                      <span className="font-sm-bold color-brand-3">Order Number</span>
                      <span className="font-sm color-gray-500">{order.order_number}</span>
                    </div>
                    <div className="d-flex justify-content-between mb-10">
                      <span className="font-sm-bold color-brand-3">Total Paid</span>
                      <span className="font-sm color-gray-500">${parseFloat(order.total).toFixed(2)}</span>
                    </div>
                    <div className="d-flex justify-content-between mb-10">
                      <span className="font-sm-bold color-brand-3">Payment Method</span>
                      <span className="font-sm color-gray-500">Credit Card (Stripe)</span>
                    </div>
                    <div className="d-flex justify-content-between mb-20">
                      <span className="font-sm-bold color-brand-3">Status</span>
                      <span className="font-sm" style={{ color: '#3BB77E', fontWeight: 600 }}>Processing</span>
                    </div>

                    {order.items && order.items.length > 0 && (
                      <>
                        <hr />
                        <h6 className="color-brand-3 mb-10">Items Ordered</h6>
                        {order.items.map(item => (
                          <div key={item.id} className="d-flex justify-content-between mb-5">
                            <span className="font-xs color-gray-700">{item.product_name} &times; {item.quantity}</span>
                            <span className="font-xs color-gray-500">${parseFloat(item.subtotal).toFixed(2)}</span>
                          </div>
                        ))}
                      </>
                    )}
                  </div>
                )}

                <div className="row mt-30 g-2">
                  <div className="col-6">
                    <Link to="/account?tab=orders" className="btn btn-cart w-100">View Orders</Link>
                  </div>
                  <div className="col-6">
                    <Link to="/shop" className="btn btn-buy w-100">Continue Shopping</Link>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>
    </main>
  )
}

export default CheckoutSuccess
