import { useState, useEffect } from 'react'
import { Link, useNavigate } from 'react-router-dom'
import { useCart } from '../context/CartContext'
import { useAuth } from '../context/AuthContext'
import { placeOrder, createStripeSession } from '../services/api'
import Loading from '../components/ui/Loading'
import toast from 'react-hot-toast'

const Checkout = () => {
  const { cart, cartLoading, clearCart } = useCart()
  const { user } = useAuth()
  const navigate = useNavigate()
  const [submitting, setSubmitting] = useState(false)
  const [paymentMethod, setPaymentMethod] = useState('stripe')
  const [error, setError] = useState('')

  const [form, setForm] = useState({
    email: '',
    first_name: '',
    last_name: '',
    address: '',
    address2: '',
    city: '',
    zip: '',
    phone: '',
    notes: '',
    newsletter: false,
  })

  // Prefill from user context if logged in
  useEffect(() => {
    if (user) {
      setForm(f => ({
        ...f,
        email: user.email || '',
        first_name: user.name?.split(' ')[0] || '',
        last_name: user.name?.split(' ').slice(1).join(' ') || '',
        phone: user.phone || '',
      }))
    }
  }, [user])

  const handleChange = e => {
    const { name, value, type, checked } = e.target
    setForm(f => ({ ...f, [name]: type === 'checkbox' ? checked : value }))
  }

  const subtotal = cart.reduce((sum, item) => {
    const price = parseFloat(item.product?.sale_price || item.product?.price || item.price || 0)
    return sum + price * item.quantity
  }, 0)

  const total = subtotal // Shipping is free

  const handlePlaceOrder = async () => {
    setError('')

    // Basic validation
    if (!form.email) { setError('Email is required'); return }
    if (!form.first_name) { setError('First name is required'); return }
    if (!form.last_name) { setError('Last name is required'); return }
    if (!form.address) { setError('Address is required'); return }
    if (!form.city) { setError('City is required'); return }
    if (cart.length === 0) { toast.error('Your cart is empty'); return }

    setSubmitting(true)
    try {
      const orderData = {
        shipping_name:    `${form.first_name} ${form.last_name}`.trim(),
        shipping_email:   form.email,
        shipping_phone:   form.phone || null,
        shipping_address: form.address,
        shipping_city:    form.city,
        shipping_state:   form.state || null,
        shipping_zip:     form.zip || '00000',
        shipping_country: form.country || 'US',
        payment_method:   paymentMethod,
        notes:            form.notes || null,
      }

      if (paymentMethod === 'stripe') {
        // Redirect to Stripe hosted checkout page
        const res = await createStripeSession(orderData)
        window.location.href = res.data.url
        return
      }

      const res = await placeOrder(orderData)
      const order = res?.data?.data || res?.data
      await clearCart()
      toast.success('Order placed successfully!')
      navigate(`/account?tab=orders${order?.id ? `&order=${order.id}` : ''}`)
    } catch (err) {
      const msg = err?.response?.data?.message || 'Failed to place order. Please try again.'
      setError(msg)
      toast.error(msg)
    } finally {
      setSubmitting(false)
    }
  }

  if (cartLoading) return <Loading height={300} />

  return (
    <main className="main">
      {/* Breadcrumb */}
      <div className="section-box">
        <div className="breadcrumbs-div">
          <div className="container">
            <ul className="breadcrumb">
              <li><Link className="font-xs color-gray-1000" to="/">Home</Link></li>
              <li><Link className="font-xs color-gray-500" to="/shop">Shop</Link></li>
              <li><a className="font-xs color-gray-500">Checkout</a></li>
            </ul>
          </div>
        </div>
      </div>

      <section className="section-box shop-template">
        <div className="container">
          <div className="row">
            {/* Left: Contact + Shipping Form */}
            <div className="col-lg-6">
              <div className="box-border">
                {/* Payment Shortcuts */}
                <div className="box-payment">
                  <a className="btn btn-gpay">
                    <img
                      src="/assets/imgs/page/checkout/gpay.svg"
                      alt="Google Pay"
                      onError={e => e.target.style.display = 'none'}
                    />
                  </a>
                  <a className="btn btn-paypal">
                    <img
                      src="/assets/imgs/page/checkout/paypal.svg"
                      alt="PayPal"
                      onError={e => e.target.style.display = 'none'}
                    />
                  </a>
                  <a className="btn btn-amazon">
                    <img
                      src="/assets/imgs/page/checkout/amazon.svg"
                      alt="Amazon Pay"
                      onError={e => e.target.style.display = 'none'}
                    />
                  </a>
                </div>

                {/* Or Divider */}
                <div className="border-bottom-4 text-center mb-20">
                  <div className="text-or font-md color-gray-500">Or</div>
                </div>

                <div className="row">
                  {/* Contact Info Header */}
                  <div className="col-lg-6 col-sm-6 mb-20">
                    <h5 className="font-md-bold color-brand-3 text-sm-start text-center">
                      Contact information
                    </h5>
                  </div>
                  <div className="col-lg-6 col-sm-6 mb-20 text-sm-end text-center">
                    <span className="font-sm color-brand-3">Already have an account?</span>
                    <Link className="font-sm color-brand-1" to="/login"> Login</Link>
                  </div>

                  {/* Email */}
                  <div className="col-lg-12">
                    <div className="form-group">
                      <input
                        className="form-control font-sm"
                        type="email"
                        name="email"
                        placeholder="Email*"
                        value={form.email}
                        onChange={handleChange}
                        required
                      />
                    </div>
                  </div>

                  {/* Newsletter checkbox */}
                  <div className="col-lg-12">
                    <div className="form-group">
                      <label className="font-sm color-brand-3" htmlFor="checkboxOffers">
                        <input
                          className="checkboxOffer"
                          id="checkboxOffers"
                          type="checkbox"
                          name="newsletter"
                          checked={form.newsletter}
                          onChange={handleChange}
                        />
                        {' '}Keep me up to date on news and exclusive offers
                      </label>
                    </div>
                  </div>

                  {/* Shipping Address Header */}
                  <div className="col-lg-12">
                    <h5 className="font-md-bold color-brand-3 mt-15 mb-20">Shipping address</h5>
                  </div>

                  {/* First Name */}
                  <div className="col-lg-6">
                    <div className="form-group">
                      <input
                        className="form-control font-sm"
                        type="text"
                        name="first_name"
                        placeholder="First name*"
                        value={form.first_name}
                        onChange={handleChange}
                        required
                      />
                    </div>
                  </div>

                  {/* Last Name */}
                  <div className="col-lg-6">
                    <div className="form-group">
                      <input
                        className="form-control font-sm"
                        type="text"
                        name="last_name"
                        placeholder="Last name*"
                        value={form.last_name}
                        onChange={handleChange}
                        required
                      />
                    </div>
                  </div>

                  {/* Address 1 */}
                  <div className="col-lg-12">
                    <div className="form-group">
                      <input
                        className="form-control font-sm"
                        type="text"
                        name="address"
                        placeholder="Address 1*"
                        value={form.address}
                        onChange={handleChange}
                        required
                      />
                    </div>
                  </div>

                  {/* Address 2 */}
                  <div className="col-lg-12">
                    <div className="form-group">
                      <input
                        className="form-control font-sm"
                        type="text"
                        name="address2"
                        placeholder="Address 2 (optional)"
                        value={form.address2}
                        onChange={handleChange}
                      />
                    </div>
                  </div>

                  {/* City */}
                  <div className="col-lg-6">
                    <div className="form-group">
                      <input
                        className="form-control font-sm"
                        type="text"
                        name="city"
                        placeholder="City*"
                        value={form.city}
                        onChange={handleChange}
                        required
                      />
                    </div>
                  </div>

                  {/* ZIP */}
                  <div className="col-lg-6">
                    <div className="form-group">
                      <input
                        className="form-control font-sm"
                        type="text"
                        name="zip"
                        placeholder="PostCode / ZIP*"
                        value={form.zip}
                        onChange={handleChange}
                      />
                    </div>
                  </div>

                  {/* Phone */}
                  <div className="col-lg-6">
                    <div className="form-group">
                      <input
                        className="form-control font-sm"
                        type="text"
                        name="phone"
                        placeholder="Phone*"
                        value={form.phone}
                        onChange={handleChange}
                      />
                    </div>
                  </div>

                  {/* Notes */}
                  <div className="col-lg-12">
                    <div className="form-group mb-0">
                      <textarea
                        className="form-control font-sm"
                        name="notes"
                        placeholder="Additional Information"
                        rows="5"
                        value={form.notes}
                        onChange={handleChange}
                      />
                    </div>
                  </div>
                </div>

                {/* Payment Method */}
                <div className="mt-20">
                  <h5 className="font-md-bold color-brand-3 mb-15">Payment Method</h5>
                  <div className="payment-methods">
                    {/* Stripe / Credit Card */}
                    <div
                      className="form-check mb-15 p-3 border rounded"
                      style={{ background: paymentMethod === 'stripe' ? '#f0fdf4' : '#f8f9fa' }}
                    >
                      <input
                        className="form-check-input"
                        type="radio"
                        name="payment_method"
                        id="payment_stripe"
                        value="stripe"
                        checked={paymentMethod === 'stripe'}
                        onChange={() => setPaymentMethod('stripe')}
                      />
                      <label className="form-check-label w-100" htmlFor="payment_stripe">
                        <div className="d-flex align-items-center justify-content-between">
                          <div>
                            <strong className="color-brand-3">Credit/Debit Card</strong>
                            <p className="font-xs color-gray-500 mb-0">Pay securely with Stripe</p>
                          </div>
                          <div>
                            <img
                              src="/assets/imgs/page/checkout/visa.svg"
                              alt="Visa"
                              style={{ height: 24 }}
                              onError={e => e.target.style.display = 'none'}
                            />
                            <img
                              src="/assets/imgs/page/checkout/mastercard.svg"
                              alt="Mastercard"
                              style={{ height: 24, marginLeft: 4 }}
                              onError={e => e.target.style.display = 'none'}
                            />
                          </div>
                        </div>
                      </label>
                    </div>

                    {/* Cash on Delivery */}
                    <div
                      className="form-check mb-15 p-3 border rounded"
                      style={{ background: paymentMethod === 'cod' ? '#f0fdf4' : '#f8f9fa' }}
                    >
                      <input
                        className="form-check-input"
                        type="radio"
                        name="payment_method"
                        id="payment_cod"
                        value="cod"
                        checked={paymentMethod === 'cod'}
                        onChange={() => setPaymentMethod('cod')}
                      />
                      <label className="form-check-label w-100" htmlFor="payment_cod">
                        <div>
                          <strong className="color-brand-3">Cash on Delivery</strong>
                          <p className="font-xs color-gray-500 mb-0">Pay when you receive your order</p>
                        </div>
                      </label>
                    </div>
                  </div>
                </div>

                {/* Error message */}
                {error && (
                  <div className="form-group mt-10">
                    <p className="font-sm" style={{ color: '#e30000' }}>{error}</p>
                  </div>
                )}
              </div>

              {/* Return / Place Order Buttons */}
              <div className="row mt-20">
                <div className="col-lg-6 col-5 mb-20">
                  <Link className="btn font-sm-bold color-brand-1 arrow-back-1" to="/cart">
                    Return to Cart
                  </Link>
                </div>
                <div className="col-lg-6 col-7 mb-20 text-end">
                  <button
                    className="btn btn-buy w-auto arrow-next"
                    onClick={handlePlaceOrder}
                    disabled={submitting || cart.length === 0}
                  >
                    {submitting ? 'Placing Order...' : 'Place an Order'}
                  </button>
                </div>
              </div>
            </div>

            {/* Right: Order Summary */}
            <div className="col-lg-6">
              <div className="box-border">
                <h5 className="font-md-bold mb-20">Your Order</h5>

                <div className="listCheckout">
                  {cart.length === 0 ? (
                    <p className="text-center color-gray-500 font-sm py-20">Your cart is empty</p>
                  ) : (
                    cart.map(item => {
                      const prod = item.product || {}
                      const price = parseFloat(prod.sale_price || prod.price || item.price || 0)
                      const image = prod.primary_image?.url || '/assets/imgs/page/product/img-sub.png'

                      return (
                        <div className="item-wishlist" key={item.id}>
                          {/* Product info */}
                          <div className="wishlist-product">
                            <div className="product-wishlist">
                              <div className="product-image">
                                <Link to={`/product/${prod.slug || prod.id}`}>
                                  <img
                                    src={image}
                                    alt={prod.name || 'Product'}
                                    onError={e => { e.target.src = '/assets/imgs/page/product/img-sub.png' }}
                                  />
                                </Link>
                              </div>
                              <div className="product-info">
                                <Link to={`/product/${prod.slug || prod.id}`}>
                                  <h6 className="color-brand-3">{prod.name || item.name || 'Product'}</h6>
                                </Link>
                              </div>
                            </div>
                          </div>

                          {/* Quantity */}
                          <div className="wishlist-status">
                            <h5 className="color-gray-500">x{item.quantity}</h5>
                          </div>

                          {/* Line Total Price */}
                          <div className="wishlist-price">
                            <h4 className="color-brand-3 font-lg-bold">
                              ${(price * item.quantity).toFixed(2)}
                            </h4>
                          </div>
                        </div>
                      )
                    })
                  )}
                </div>

                {/* Order Totals */}
                <div className="form-group mb-0 mt-20">
                  <div className="row mb-10">
                    <div className="col-lg-6 col-6">
                      <span className="font-md-bold color-brand-3">Subtotal</span>
                    </div>
                    <div className="col-lg-6 col-6 text-end">
                      <span className="font-lg-bold color-brand-3">${subtotal.toFixed(2)}</span>
                    </div>
                  </div>
                  <div className="border-bottom mb-10 pb-5">
                    <div className="row">
                      <div className="col-lg-6 col-6">
                        <span className="font-md-bold color-brand-3">Shipping</span>
                      </div>
                      <div className="col-lg-6 col-6 text-end">
                        <span className="font-lg-bold color-brand-3">Free</span>
                      </div>
                    </div>
                  </div>
                  <div className="row">
                    <div className="col-lg-6 col-6">
                      <span className="font-md-bold color-brand-3">Total</span>
                    </div>
                    <div className="col-lg-6 col-6 text-end">
                      <span className="font-lg-bold color-brand-3">${total.toFixed(2)}</span>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>
    </main>
  )
}

export default Checkout
