import { useState } from 'react'
import { Link } from 'react-router-dom'
import { useCart } from '../context/CartContext'
import Loading from '../components/ui/Loading'
import toast from 'react-hot-toast'
import { applyCoupon } from '../services/api'

const Cart = () => {
  const { cart, cartCount, updateCart, removeFromCart, cartLoading } = useCart()

  const [couponCode, setCouponCode] = useState('')
  const [couponLoading, setCouponLoading] = useState(false)
  const [discount, setDiscount] = useState(0)

  const handleQtyChange = async (item, newQty) => {
    if (newQty < 1) return
    try {
      await updateCart(item.id, newQty)
    } catch {
      toast.error('Failed to update quantity')
    }
  }

  const handleRemove = async (item) => {
    try {
      await removeFromCart(item.id)
      toast.success('Item removed from cart')
    } catch {
      toast.error('Failed to remove item')
    }
  }

  const handleApplyCoupon = async () => {
    if (!couponCode.trim()) {
      toast.error('Please enter a coupon code')
      return
    }
    setCouponLoading(true)
    try {
      const res = await applyCoupon(couponCode.trim())
      const disc = res?.data?.data?.discount_amount || res?.data?.discount || 0
      setDiscount(parseFloat(disc))
      toast.success('Coupon applied successfully!')
    } catch (err) {
      toast.error(err?.response?.data?.message || 'Invalid coupon code')
    } finally {
      setCouponLoading(false)
    }
  }

  const subtotal = cart.reduce((sum, item) => {
    const price = parseFloat(item.product?.sale_price || item.product?.price || item.price || 0)
    return sum + price * item.quantity
  }, 0)

  const total = Math.max(0, subtotal - discount)

  if (cartLoading) return <Loading height={300} />

  return (
    <main className="main">
      {/* Breadcrumb */}
      <div className="section-box">
        <div className="breadcrumbs-div">
          <div className="container">
            <ul className="breadcrumb">
              <li><Link className="font-xs color-gray-1000" to="/">Home</Link></li>
              <li><Link className="font-xs color-gray-500" to="/shop">Shop</Link></li>
              <li><a className="font-xs color-gray-500">Cart</a></li>
            </ul>
          </div>
        </div>
      </div>

      <section className="section-box shop-template">
        <div className="container">
          <div className="row">
            {/* Cart Items - Left Column */}
            <div className="col-lg-9">
              <div className="box-carts">
                {/* Cart Table Header */}
                <div className="head-wishlist">
                  <div className="item-wishlist">
                    <div className="wishlist-cb">
                      <input className="cb-layout cb-all" type="checkbox" />
                    </div>
                    <div className="wishlist-product">
                      <span className="font-md-bold color-brand-3">Product</span>
                    </div>
                    <div className="wishlist-price">
                      <span className="font-md-bold color-brand-3">Unit Price</span>
                    </div>
                    <div className="wishlist-status">
                      <span className="font-md-bold color-brand-3">Quantity</span>
                    </div>
                    <div className="wishlist-action">
                      <span className="font-md-bold color-brand-3">Subtotal</span>
                    </div>
                    <div className="wishlist-remove">
                      <span className="font-md-bold color-brand-3">Remove</span>
                    </div>
                  </div>
                </div>

                {/* Cart Items Body */}
                <div className="content-wishlist mb-20">
                  {cart.length === 0 ? (
                    <div className="text-center py-30">
                      <p className="color-gray-500 font-sm">Your cart is empty</p>
                      <Link className="btn btn-buy w-auto mt-15" to="/shop">
                        Start Shopping
                      </Link>
                    </div>
                  ) : (
                    cart.map(item => {
                      const prod = item.product || {}
                      const price = parseFloat(prod.sale_price || prod.price || item.price || 0)
                      const lineSubtotal = price * item.quantity
                      const image = prod.primary_image?.url || '/assets/imgs/page/product/img-sub.png'

                      return (
                        <div className="item-wishlist" key={item.id}>
                          {/* Checkbox */}
                          <div className="wishlist-cb">
                            <input className="cb-layout" type="checkbox" />
                          </div>

                          {/* Product Image + Name */}
                          <div className="wishlist-product">
                            <div className="product-wishlist">
                              <div className="product-image">
                                <Link to={`/product/${prod.slug || prod.id}`}>
                                  <img
                                    src={image}
                                    alt={prod.name || 'Product'}
                                    onError={e => { e.target.src = '/assets/imgs/page/product/img-sub.png' }}
                                  />
                                </Link>
                              </div>
                              <div className="product-info">
                                <Link to={`/product/${prod.slug || prod.id}`}>
                                  <h6 className="color-brand-3">{prod.name || item.name || 'Product'}</h6>
                                </Link>
                                {prod.sku && (
                                  <span className="font-xs color-gray-500">SKU: {prod.sku}</span>
                                )}
                              </div>
                            </div>
                          </div>

                          {/* Unit Price */}
                          <div className="wishlist-price">
                            <h4 className="color-brand-3 font-lg-bold">
                              ${price.toFixed(2)}
                            </h4>
                            {prod.sale_price && prod.price && parseFloat(prod.sale_price) < parseFloat(prod.price) && (
                              <span className="font-xs color-gray-500" style={{ textDecoration: 'line-through' }}>
                                ${parseFloat(prod.price).toFixed(2)}
                              </span>
                            )}
                          </div>

                          {/* Quantity Controls */}
                          <div className="wishlist-status">
                            <div className="input-quantity">
                              <input
                                type="number"
                                value={item.quantity}
                                min="1"
                                onChange={e => handleQtyChange(item, parseInt(e.target.value) || 1)}
                                readOnly
                              />
                              <span
                                className="minus-cart"
                                onClick={() => handleQtyChange(item, item.quantity - 1)}
                              />
                              <span
                                className="plus-cart"
                                onClick={() => handleQtyChange(item, item.quantity + 1)}
                              />
                            </div>
                          </div>

                          {/* Line Subtotal */}
                          <div className="wishlist-action">
                            <h4 className="color-brand-3 font-lg-bold">
                              ${lineSubtotal.toFixed(2)}
                            </h4>
                          </div>

                          {/* Remove Button */}
                          <div className="wishlist-remove">
                            <button
                              className="btn btn-delete"
                              onClick={() => handleRemove(item)}
                              title="Remove item"
                            />
                          </div>
                        </div>
                      )
                    })
                  )}
                </div>

                {/* Action Row */}
                <div className="row mb-40">
                  <div className="col-lg-6 col-md-6 col-sm-6">
                    <Link className="btn btn-buy w-auto arrow-back mb-10" to="/shop">
                      Continue shopping
                    </Link>
                  </div>
                  <div className="col-lg-6 col-md-6 col-sm-6 text-md-end">
                    <button
                      className="btn btn-buy w-auto update-cart mb-10"
                      onClick={() => toast.success('Cart updated')}
                    >
                      Update cart
                    </button>
                  </div>
                </div>

                {/* Calculate Shipping + Apply Coupon */}
                <div className="row mb-50">
                  <div className="col-lg-6 col-md-6">
                    <div className="box-cart-left">
                      <h5 className="font-md-bold mb-10">Calculate Shipping</h5>
                      <span className="font-sm-bold mb-5 d-inline-block color-gray-500">Flat rate:</span>
                      <span className="font-sm-bold d-inline-block color-brand-3"> Free</span>
                      <div className="form-group mt-10">
                        <select className="form-control select-style1 color-gray-700">
                          <option value="US">USA</option>
                          <option value="EU">EURO</option>
                          <option value="UK">United Kingdom</option>
                        </select>
                      </div>
                      <div className="row">
                        <div className="col-lg-6 mb-10">
                          <input className="form-control" placeholder="State / Country" />
                        </div>
                        <div className="col-lg-6 mb-10">
                          <input className="form-control" placeholder="PostCode / ZIP" />
                        </div>
                      </div>
                    </div>
                  </div>

                  <div className="col-lg-6 col-md-6">
                    <div className="box-cart-right p-20">
                      <h5 className="font-md-bold mb-10">Apply Coupon</h5>
                      <span className="font-sm-bold mb-5 d-inline-block color-gray-500">
                        Using A Promo Code?
                      </span>
                      <div className="form-group d-flex">
                        <input
                          className="form-control mr-15"
                          placeholder="Enter Your Coupon"
                          value={couponCode}
                          onChange={e => setCouponCode(e.target.value)}
                          onKeyDown={e => e.key === 'Enter' && handleApplyCoupon()}
                        />
                        <button
                          className="btn btn-buy w-auto"
                          onClick={handleApplyCoupon}
                          disabled={couponLoading}
                        >
                          {couponLoading ? '...' : 'Apply'}
                        </button>
                      </div>
                      {discount > 0 && (
                        <p className="font-xs color-brand-3 mt-5">
                          Coupon applied! Discount: -${discount.toFixed(2)}
                        </p>
                      )}
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* Order Summary - Right Column */}
            <div className="col-lg-3">
              <div className="summary-cart">
                {/* Subtotal Row */}
                <div className="border-bottom mb-10">
                  <div className="row">
                    <div className="col-6">
                      <span className="font-md-bold color-gray-500">Subtotal</span>
                    </div>
                    <div className="col-6 text-end">
                      <h4>${subtotal.toFixed(2)}</h4>
                    </div>
                  </div>
                </div>

                {/* Shipping Row */}
                <div className="border-bottom mb-10">
                  <div className="row">
                    <div className="col-6">
                      <span className="font-md-bold color-gray-500">Shipping</span>
                    </div>
                    <div className="col-6 text-end">
                      <h4>Free</h4>
                    </div>
                  </div>
                </div>

                {/* Discount Row */}
                {discount > 0 && (
                  <div className="border-bottom mb-10">
                    <div className="row">
                      <div className="col-6">
                        <span className="font-md-bold color-gray-500">Discount</span>
                      </div>
                      <div className="col-6 text-end">
                        <h4 className="color-brand-3">-${discount.toFixed(2)}</h4>
                      </div>
                    </div>
                  </div>
                )}

                {/* Estimate For Row */}
                <div className="border-bottom mb-10">
                  <div className="row">
                    <div className="col-6">
                      <span className="font-md-bold color-gray-500">Estimate for</span>
                    </div>
                    <div className="col-6 text-end">
                      <h6>United States</h6>
                    </div>
                  </div>
                </div>

                {/* Total Row */}
                <div className="mb-10">
                  <div className="row">
                    <div className="col-6">
                      <span className="font-md-bold color-gray-500">Total</span>
                    </div>
                    <div className="col-6 text-end">
                      <h4>${total.toFixed(2)}</h4>
                    </div>
                  </div>
                </div>

                {/* Checkout Button */}
                <div className="box-button">
                  <Link
                    className="btn btn-buy"
                    to="/checkout"
                    onClick={e => {
                      if (cart.length === 0) {
                        e.preventDefault()
                        toast.error('Your cart is empty')
                      }
                    }}
                  >
                    Proceed To CheckOut
                  </Link>
                </div>

                {/* Items count */}
                <p className="font-xs color-gray-500 text-center mt-10">
                  {cartCount} item{cartCount !== 1 ? 's' : ''} in your cart
                </p>
              </div>
            </div>
          </div>
        </div>
      </section>
    </main>
  )
}

export default Cart
