import { useState, useEffect } from 'react'
import { useSearchParams, useNavigate, Link } from 'react-router-dom'
import { useAuth } from '../context/AuthContext'
import { useWishlist } from '../context/WishlistContext'
import { useCart } from '../context/CartContext'
import { getOrders, updateProfile, changePassword } from '../services/api'
import Loading from '../components/ui/Loading'
import toast from 'react-hot-toast'

const Account = () => {
  const { user, setUser, logout, isLoggedIn } = useAuth()
  const { wishlist, wishlistLoading, removeFromWishlist } = useWishlist()
  const { addToCart } = useCart()
  const navigate = useNavigate()
  const [searchParams, setSearchParams] = useSearchParams()

  const tab = searchParams.get('tab') || 'orders'

  const [orders, setOrders] = useState([])
  const [ordersLoading, setOrdersLoading] = useState(false)

  const [profileForm, setProfileForm] = useState({
    name: user?.name || '',
    email: user?.email || '',
    phone: user?.phone || '',
  })
  const [passwordForm, setPasswordForm] = useState({
    current_password: '',
    password: '',
    password_confirmation: '',
  })
  const [saving, setSaving] = useState(false)

  // Redirect to login if not authenticated
  useEffect(() => {
    if (!isLoggedIn) {
      navigate('/login')
    }
  }, [isLoggedIn, navigate])

  // Sync profile form when user loads
  useEffect(() => {
    if (user) {
      setProfileForm({
        name: user.name || '',
        email: user.email || '',
        phone: user.phone || '',
      })
    }
  }, [user])

  // Load orders when orders tab is active
  useEffect(() => {
    if (tab === 'orders') {
      setOrdersLoading(true)
      getOrders()
        .then(r => {
          const data = r.data.data
          setOrders(Array.isArray(data) ? data : (data?.data || []))
        })
        .catch(() => setOrders([]))
        .finally(() => setOrdersLoading(false))
    }
  }, [tab])

  const setTab = (newTab) => {
    const params = new URLSearchParams(searchParams)
    params.set('tab', newTab)
    setSearchParams(params)
  }

  const handleUpdateProfile = async (e) => {
    e.preventDefault()
    setSaving(true)
    try {
      const res = await updateProfile(profileForm)
      // updateProfile returns user directly
      const updatedUser = res.data?.data || res.data?.user || res.data
      if (setUser && updatedUser) setUser(updatedUser)
      toast.success('Profile updated successfully!')
    } catch (err) {
      toast.error(err?.response?.data?.message || 'Failed to update profile')
    } finally {
      setSaving(false)
    }
  }

  const handleChangePassword = async (e) => {
    e.preventDefault()
    if (passwordForm.password !== passwordForm.password_confirmation) {
      toast.error('Passwords do not match')
      return
    }
    setSaving(true)
    try {
      await changePassword(passwordForm)
      toast.success('Password changed successfully!')
      setPasswordForm({ current_password: '', password: '', password_confirmation: '' })
    } catch (err) {
      toast.error(err?.response?.data?.message || 'Failed to change password')
    } finally {
      setSaving(false)
    }
  }

  const handleAddToCart = async (productId) => {
    try {
      await addToCart(productId, 1)
      toast.success('Added to cart!')
    } catch {
      toast.error('Failed to add to cart')
    }
  }

  const handleRemoveFromWishlist = async (productId) => {
    try {
      await removeFromWishlist(productId)
      toast.success('Removed from wishlist')
    } catch {
      toast.error('Failed to remove from wishlist')
    }
  }

  const getStatusLabel = (status) => {
    const map = {
      pending: 'Pending',
      processing: 'Processing',
      shipped: 'Delivery in progress',
      delivered: 'Delivered',
      cancelled: 'Cancelled',
      refunded: 'Refunded',
    }
    return map[status] || status
  }

  const getStatusClass = (status) => {
    return status === 'delivered' ? 'label-delivery label-delivered' : 'label-delivery'
  }

  if (!isLoggedIn) return null

  return (
    <main className="main">
      <section className="section-box shop-template mt-30">
        <div className="container box-account-template">
          <h3>Hello {user?.name}</h3>
          <p className="font-md color-gray-500">
            From your account dashboard. you can easily check &amp; view your recent orders,
            <br className="d-none d-lg-block" />
            manage your shipping and billing addresses and edit your password and account details.
          </p>

          <div className="box-tabs mb-100">
            <ul className="nav nav-tabs nav-tabs-account" role="tablist">
              <li>
                <a
                  className={tab === 'orders' ? 'active' : ''}
                  onClick={(e) => { e.preventDefault(); setTab('orders') }}
                  href="#"
                  role="tab"
                >
                  Orders
                </a>
              </li>
              <li>
                <a
                  className={tab === 'wishlist' ? 'active' : ''}
                  onClick={(e) => { e.preventDefault(); setTab('wishlist') }}
                  href="#"
                  role="tab"
                >
                  Wishlist
                </a>
              </li>
              <li>
                <a
                  className={tab === 'setting' ? 'active' : ''}
                  onClick={(e) => { e.preventDefault(); setTab('setting') }}
                  href="#"
                  role="tab"
                >
                  Setting
                </a>
              </li>
            </ul>

            <div className="border-bottom mt-20 mb-40"></div>

            <div className="tab-content mt-30">

              {/* ── Orders Tab ── */}
              {tab === 'orders' && (
                <div>
                  {ordersLoading ? (
                    <Loading height={200} />
                  ) : orders.length === 0 ? (
                    <div className="text-center" style={{ padding: '40px 0' }}>
                      <p className="font-md color-gray-500 mb-20">You haven&apos;t placed any orders yet.</p>
                      <Link to="/shop" className="btn btn-buy w-auto">Start Shopping</Link>
                    </div>
                  ) : (
                    orders.map(order => (
                      <div className="box-orders" key={order.id}>
                        <div className="head-orders">
                          <div className="head-left">
                            <h5 className="mr-20">Order ID: #{order.order_number || order.id}</h5>
                            <span className="font-md color-brand-3 mr-20">
                              Date: {new Date(order.created_at).toLocaleDateString()}
                            </span>
                            <span className={getStatusClass(order.status)}>
                              {getStatusLabel(order.status)}
                            </span>
                          </div>
                          <div className="head-right">
                            <span className="font-md-bold color-brand-3 mr-20">
                              ${parseFloat(order.total_amount || order.total || 0).toFixed(2)}
                            </span>
                          </div>
                        </div>

                        {order.items && order.items.length > 0 && (
                          <div className="body-orders">
                            <div className="list-orders">
                              {order.items.map((item, idx) => {
                                const prod = item.product || {}
                                const image = prod.primary_image?.url || prod.image_url || '/assets/imgs/no-image.png'
                                return (
                                  <div className="item-orders" key={item.id || idx}>
                                    <div className="image-orders">
                                      <img src={image} alt={prod.name || 'Product'} />
                                    </div>
                                    <div className="info-orders">
                                      <h5>{prod.name || item.product_name || 'Product'}</h5>
                                    </div>
                                    <div className="quantity-orders">
                                      <h5>Quantity: {String(item.quantity || 1).padStart(2, '0')}</h5>
                                    </div>
                                    <div className="price-orders">
                                      <h3>${parseFloat(item.price || item.subtotal || 0).toFixed(2)}</h3>
                                    </div>
                                  </div>
                                )
                              })}
                            </div>
                          </div>
                        )}
                      </div>
                    ))
                  )}
                </div>
              )}

              {/* ── Wishlist Tab ── */}
              {tab === 'wishlist' && (
                <div className="box-wishlist">
                  <div className="head-wishlist">
                    <div className="item-wishlist">
                      <div className="wishlist-cb">
                        <input className="cb-layout cb-all" type="checkbox" />
                      </div>
                      <div className="wishlist-product">
                        <span className="font-md-bold color-brand-3">Product</span>
                      </div>
                      <div className="wishlist-price">
                        <span className="font-md-bold color-brand-3">Price</span>
                      </div>
                      <div className="wishlist-status">
                        <span className="font-md-bold color-brand-3">Stock status</span>
                      </div>
                      <div className="wishlist-action">
                        <span className="font-md-bold color-brand-3">Action</span>
                      </div>
                      <div className="wishlist-remove">
                        <span className="font-md-bold color-brand-3">Remove</span>
                      </div>
                    </div>
                  </div>

                  <div className="content-wishlist">
                    {wishlistLoading ? (
                      <Loading height={150} />
                    ) : wishlist.length === 0 ? (
                      <p className="text-center color-gray-500 font-sm py-30">
                        Your wishlist is empty.{' '}
                        <Link className="color-brand-3" to="/shop">Browse products</Link>
                      </p>
                    ) : (
                      wishlist.map(item => {
                        const product = item.product || item
                        const productId = product.id || item.product_id
                        const price = parseFloat(product.sale_price || product.price || 0)
                        const image = product.primary_image?.url || product.image_url || '/assets/imgs/no-image.png'
                        const inStock = product.stock > 0

                        return (
                          <div className="item-wishlist" key={item.id || productId}>
                            <div className="wishlist-cb">
                              <input className="cb-layout cb-select" type="checkbox" />
                            </div>

                            <div className="wishlist-product">
                              <div className="product-wishlist">
                                <div className="product-image">
                                  <Link to={`/product/${product.slug}`}>
                                    <img src={image} alt={product.name} />
                                  </Link>
                                </div>
                                <div className="product-info">
                                  <Link to={`/product/${product.slug}`}>
                                    <h6 className="color-brand-3">{product.name}</h6>
                                  </Link>
                                  <div className="rating">
                                    {[1, 2, 3, 4, 5].map(star => (
                                      <img
                                        key={star}
                                        src={star <= Math.round(product.rating || 0)
                                          ? '/assets/imgs/template/icons/star.svg'
                                          : '/assets/imgs/template/icons/star-gray.svg'}
                                        alt="star"
                                      />
                                    ))}
                                    {product.reviews_count != null && (
                                      <span className="font-xs color-gray-500"> ({product.reviews_count})</span>
                                    )}
                                  </div>
                                </div>
                              </div>
                            </div>

                            <div className="wishlist-price">
                              <h4 className="color-brand-3">${price.toFixed(2)}</h4>
                              {product.sale_price && product.price && (
                                <span className="font-xs color-gray-500" style={{ textDecoration: 'line-through' }}>
                                  ${parseFloat(product.price).toFixed(2)}
                                </span>
                              )}
                            </div>

                            <div className="wishlist-status">
                              <span className="btn btn-gray font-md-bold color-brand-3">
                                {inStock ? 'In Stock' : 'Out of Stock'}
                              </span>
                            </div>

                            <div className="wishlist-action">
                              <button
                                className="btn btn-cart font-sm-bold"
                                onClick={() => handleAddToCart(productId)}
                                disabled={!inStock}
                              >
                                Add to Cart
                              </button>
                            </div>

                            <div className="wishlist-remove">
                              <button
                                className="btn btn-delete"
                                onClick={() => handleRemoveFromWishlist(productId)}
                                aria-label="Remove from wishlist"
                              ></button>
                            </div>
                          </div>
                        )
                      })
                    )}
                  </div>
                </div>
              )}

              {/* ── Setting Tab ── */}
              {tab === 'setting' && (
                <div className="row">
                  {/* Profile Information */}
                  <div className="col-lg-6">
                    <h5 className="font-md-bold mb-20">Profile Information</h5>
                    <form onSubmit={handleUpdateProfile}>
                      <div className="form-group">
                        <label className="font-sm color-gray-700">Full Name</label>
                        <input
                          className="form-control"
                          name="name"
                          type="text"
                          value={profileForm.name}
                          onChange={e => setProfileForm(f => ({ ...f, name: e.target.value }))}
                          required
                        />
                      </div>
                      <div className="form-group">
                        <label className="font-sm color-gray-700">Email</label>
                        <input
                          className="form-control"
                          name="email"
                          type="email"
                          value={profileForm.email}
                          onChange={e => setProfileForm(f => ({ ...f, email: e.target.value }))}
                          required
                        />
                      </div>
                      <div className="form-group">
                        <label className="font-sm color-gray-700">Phone</label>
                        <input
                          className="form-control"
                          name="phone"
                          type="text"
                          value={profileForm.phone}
                          onChange={e => setProfileForm(f => ({ ...f, phone: e.target.value }))}
                        />
                      </div>
                      <button className="btn btn-buy" type="submit" disabled={saving}>
                        {saving ? 'Saving...' : 'Update Profile'}
                      </button>
                    </form>
                  </div>

                  {/* Change Password */}
                  <div className="col-lg-6">
                    <h5 className="font-md-bold mb-20">Change Password</h5>
                    <form onSubmit={handleChangePassword}>
                      <div className="form-group">
                        <label className="font-sm color-gray-700">Current Password</label>
                        <input
                          className="form-control"
                          type="password"
                          name="current_password"
                          value={passwordForm.current_password}
                          onChange={e => setPasswordForm(f => ({ ...f, current_password: e.target.value }))}
                          required
                        />
                      </div>
                      <div className="form-group">
                        <label className="font-sm color-gray-700">New Password</label>
                        <input
                          className="form-control"
                          type="password"
                          name="password"
                          value={passwordForm.password}
                          onChange={e => setPasswordForm(f => ({ ...f, password: e.target.value }))}
                          required
                          minLength={8}
                        />
                      </div>
                      <div className="form-group">
                        <label className="font-sm color-gray-700">Confirm Password</label>
                        <input
                          className="form-control"
                          type="password"
                          name="password_confirmation"
                          value={passwordForm.password_confirmation}
                          onChange={e => setPasswordForm(f => ({ ...f, password_confirmation: e.target.value }))}
                          required
                        />
                      </div>
                      <button className="btn btn-buy" type="submit" disabled={saving}>
                        {saving ? 'Changing...' : 'Change Password'}
                      </button>
                    </form>
                  </div>

                  {/* Logout */}
                  <div className="col-12 mt-40">
                    <div className="border-bottom mb-20"></div>
                    <button
                      className="btn btn-gray font-sm-bold"
                      onClick={logout}
                      style={{ cursor: 'pointer' }}
                    >
                      Sign Out
                    </button>
                  </div>
                </div>
              )}

            </div>
          </div>
        </div>
      </section>
    </main>
  )
}

export default Account
