import { createContext, useContext, useState, useEffect, useCallback } from 'react'
import { getWishlist, addWishlistItem, removeWishlistItem } from '../services/api'
import { useAuth } from './AuthContext'

const WishlistContext = createContext()

export const WishlistProvider = ({ children }) => {
  const [wishlist, setWishlist] = useState([])
  const [wishlistLoading, setWishlistLoading] = useState(false)
  const { isLoggedIn } = useAuth()

  const fetchWishlist = useCallback(async () => {
    if (!isLoggedIn) return
    setWishlistLoading(true)
    try {
      const res = await getWishlist()
      // Wishlist API returns array directly
      setWishlist(Array.isArray(res.data) ? res.data : (res.data?.data || []))
    } catch {}
    finally { setWishlistLoading(false) }
  }, [isLoggedIn])

  useEffect(() => {
    if (isLoggedIn) {
      fetchWishlist()
    } else {
      try { setWishlist(JSON.parse(localStorage.getItem('wishlist')) || []) } catch { setWishlist([]) }
    }
  }, [isLoggedIn, fetchWishlist])

  const addToWishlist = async (productId) => {
    if (isLoggedIn) {
      await addWishlistItem(productId)
      await fetchWishlist()
    } else {
      setWishlist(prev => {
        if (prev.find(i => (i.id || i.product_id) === productId)) return prev
        const updated = [...prev, { id: productId, product_id: productId }]
        localStorage.setItem('wishlist', JSON.stringify(updated))
        return updated
      })
    }
  }

  const removeFromWishlist = async (productId) => {
    if (isLoggedIn) {
      await removeWishlistItem(productId)
      setWishlist(prev => prev.filter(i => (i.product_id || i.product?.id || i.id) !== productId))
    } else {
      setWishlist(prev => {
        const updated = prev.filter(i => (i.id || i.product_id) !== productId)
        localStorage.setItem('wishlist', JSON.stringify(updated))
        return updated
      })
    }
  }

  const isInWishlist = (productId) =>
    wishlist.some(i => i.id === productId || i.product_id === productId || i.product?.id === productId)

  return (
    <WishlistContext.Provider value={{ wishlist, wishlistLoading, addToWishlist, removeFromWishlist, isInWishlist, fetchWishlist }}>
      {children}
    </WishlistContext.Provider>
  )
}

export const useWishlist = () => useContext(WishlistContext)
