import { createContext, useContext, useState, useEffect, useCallback } from 'react'
import { getCart, addCartItem, updateCartItem, removeCartItem, clearCartApi } from '../services/api'
import { useAuth } from './AuthContext'

const CartContext = createContext()

export const CartProvider = ({ children }) => {
  const [cart, setCart] = useState([])
  const [cartLoading, setCartLoading] = useState(false)
  const { isLoggedIn } = useAuth()

  // Guest cart from localStorage
  const [localCart, setLocalCart] = useState(() => {
    try { return JSON.parse(localStorage.getItem('guestCart')) || [] } catch { return [] }
  })

  const fetchCart = useCallback(async () => {
    if (!isLoggedIn) return
    setCartLoading(true)
    try {
      const res = await getCart()
      const data = res.data
      // Cart API returns { items: [...], subtotal, total }
      setCart(Array.isArray(data) ? data : (data?.items || []))
    } catch {}
    finally { setCartLoading(false) }
  }, [isLoggedIn])

  useEffect(() => {
    if (isLoggedIn) {
      fetchCart()
    } else {
      setCart([])
    }
  }, [isLoggedIn, fetchCart])

  useEffect(() => {
    localStorage.setItem('guestCart', JSON.stringify(localCart))
  }, [localCart])

  // Merge guest cart on login
  useEffect(() => {
    if (isLoggedIn && localCart.length > 0) {
      Promise.all(localCart.map(item => addCartItem(item.product_id, item.quantity).catch(() => {})))
        .then(() => {
          setLocalCart([])
          fetchCart()
        })
    }
  }, [isLoggedIn]) // eslint-disable-line

  const addToCart = async (productId, quantity = 1) => {
    if (isLoggedIn) {
      await addCartItem(productId, quantity)
      await fetchCart()
    } else {
      setLocalCart(prev => {
        const existing = prev.find(i => i.product_id === productId)
        if (existing) return prev.map(i => i.product_id === productId ? { ...i, quantity: i.quantity + quantity } : i)
        return [...prev, { id: Date.now(), product_id: productId, quantity, product: { id: productId } }]
      })
    }
  }

  const updateCart = async (itemId, quantity) => {
    if (quantity < 1) return
    if (isLoggedIn) {
      await updateCartItem(itemId, quantity)
      await fetchCart()
    } else {
      setLocalCart(prev => prev.map(i => i.id === itemId ? { ...i, quantity } : i))
    }
  }

  const removeFromCart = async (itemId) => {
    if (isLoggedIn) {
      await removeCartItem(itemId)
      await fetchCart()
    } else {
      setLocalCart(prev => prev.filter(i => i.id !== itemId))
    }
  }

  const clearCart = async () => {
    if (isLoggedIn) {
      try { await clearCartApi() } catch {}
      setCart([])
    } else {
      setLocalCart([])
    }
  }

  const activeCart = isLoggedIn ? cart : localCart
  const cartCount = activeCart.reduce((sum, item) => sum + (item.quantity || 0), 0)

  return (
    <CartContext.Provider value={{
      cart: activeCart,
      cartCount,
      cartLoading,
      addToCart,
      updateCart,
      removeFromCart,
      clearCart,
      fetchCart,
    }}>
      {children}
    </CartContext.Provider>
  )
}

export const useCart = () => useContext(CartContext)
