import { Link } from 'react-router-dom'
import { useCart } from '../../context/CartContext'
import { useWishlist } from '../../context/WishlistContext'
import { addToCompare } from '../../services/api'
import toast from 'react-hot-toast'

const ProductCard = ({ product }) => {
  const { addToCart } = useCart()
  const { addToWishlist, removeFromWishlist, isInWishlist } = useWishlist()
  const inWishlist = isInWishlist(product.id)

  const handleAddToCart = async (e) => {
    e.preventDefault()
    try {
      await addToCart(product.id, 1)
      toast.success('Added to cart!')
    } catch {
      toast.error('Failed to add to cart')
    }
  }

  const handleWishlist = async (e) => {
    e.preventDefault()
    try {
      if (inWishlist) {
        await removeFromWishlist(product.id)
        toast.success('Removed from wishlist')
      } else {
        await addToWishlist(product.id)
        toast.success('Added to wishlist!')
      }
    } catch {
      toast.error('Action failed')
    }
  }

  const handleCompare = async (e) => {
    e.preventDefault()
    try {
      await addToCompare(product.id)
      toast.success('Added to compare!')
    } catch {
      toast.error('Failed to add to compare')
    }
  }

  const image = product.primary_image?.url || product.images?.[0]?.url || '/assets/imgs/no-image.png'
  const price = parseFloat(product.price || 0)
  const salePrice = product.sale_price ? parseFloat(product.sale_price) : null
  const discount = salePrice ? Math.round((1 - salePrice / price) * 100) : null

  return (
    <div className="col-lg-3 col-md-4 col-sm-6 col-12">
      <div className="card-grid-style-3 hover-up">
        <div className="card-grid-inner">
          <div className="tools">
            <a className="btn btn-wishlist mb-10" href="#" onClick={handleWishlist}
              title={inWishlist ? 'Remove from Wishlist' : 'Add to Wishlist'}
              style={{ opacity: inWishlist ? 1 : undefined }}></a>
            <br />
            <a className="btn btn-compare mb-10" href="#" onClick={handleCompare} title="Compare"></a>
            <br />
            <Link className="btn btn-quickview" to={`/product/${product.slug}`} title="Quick View"></Link>
          </div>
          {discount && (
            <div className="product-badges">
              <span className="hot">-{discount}%</span>
            </div>
          )}
          {product.stock === 0 && !discount && (
            <div className="product-badges">
              <span style={{ background: '#aaa' }}>Out of Stock</span>
            </div>
          )}
          <div className="image-box">
            <Link to={`/product/${product.slug}`}>
              <img src={image} alt={product.name}
                style={{ width: '100%', height: 200, objectFit: 'contain', padding: 10 }}
                onError={(e) => { e.target.onerror = null; e.target.src = '/assets/imgs/no-image.png' }} />
            </Link>
          </div>
          <div className="info-right">
            <a className="font-xs color-gray-500" href="#">
              {product.category?.name || 'General'}
            </a>
            <br />
            <Link className="color-brand-3 font-sm-bold" to={`/product/${product.slug}`}>
              {product.name}
            </Link>
            <div className="rating mt-5 mb-5">
              {[1, 2, 3, 4, 5].map(s => (
                <img key={s}
                  src={s <= Math.round(product.rating || 0)
                    ? '/assets/imgs/template/icons/star.svg'
                    : '/assets/imgs/template/icons/star-gray.svg'}
                  alt="star" />
              ))}
              <span className="font-xs color-gray-500 font-medium"> ({product.reviews_count || 0})</span>
            </div>
            {product.vendor && (
              <div className="font-xs color-gray-500 mb-5">
                by <Link className="color-brand-3" to={`/vendors/${product.vendor.id}`}>{product.vendor.store_name}</Link>
              </div>
            )}
            <div className="price-info">
              {salePrice ? (
                <>
                  <strong className="font-lg-bold color-brand-3 price-main">${salePrice.toFixed(2)}</strong>
                  <span className="color-gray-500 price-line font-md line-througt ml-10">${price.toFixed(2)}</span>
                </>
              ) : (
                <strong className="font-lg-bold color-brand-3 price-main">${price.toFixed(2)}</strong>
              )}
            </div>
            <div className="mt-20 d-flex gap-2">
              <a className="btn btn-cart" href="#" onClick={handleAddToCart}
                style={{ pointerEvents: product.stock === 0 ? 'none' : 'auto', opacity: product.stock === 0 ? 0.5 : 1 }}>
                {product.stock === 0 ? 'Out of Stock' : 'Add to cart'}
              </a>
              <Link className="btn btn-buy" to={`/product/${product.slug}`}>Buy now</Link>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
export default ProductCard
