import { Link, useNavigate } from 'react-router-dom'
import { useCart } from '../../context/CartContext'
import { useWishlist } from '../../context/WishlistContext'
import { useAuth } from '../../context/AuthContext'
import { useState, useEffect, useRef } from 'react'
import { getCategories, searchProducts } from '../../services/api'

const Header = () => {
  const { cartCount } = useCart()
  const { wishlist } = useWishlist()
  const { user, logout, isLoggedIn } = useAuth()
  const [categories, setCategories] = useState([])
  const [searchQuery, setSearchQuery] = useState('')
  const [searchResults, setSearchResults] = useState([])
  const [showSearch, setShowSearch] = useState(false)
  const [showAccount, setShowAccount] = useState(false)
  const [showCart, setShowCart] = useState(false)
  const searchRef = useRef(null)
  const navigate = useNavigate()

  useEffect(() => {
    getCategories().then(res => setCategories(res.data.data || [])).catch(() => {})
  }, [])

  useEffect(() => {
    const handler = (e) => {
      if (searchRef.current && !searchRef.current.contains(e.target)) setShowSearch(false)
      if (!e.target.closest('.box-dropdown-cart') && !e.target.closest('.dropdown-account') && !e.target.closest('.dropdown-cart')) {
        setShowAccount(false)
        setShowCart(false)
      }
    }
    document.addEventListener('mousedown', handler)
    return () => document.removeEventListener('mousedown', handler)
  }, [])

  const handleSearch = async (e) => {
    const q = e.target.value
    setSearchQuery(q)
    if (q.length > 2) {
      try {
        const res = await searchProducts(q)
        setSearchResults(res.data.data?.data || [])
        setShowSearch(true)
      } catch {}
    } else {
      setShowSearch(false)
    }
  }

  const handleSearchSubmit = (e) => {
    e.preventDefault()
    if (searchQuery) {
      navigate(`/shop?search=${encodeURIComponent(searchQuery)}`)
      setShowSearch(false)
    }
  }

  return (
    <>
      {/* Topbar */}
      <div className="topbar">
        <div className="container-topbar">
          <div className="menu-topbar-left d-none d-xl-block">
            <ul className="nav-small">
              <li><Link className="font-xs" to="/about">About Us</Link></li>
              <li><Link className="font-xs" to="/contact">Contact Us</Link></li>
              <li><Link className="font-xs" to="/careers">Careers</Link></li>
            </ul>
          </div>
          <div className="menu-topbar-right">
            <span className="font-xs color-brand-3">Need help? Call Us:</span>
            <span className="font-sm-bold color-success"> 1900-888</span>
          </div>
        </div>
      </div>

      {/* Main Header */}
      <header className="header sticky-bar">
        <div className="container">
          <div className="main-header">
            <div className="header-left">
              {/* Logo */}
              <div className="header-logo">
                <Link className="d-flex" to="/">
                  <img alt="Kalses" src="/assets/imgs/template/logo.png"
                    onError={e => { e.target.onerror = null; e.target.src = '/assets/imgs/template/logo.svg' }} />
                </Link>
              </div>

              {/* Search */}
              <div className="header-search" ref={searchRef} style={{ position: 'relative' }}>
                <div className="box-header-search">
                  <form className="form-search" onSubmit={handleSearchSubmit}>
                    <div className="box-category">
                      <select className="select-active" onChange={e => { if (e.target.value) navigate(`/shop/category/${e.target.value}`) }}>
                        <option value="">All categories</option>
                        {categories.map(cat => (
                          <option key={cat.id} value={cat.slug}>{cat.name}</option>
                        ))}
                      </select>
                    </div>
                    <div className="box-keysearch">
                      <input
                        className="form-control font-xs"
                        type="text"
                        value={searchQuery}
                        onChange={handleSearch}
                        placeholder="Search for items..."
                      />
                    </div>
                  </form>
                </div>
                {showSearch && searchResults.length > 0 && (
                  <div style={{
                    position: 'absolute', top: '100%', left: 0, right: 0,
                    background: '#fff', border: '1px solid #eee', zIndex: 9999,
                    maxHeight: 320, overflowY: 'auto', borderRadius: '0 0 8px 8px',
                    boxShadow: '0 8px 24px rgba(0,0,0,0.12)'
                  }}>
                    {searchResults.slice(0, 6).map(p => (
                      <Link key={p.id} to={`/product/${p.slug}`}
                        onClick={() => setShowSearch(false)}
                        style={{ display: 'flex', alignItems: 'center', padding: '10px 16px', borderBottom: '1px solid #f5f5f5', textDecoration: 'none', color: 'inherit' }}>
                        <img src={p.primary_image?.url || '/assets/imgs/no-image.png'}
                          style={{ width: 44, height: 44, objectFit: 'cover', marginRight: 12, borderRadius: 6, flexShrink: 0 }}
                          alt={p.name} />
                        <div>
                          <div style={{ fontWeight: 600, fontSize: 14 }}>{p.name}</div>
                          <div className="color-brand-3" style={{ fontSize: 13 }}>
                            ${parseFloat(p.sale_price || p.price || 0).toFixed(2)}
                          </div>
                        </div>
                      </Link>
                    ))}
                    <div style={{ padding: '10px 16px', textAlign: 'center' }}>
                      <button onClick={handleSearchSubmit}
                        style={{ background: 'none', border: 'none', color: '#3BB77E', cursor: 'pointer', fontWeight: 600, fontSize: 13 }}>
                        See all results →
                      </button>
                    </div>
                  </div>
                )}
              </div>

              {/* Nav */}
              <div className="header-nav">
                <nav className="nav-main-menu d-none d-xl-block">
                  <ul className="main-menu">
                    <li><Link to="/">Home</Link></li>
                    <li className="has-children">
                      <Link to="/shop">Shop</Link>
                      {categories.length > 0 && (
                        <ul className="sub-menu">
                          {categories.slice(0, 8).map(cat => (
                            <li key={cat.id}><Link to={`/shop/category/${cat.slug}`}>{cat.name}</Link></li>
                          ))}
                          <li><Link to="/shop" className="color-brand-3">All Categories →</Link></li>
                        </ul>
                      )}
                    </li>
                    <li><Link to="/vendors">Vendors</Link></li>
                    <li><Link to="/wholesale">Wholesale (B2B)</Link></li>
                    <li><Link to="/about">About Us</Link></li>
                    <li><Link to="/contact">Contact Us</Link></li>
                  </ul>
                </nav>
                <div className="burger-icon burger-icon-white">
                  <span className="burger-icon-top"></span>
                  <span className="burger-icon-mid"></span>
                  <span className="burger-icon-bottom"></span>
                </div>
              </div>

              {/* Shop icons */}
              <div className="header-shop">
                {/* Account */}
                <div className="d-inline-block box-dropdown-cart">
                  <span className="font-lg icon-list icon-account"
                    onClick={() => { setShowAccount(!showAccount); setShowCart(false) }}>
                    <span>{isLoggedIn ? user?.name?.split(' ')[0] || 'Account' : 'Account'}</span>
                  </span>
                  {showAccount && (
                    <div className="dropdown-account dropdown-open">
                      <ul>
                        {isLoggedIn ? (
                          <>
                            <li><Link to="/account" onClick={() => setShowAccount(false)}>My Account</Link></li>
                            <li><Link to="/account?tab=orders" onClick={() => setShowAccount(false)}>My Orders</Link></li>
                            <li><Link to="/wishlist" onClick={() => setShowAccount(false)}>My Wishlist</Link></li>
                            <li><Link to="/wholesale" onClick={() => setShowAccount(false)}>Wholesale Application</Link></li>
                            <li>
                              <button onClick={() => { logout(); setShowAccount(false) }}
                                style={{ background: 'none', border: 'none', padding: 0, cursor: 'pointer', width: '100%', textAlign: 'left', color: 'inherit', fontSize: 'inherit' }}>
                                Sign out
                              </button>
                            </li>
                          </>
                        ) : (
                          <>
                            <li><Link to="/login" onClick={() => setShowAccount(false)}>Sign In</Link></li>
                            <li><Link to="/register" onClick={() => setShowAccount(false)}>Create Account</Link></li>
                            <li><Link to="/wholesale" onClick={() => setShowAccount(false)}>Wholesale Application</Link></li>
                          </>
                        )}
                      </ul>
                    </div>
                  )}
                </div>

                {/* Wishlist */}
                <Link className="font-lg icon-list icon-wishlist" to="/wishlist">
                  <span>Wishlist</span>
                  <span className="number-item font-xs">{wishlist.length}</span>
                </Link>

                {/* Cart */}
                <div className="d-inline-block box-dropdown-cart">
                  <span className="font-lg icon-list icon-cart"
                    onClick={() => { setShowCart(!showCart); setShowAccount(false) }}>
                    <span>Cart</span>
                    <span className="number-item font-xs">{cartCount}</span>
                  </span>
                  {showCart && (
                    <div className="dropdown-cart dropdown-open">
                      <div className="text-center py-4">
                        <p className="color-gray-500">
                          {cartCount > 0 ? `${cartCount} item(s) in cart` : 'Your cart is empty'}
                        </p>
                        {cartCount > 0 ? (
                          <Link className="btn btn-buy mt-10" to="/cart" onClick={() => setShowCart(false)}>View Cart</Link>
                        ) : (
                          <Link className="btn btn-brand-2 mt-10" to="/shop" onClick={() => setShowCart(false)}>Start Shopping</Link>
                        )}
                      </div>
                    </div>
                  )}
                </div>
              </div>
            </div>
          </div>
        </div>
      </header>

      {/* Mobile menu */}
      <div className="mobile-header-active mobile-header-wrapper-style perfect-scrollbar">
        <div className="mobile-header-wrapper-inner">
          <div className="mobile-header-content-area">
            <div className="mobile-logo">
              <Link className="d-flex" to="/">
                <img alt="Kalses" src="/assets/imgs/template/logo.png"
                  onError={e => { e.target.onerror = null; e.target.src = '/assets/imgs/template/logo.svg' }} />
              </Link>
            </div>
            <div className="perfect-scroll">
              <div className="mobile-menu-wrap mobile-header-border">
                <nav className="mt-15">
                  <ul className="mobile-menu font-heading">
                    <li><Link to="/">Home</Link></li>
                    <li><Link to="/shop">Shop</Link></li>
                    <li><Link to="/vendors">Vendors</Link></li>
                    <li><Link to="/wholesale">Wholesale</Link></li>
                    <li><Link to="/about">About Us</Link></li>
                    <li><Link to="/contact">Contact</Link></li>
                  </ul>
                </nav>
              </div>
              <div className="mobile-account">
                <div className="mobile-header-top">
                  <div className="user-account">
                    <img src="/assets/imgs/template/ava_1.png" alt="Kalses" />
                    <div className="content">
                      {isLoggedIn ? (
                        <>
                          <h6 className="user-name">Hello<span className="text-brand"> {user?.name?.split(' ')[0]}!</span></h6>
                          <p className="font-xs text-muted">Welcome back</p>
                        </>
                      ) : (
                        <>
                          <h6 className="user-name">Hello<span className="text-brand"> Guest!</span></h6>
                          <p className="font-xs text-muted">Sign in to your account</p>
                        </>
                      )}
                    </div>
                  </div>
                </div>
                <ul className="mobile-menu">
                  {isLoggedIn ? (
                    <>
                      <li><Link to="/account">My Account</Link></li>
                      <li><Link to="/account?tab=orders">My Orders</Link></li>
                      <li><Link to="/wishlist">My Wishlist</Link></li>
                      <li><button onClick={logout} style={{ background: 'none', border: 'none', padding: '5px 0', cursor: 'pointer', color: 'inherit' }}>Sign out</button></li>
                    </>
                  ) : (
                    <>
                      <li><Link to="/login">Sign In</Link></li>
                      <li><Link to="/register">Create Account</Link></li>
                    </>
                  )}
                </ul>
              </div>
              <div className="site-copyright color-gray-400 mt-30">
                Copyright &copy; {new Date().getFullYear()} Kalses Store
              </div>
            </div>
          </div>
        </div>
      </div>
    </>
  )
}
export default Header
