(function() {
    'use strict';

    const API_BASE = window.KALSES_API_BASE || 'http://127.0.0.1:8000/api/v1';
    let currentPage = 1;
    let currentCategory = null;
    let currentSearch = '';
    let currentMinPrice = null;
    let currentMaxPrice = null;
    let currentVendor = null;
    let currentSort = 'created_at';
    let currentSortOrder = 'desc';
    const perPage = 12;

    // Initialize when DOM is ready
    document.addEventListener('DOMContentLoaded', function() {
        // Check URL parameters
        const urlParams = new URLSearchParams(window.location.search);
        if (urlParams.get('category')) {
            currentCategory = urlParams.get('category');
        }
        if (urlParams.get('search')) {
            currentSearch = urlParams.get('search');
        }
        if (urlParams.get('page')) {
            currentPage = parseInt(urlParams.get('page')) || 1;
        }
        if (urlParams.get('vendor')) {
            currentVendor = urlParams.get('vendor');
        }

        // Load data
        loadProducts();
        loadCategories();
        loadVendors();
        setupPriceFilter();

        // Setup search form
        setupSearchForm();

        // Setup sort dropdown
        setupSortDropdown();
    });

    function loadProducts() {
        const container = document.getElementById('products-container');
        if (!container) return;

        // Show loading
        container.innerHTML = '<div class="col-12 text-center py-5"><div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading...</span></div></div>';

        // Build query string
        let queryParams = new URLSearchParams();
        queryParams.append('page', currentPage);
        queryParams.append('per_page', perPage);
        queryParams.append('sort_by', currentSort);
        queryParams.append('sort_order', currentSortOrder);

        if (currentCategory) queryParams.append('category', currentCategory);
        if (currentSearch) queryParams.append('search', currentSearch);
        if (currentMinPrice) queryParams.append('min_price', currentMinPrice);
        if (currentMaxPrice) queryParams.append('max_price', currentMaxPrice);
        if (currentVendor) queryParams.append('vendor', currentVendor);

        fetch(API_BASE + '/products?' + queryParams.toString())
            .then(response => response.json())
            .then(data => {
                if (data.success && data.data) {
                    const products = data.data.data || data.data;
                    renderProducts(products, container);
                    renderPagination(data.data);
                    updateProductCount(data.data);
                } else {
                    container.innerHTML = '<div class="col-12 text-center py-5"><p>No products found</p></div>';
                }
            })
            .catch(error => {
                console.error('Error loading products:', error);
                container.innerHTML = '<div class="col-12 text-center py-5"><p>Error loading products</p></div>';
            });
    }

    function renderProducts(products, container) {
        if (!products || products.length === 0) {
            container.innerHTML = '<div class="col-12 text-center py-5"><p>No products found</p></div>';
            return;
        }

        // Check if list view
        const isListView = container.classList.contains('display-list');

        let html = '';
        products.forEach(product => {
            const imageUrl = product.primary_image?.url || product.image_url || 'assets/imgs/page/homepage1/imgsp1.png';
            const discount = product.sale_price && product.price > product.sale_price
                ? Math.round((1 - product.sale_price / product.price) * 100)
                : 0;
            const displayPrice = product.sale_price || product.price;
            const vendorName = product.vendor?.name || 'Fareen Store';
            const vendorSlug = product.vendor?.slug || 'fareen-store';
            const rating = product.average_rating || 5;
            const reviewCount = product.reviews_count || 0;

            if (isListView) {
                // List view template
                html += `
                <div class="col-lg-12">
                    <div class="card-grid-style-3">
                        <div class="card-grid-inner">
                            <div class="tools">
                                <a class="btn btn-wishlist btn-tooltip mb-10" href="shop-wishlist.html" aria-label="Add To Wishlist"></a>
                                <a class="btn btn-quickview btn-tooltip" aria-label="Quick view" href="#ModalQuickview" data-bs-toggle="modal"></a>
                            </div>
                            <div class="image-box">
                                ${discount > 0 ? `<span class="label bg-brand-2">-${discount}%</span>` : ''}
                                <a href="shop-single-product.html?slug=${product.slug}">
                                    <img src="${imageUrl}" alt="${product.name}">
                                </a>
                            </div>
                            <div class="info-right">
                                <span class="font-xs color-gray-500">${vendorName}</span><br>
                                <a href="shop-single-product.html?slug=${product.slug}">
                                    <h4 class="color-brand-3">${product.name}</h4>
                                </a>
                                <div class="rating">
                                    ${generateStars(rating)}
                                    <span class="font-xs color-gray-500"> (${reviewCount})</span>
                                </div>
                                <div class="price-info">
                                    <strong class="font-lg-bold color-brand-3 price-main">$${parseFloat(displayPrice).toFixed(2)}</strong>
                                    ${product.sale_price && product.price > product.sale_price ? `<span class="color-gray-500 price-line">$${parseFloat(product.price).toFixed(2)}</span>` : ''}
                                </div>
                                <p class="font-sm color-gray-500 mt-10">${product.description ? product.description.substring(0, 150) + '...' : ''}</p>
                                <div class="mt-20">
                                    <a class="btn btn-cart" href="#" onclick="addToCart(${product.id}); return false;">Add To Cart</a>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>`;
            } else {
                // Grid view template
                html += `
                <div class="col-xl-3 col-lg-4 col-md-6 col-sm-6 col-12">
                    <div class="card-grid-style-3">
                        <div class="card-grid-inner">
                            <div class="tools">
                                <a class="btn btn-wishlist btn-tooltip mb-10" href="shop-wishlist.html" aria-label="Add To Wishlist"></a>
                                <a class="btn btn-quickview btn-tooltip" aria-label="Quick view" href="#ModalQuickview" data-bs-toggle="modal" data-product-id="${product.id}"></a>
                            </div>
                            <div class="image-box">
                                ${discount > 0 ? `<span class="label bg-brand-2">-${discount}%</span>` : ''}
                                <a href="shop-single-product.html?slug=${product.slug}">
                                    <img src="${imageUrl}" alt="${product.name}">
                                </a>
                            </div>
                            <div class="info-right">
                                <a class="font-xs color-gray-500" href="shop-vendor-single.html?slug=${vendorSlug}">${vendorName}</a><br>
                                <a class="color-brand-3 font-sm-bold" href="shop-single-product.html?slug=${product.slug}">${product.name}</a>
                                <div class="rating">
                                    ${generateStars(rating)}
                                    <span class="font-xs color-gray-500">(${reviewCount})</span>
                                </div>
                                <div class="price-info">
                                    <strong class="font-lg-bold color-brand-3 price-main">$${parseFloat(displayPrice).toFixed(2)}</strong>
                                    ${product.sale_price && product.price > product.sale_price ? `<span class="color-gray-500 price-line">$${parseFloat(product.price).toFixed(2)}</span>` : ''}
                                </div>
                                <div class="mt-20 box-btn-cart">
                                    <a class="btn btn-cart" href="#" onclick="addToCart(${product.id}); return false;">Add To Cart</a>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>`;
            }
        });

        container.innerHTML = html;
    }

    function generateStars(rating) {
        let stars = '';
        for (let i = 0; i < 5; i++) {
            if (i < Math.floor(rating)) {
                stars += '<img src="assets/imgs/template/icons/star.svg" alt="star">';
            } else {
                stars += '<img src="assets/imgs/template/icons/star-gray.svg" alt="star" onerror="this.src=\'assets/imgs/template/icons/star.svg\'">';
            }
        }
        return stars;
    }

    function renderPagination(data) {
        const paginationContainer = document.querySelector('.pagination');
        if (!paginationContainer) return;

        const totalPages = data.last_page || 1;
        const currentPageNum = data.current_page || 1;

        let html = '';

        // Previous button
        html += `<li class="page-item ${currentPageNum <= 1 ? 'disabled' : ''}">
            <a class="page-link page-prev" href="#" data-page="${currentPageNum - 1}"></a>
        </li>`;

        // Page numbers
        let startPage = Math.max(1, currentPageNum - 2);
        let endPage = Math.min(totalPages, currentPageNum + 2);

        if (startPage > 1) {
            html += `<li class="page-item"><a class="page-link" href="#" data-page="1">1</a></li>`;
            if (startPage > 2) {
                html += `<li class="page-item disabled"><span class="page-link">...</span></li>`;
            }
        }

        for (let i = startPage; i <= endPage; i++) {
            html += `<li class="page-item">
                <a class="page-link ${i === currentPageNum ? 'active' : ''}" href="#" data-page="${i}">${i}</a>
            </li>`;
        }

        if (endPage < totalPages) {
            if (endPage < totalPages - 1) {
                html += `<li class="page-item disabled"><span class="page-link">...</span></li>`;
            }
            html += `<li class="page-item"><a class="page-link" href="#" data-page="${totalPages}">${totalPages}</a></li>`;
        }

        // Next button
        html += `<li class="page-item ${currentPageNum >= totalPages ? 'disabled' : ''}">
            <a class="page-link page-next" href="#" data-page="${currentPageNum + 1}"></a>
        </li>`;

        paginationContainer.innerHTML = html;

        // Add click handlers
        paginationContainer.querySelectorAll('.page-link[data-page]').forEach(link => {
            link.addEventListener('click', function(e) {
                e.preventDefault();
                const page = parseInt(this.dataset.page);
                if (page >= 1 && page <= totalPages) {
                    currentPage = page;
                    loadProducts();
                    window.scrollTo({ top: 0, behavior: 'smooth' });
                }
            });
        });
    }

    function updateProductCount(data) {
        const countEl = document.getElementById('product-count');
        if (countEl) {
            countEl.textContent = data.total || 0;
        }

        const showingEl = document.getElementById('showing-count');
        if (showingEl) {
            const from = ((data.current_page - 1) * data.per_page) + 1;
            const to = Math.min(data.current_page * data.per_page, data.total);
            showingEl.textContent = `Showing ${from}-${to} of ${data.total} products`;
        }
    }

    function loadCategories() {
        const container = document.getElementById('categories-list');
        if (!container) return;

        fetch(API_BASE + '/categories')
            .then(response => response.json())
            .then(data => {
                if (data.success && data.data) {
                    renderCategories(data.data, container);
                }
            })
            .catch(error => {
                console.error('Error loading categories:', error);
            });
    }

    function renderCategories(categories, container) {
        let html = '';
        categories.forEach(category => {
            const productCount = category.products_count || 0;
            const isActive = currentCategory == category.id ? 'active' : '';
            html += `<li>
                <a href="#" class="${isActive}" data-category="${category.id}">
                    ${category.name}<span class="number">${productCount}</span>
                </a>
            </li>`;
        });

        // Add "All Categories" option
        html = `<li><a href="#" class="${!currentCategory ? 'active' : ''}" data-category="">All Categories</a></li>` + html;

        container.innerHTML = html;

        // Add click handlers
        container.querySelectorAll('a[data-category]').forEach(link => {
            link.addEventListener('click', function(e) {
                e.preventDefault();
                currentCategory = this.dataset.category || null;
                currentPage = 1;
                loadProducts();

                // Update active state
                container.querySelectorAll('a').forEach(a => a.classList.remove('active'));
                this.classList.add('active');
            });
        });
    }

    function loadVendors() {
        const container = document.getElementById('vendors-list');
        if (!container) return;

        fetch(API_BASE + '/vendors?per_page=50')
            .then(response => response.json())
            .then(data => {
                if (data.success && data.data) {
                    const vendors = data.data.data || data.data;
                    renderVendors(vendors, container);
                }
            })
            .catch(error => {
                console.error('Error loading vendors:', error);
            });
    }

    function renderVendors(vendors, container) {
        if (!vendors || vendors.length === 0) {
            container.innerHTML = '<li class="text-muted">No brands available</li>';
            return;
        }

        let html = '';
        vendors.forEach(vendor => {
            const productCount = vendor.products_count || 0;
            const isActive = currentVendor == vendor.id ? 'checked' : '';
            html += `<li>
                <label class="cb-container">
                    <input type="checkbox" data-vendor="${vendor.id}" ${isActive}>
                    <span class="text-small">${vendor.name}</span>
                    <span class="checkmark"></span>
                </label>
                <span class="number-item">${productCount}</span>
            </li>`;
        });

        container.innerHTML = html;

        // Add click handlers
        container.querySelectorAll('input[data-vendor]').forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                // Uncheck other checkboxes for single selection
                if (this.checked) {
                    container.querySelectorAll('input[data-vendor]').forEach(cb => {
                        if (cb !== this) cb.checked = false;
                    });
                    currentVendor = this.dataset.vendor;
                } else {
                    currentVendor = null;
                }
                currentPage = 1;
                loadProducts();
            });
        });
    }

    function setupPriceFilter() {
        const priceCheckboxes = document.querySelectorAll('.price-filter-checkbox');
        const applyPriceBtn = document.getElementById('apply-price-filter');
        const minPriceInput = document.querySelector('.min-value');
        const maxPriceInput = document.querySelector('.max-value');

        // Setup price range checkboxes if they exist
        priceCheckboxes.forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                // Uncheck others
                priceCheckboxes.forEach(cb => {
                    if (cb !== this) cb.checked = false;
                });

                if (this.checked) {
                    currentMinPrice = this.dataset.min || null;
                    currentMaxPrice = this.dataset.max || null;
                } else {
                    currentMinPrice = null;
                    currentMaxPrice = null;
                }
                currentPage = 1;
                loadProducts();
            });
        });

        // Setup manual price filter button if exists
        if (applyPriceBtn && minPriceInput && maxPriceInput) {
            applyPriceBtn.addEventListener('click', function() {
                currentMinPrice = minPriceInput.value || null;
                currentMaxPrice = maxPriceInput.value || null;
                currentPage = 1;
                loadProducts();
            });
        }

        // Setup noUiSlider if available
        const sliderRange = document.getElementById('slider-range');
        if (sliderRange && typeof noUiSlider !== 'undefined' && !sliderRange.noUiSlider) {
            noUiSlider.create(sliderRange, {
                start: [0, 1000],
                connect: true,
                range: {
                    'min': 0,
                    'max': 1000
                },
                format: {
                    to: function(value) { return Math.round(value); },
                    from: function(value) { return Number(value); }
                }
            });

            sliderRange.noUiSlider.on('change', function(values) {
                currentMinPrice = values[0];
                currentMaxPrice = values[1];
                currentPage = 1;
                loadProducts();
            });

            sliderRange.noUiSlider.on('update', function(values) {
                const minDisplay = document.querySelector('.min-value-money');
                const maxDisplay = document.querySelector('.max-value-money');
                if (minDisplay) minDisplay.textContent = '$' + values[0];
                if (maxDisplay) maxDisplay.textContent = '$' + values[1];
            });
        }
    }

    function setupSearchForm() {
        const searchForm = document.querySelector('.form-search');
        const searchInput = document.querySelector('.form-search input[type="text"]');

        if (searchForm && searchInput) {
            searchForm.addEventListener('submit', function(e) {
                e.preventDefault();
                currentSearch = searchInput.value.trim();
                currentPage = 1;
                loadProducts();
            });
        }
    }

    function setupSortDropdown() {
        const sortDropdown = document.getElementById('dropdownSort');
        if (!sortDropdown) return;

        const sortItems = sortDropdown.closest('.dropdown').querySelectorAll('.dropdown-item');
        sortItems.forEach(item => {
            item.addEventListener('click', function(e) {
                e.preventDefault();
                const sortText = this.textContent.trim();

                // Update dropdown text
                sortDropdown.textContent = sortText;

                // Update active state
                sortItems.forEach(i => i.classList.remove('active'));
                this.classList.add('active');

                // Set sort parameters
                if (sortText.includes('Latest')) {
                    currentSort = 'created_at';
                    currentSortOrder = 'desc';
                } else if (sortText.includes('Oldest')) {
                    currentSort = 'created_at';
                    currentSortOrder = 'asc';
                } else if (sortText.includes('Price Low')) {
                    currentSort = 'price';
                    currentSortOrder = 'asc';
                } else if (sortText.includes('Price High')) {
                    currentSort = 'price';
                    currentSortOrder = 'desc';
                } else if (sortText.includes('Name')) {
                    currentSort = 'name';
                    currentSortOrder = 'asc';
                }

                currentPage = 1;
                loadProducts();
            });
        });
    }

    // Generate or get session ID for cart
    function getSessionId() {
        let sessionId = localStorage.getItem('cart_session_id');
        if (!sessionId) {
            sessionId = 'sess_' + Math.random().toString(36).substring(2) + Date.now().toString(36);
            localStorage.setItem('cart_session_id', sessionId);
        }
        return sessionId;
    }

    // Add to cart function (global)
    window.addToCart = function(productId) {
        const token = localStorage.getItem('kalses_token');
        const sessionId = getSessionId();

        const headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-Session-ID': sessionId
        };

        if (token) {
            headers['Authorization'] = 'Bearer ' + token;
        }

        fetch(API_BASE + '/cart/add', {
            method: 'POST',
            headers: headers,
            body: JSON.stringify({
                product_id: productId,
                quantity: 1,
                session_id: sessionId
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Product added to cart!');
                updateCartCount();
            } else {
                alert(data.message || 'Error adding to cart');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error adding to cart');
        });
    };

    function updateCartCount() {
        const token = localStorage.getItem('kalses_token');
        const sessionId = getSessionId();

        const headers = {
            'X-Session-ID': sessionId
        };

        if (token) {
            headers['Authorization'] = 'Bearer ' + token;
        }

        fetch(API_BASE + '/cart', {
            headers: headers
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                const count = data.data.items?.length || 0;
                document.querySelectorAll('.icon-cart .number-item').forEach(el => {
                    el.textContent = count;
                });
            }
        });
    }

    // Make functions available globally
    window.shopApi = {
        loadProducts: loadProducts,
        loadCategories: loadCategories,
        setCategory: function(categoryId) {
            currentCategory = categoryId;
            currentPage = 1;
            loadProducts();
        },
        setSearch: function(search) {
            currentSearch = search;
            currentPage = 1;
            loadProducts();
        }
    };
})();
