/**
 * Global Search Functionality
 * Handles header search form across all pages
 */
(function() {
    'use strict';

    const API_BASE = window.KALSES_API_BASE || 'http://127.0.0.1:8000/api/v1';

    document.addEventListener('DOMContentLoaded', function() {
        initSearchForm();
        loadSearchCategories();
        populateSearchFromUrl();
    });

    /**
     * Initialize the header search form
     */
    function initSearchForm() {
        const searchForm = document.getElementById('header-search-form');
        if (!searchForm) return;

        searchForm.addEventListener('submit', function(e) {
            e.preventDefault();

            const searchInput = document.getElementById('search-input');
            const categorySelect = document.getElementById('search-category');

            const searchQuery = searchInput ? searchInput.value.trim() : '';
            const category = categorySelect ? categorySelect.value : '';

            // Build the URL with query parameters
            const params = new URLSearchParams();
            if (searchQuery) params.append('search', searchQuery);
            if (category) params.append('category', category);

            // Redirect to shop page with search parameters
            window.location.href = 'shop-grid.html?' + params.toString();
        });

        // Add enter key handler for search input
        const searchInput = document.getElementById('search-input');
        if (searchInput) {
            searchInput.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    searchForm.dispatchEvent(new Event('submit'));
                }
            });
        }
    }

    /**
     * Load categories into the search dropdown
     */
    function loadSearchCategories() {
        const categorySelect = document.getElementById('search-category');
        if (!categorySelect) return;

        fetch(API_BASE + '/categories')
            .then(response => response.json())
            .then(data => {
                if (data.success && data.data) {
                    const categories = data.data;
                    let optionsHtml = '<option value="">All categories</option>';

                    categories.forEach(category => {
                        optionsHtml += `<option value="${category.id}">${category.name}</option>`;
                    });

                    categorySelect.innerHTML = optionsHtml;

                    // Re-apply the URL category if present
                    populateSearchFromUrl();
                }
            })
            .catch(error => {
                console.error('Error loading categories:', error);
            });
    }

    /**
     * Populate search form from URL parameters (for returning to search results)
     */
    function populateSearchFromUrl() {
        const urlParams = new URLSearchParams(window.location.search);

        const searchInput = document.getElementById('search-input');
        const categorySelect = document.getElementById('search-category');

        if (searchInput && urlParams.has('search')) {
            searchInput.value = urlParams.get('search');
        }

        if (categorySelect && urlParams.has('category')) {
            categorySelect.value = urlParams.get('category');
        }
    }

    // Expose functions globally if needed
    window.kalsesSearch = {
        search: function(query) {
            const params = new URLSearchParams();
            if (query) params.append('search', query);
            window.location.href = 'shop-grid.html?' + params.toString();
        }
    };
})();
