(function() {
    'use strict';

    const API_BASE = window.KALSES_API_BASE || 'http://127.0.0.1:8000/api/v1';

    document.addEventListener('DOMContentLoaded', function() {
        // Get product slug from URL
        const urlParams = new URLSearchParams(window.location.search);
        const productSlug = urlParams.get('slug');

        if (productSlug) {
            loadProduct(productSlug);
        } else {
            // Show message on page
            const nameEl = document.querySelector('h3.color-brand-3.mb-25');
            if (nameEl) {
                nameEl.textContent = 'Please select a product from the shop page';
            }
        }
    });

    function loadProduct(slug) {
        const apiUrl = API_BASE + '/products/' + slug;

        fetch(apiUrl)
            .then(response => response.json())
            .then(data => {
                if (data.success && data.data) {
                    renderProduct(data.data.product);
                    renderRelatedProducts(data.data.related_products);
                } else {
                    alert('Product not found: ' + slug);
                }
            })
            .catch(error => {
                alert('Error loading product: ' + error.message);
            });
    }

    function renderProduct(product) {
        // Update page title
        document.title = product.name + ' - Fareen Store';

        // Update product name
        const nameEl = document.getElementById('product-name');
        if (nameEl) nameEl.textContent = product.name;

        // Also update h3 product title
        document.querySelectorAll('h3.color-brand-3.mb-25').forEach(el => {
            el.textContent = product.name;
        });

        // Update vendor
        const vendorName = product.vendor?.name || 'Fareen Store';
        const vendorSlug = product.vendor?.slug || 'fareen-store';
        document.querySelectorAll('.byAUthor').forEach(el => {
            el.textContent = ' ' + vendorName;
            el.href = 'shop-vendor-single.html?slug=' + vendorSlug;
        });

        // Update price
        const displayPrice = product.sale_price || product.price;
        document.querySelectorAll('.price-main').forEach(el => {
            el.textContent = '$' + parseFloat(displayPrice).toFixed(2);
        });
        document.querySelectorAll('.price-line').forEach(el => {
            if (product.sale_price && product.price > product.sale_price) {
                el.textContent = '$' + parseFloat(product.price).toFixed(2);
                el.style.display = 'inline';
            } else {
                el.style.display = 'none';
            }
        });

        // Update discount label
        if (product.sale_price && product.price > product.sale_price) {
            const discount = Math.round((1 - product.sale_price / product.price) * 100);
            document.querySelectorAll('.detail-gallery .label').forEach(el => {
                el.textContent = '-' + discount + '%';
            });
        } else {
            document.querySelectorAll('.detail-gallery .label').forEach(el => {
                el.style.display = 'none';
            });
        }

        // Update rating
        const rating = product.average_rating || 5;
        const reviewCount = product.reviews_count || 0;
        document.querySelectorAll('.rating').forEach(el => {
            const starsHtml = generateStars(rating);
            const span = el.querySelector('span');
            if (span) {
                el.innerHTML = starsHtml + `<span class="font-xs color-gray-500 font-medium"> (${reviewCount} reviews)</span>`;
            }
        });

        // Update product images
        updateProductImages(product.images);

        // Update description
        if (product.description) {
            updateDescription(product.description);
        }

        // Update features
        if (product.features && product.features.length > 0) {
            updateFeatures(product.features);
        }

        // Update specifications
        if (product.specifications && product.specifications.length > 0) {
            updateSpecifications(product.specifications);
        }

        // Update reviews
        if (product.approved_reviews && product.approved_reviews.length > 0) {
            updateReviews(product.approved_reviews);
        }

        // Update breadcrumb
        document.querySelectorAll('.breadcrumb .active span').forEach(el => {
            el.textContent = product.name;
        });

        // Store product id for add to cart
        window.currentProductId = product.id;
    }

    function generateStars(rating) {
        let stars = '';
        for (let i = 0; i < 5; i++) {
            if (i < Math.floor(rating)) {
                stars += '<img src="assets/imgs/template/icons/star.svg" alt="star">';
            } else {
                stars += '<img src="assets/imgs/template/icons/star-gray.svg" alt="star" onerror="this.src=\'assets/imgs/template/icons/star.svg\'">';
            }
        }
        return stars;
    }

    function updateProductImages(images) {
        if (!images || images.length === 0) return;

        // Update main slider
        const mainSlider = document.querySelector('.product-image-slider');
        if (mainSlider) {
            let html = '';
            images.forEach(img => {
                const imageUrl = img.image_url || img.image;
                html += `<figure class="border-radius-10"><img src="${imageUrl}" alt="product image"></figure>`;
            });

            // If slick is initialized, destroy it first
            if ($(mainSlider).hasClass('slick-initialized')) {
                $(mainSlider).slick('unslick');
            }
            mainSlider.innerHTML = html;
        }

        // Update thumbnails
        const thumbSlider = document.querySelector('.slider-nav-thumbnails');
        if (thumbSlider) {
            let html = '';
            images.forEach(img => {
                const imageUrl = img.image_url || img.image;
                html += `<div><div class="item-thumb"><img src="${imageUrl}" alt="product image"></div></div>`;
            });

            if ($(thumbSlider).hasClass('slick-initialized')) {
                $(thumbSlider).slick('unslick');
            }
            thumbSlider.innerHTML = html;
        }

        // Reinitialize slick sliders
        setTimeout(function() {
            if (typeof $ !== 'undefined' && $.fn.slick) {
                $('.product-image-slider').slick({
                    slidesToShow: 1,
                    slidesToScroll: 1,
                    arrows: false,
                    fade: false,
                    asNavFor: '.slider-nav-thumbnails'
                });

                $('.slider-nav-thumbnails').slick({
                    slidesToShow: 5,
                    slidesToScroll: 1,
                    asNavFor: '.product-image-slider',
                    dots: false,
                    focusOnSelect: true,
                    vertical: true,
                    prevArrow: '<button type="button" class="slick-prev"><i class="fi-rs-arrow-small-left"></i></button>',
                    nextArrow: '<button type="button" class="slick-next"><i class="fi-rs-arrow-small-right"></i></button>',
                    responsive: [{
                        breakpoint: 600,
                        settings: {
                            slidesToShow: 3,
                            slidesToScroll: 1,
                            vertical: false
                        }
                    }]
                });
            }
        }, 100);
    }

    function updateDescription(description) {
        const descEl = document.querySelector('.product-description');
        if (descEl) {
            descEl.innerHTML = `<p>${description}</p>`;
        }
    }

    function updateFeatures(features) {
        const listDots = document.querySelectorAll('.list-dot');
        if (listDots.length > 0) {
            const half = Math.ceil(features.length / 2);
            const firstHalf = features.slice(0, half);
            const secondHalf = features.slice(half);

            if (listDots[0]) {
                listDots[0].innerHTML = firstHalf.map(f => `<li>${f.feature || f.value || f}</li>`).join('');
            }
            if (listDots[1]) {
                listDots[1].innerHTML = secondHalf.map(f => `<li>${f.feature || f.value || f}</li>`).join('');
            }
        }
    }

    function updateSpecifications(specifications) {
        const specTable = document.querySelector('.table-specifications tbody');
        if (specTable) {
            let html = '';
            specifications.forEach(spec => {
                html += `<tr>
                    <td class="col-4 font-md color-gray-900">${spec.name || spec.key}</td>
                    <td class="col-8 font-md color-gray-500">${spec.value}</td>
                </tr>`;
            });
            specTable.innerHTML = html;
        }
    }

    function updateReviews(reviews) {
        const reviewsContainer = document.querySelector('.list-reviews');
        if (!reviewsContainer) return;

        let html = '';
        reviews.forEach(review => {
            const customerName = review.customer?.name || 'Anonymous';
            const date = new Date(review.created_at).toLocaleDateString();
            const rating = review.rating || 5;

            html += `
            <div class="item-review">
                <div class="image-review">
                    <img src="assets/imgs/page/product/author.png" alt="${customerName}">
                </div>
                <div class="content-review">
                    <div class="rating mb-10">${generateStars(rating)}</div>
                    <p class="font-md color-gray-900 mb-5">${review.title || ''}</p>
                    <p class="font-sm color-gray-500 mb-10">${review.comment || ''}</p>
                    <p class="font-sm color-gray-500">by <strong>${customerName}</strong> on ${date}</p>
                </div>
            </div>`;
        });

        reviewsContainer.innerHTML = html;
    }

    function renderRelatedProducts(products) {
        if (!products || products.length === 0) return;

        const container = document.querySelector('.related-products .carausel-4-columns');
        if (!container) return;

        let html = '';
        products.forEach(product => {
            const imageUrl = product.primary_image?.image_url || 'assets/imgs/page/homepage1/imgsp1.png';
            const displayPrice = product.sale_price || product.price;

            html += `
            <div class="card-grid-style-3">
                <div class="card-grid-inner">
                    <div class="tools">
                        <a class="btn btn-wishlist btn-tooltip mb-10" href="shop-wishlist.html" aria-label="Add To Wishlist"></a>
                        <a class="btn btn-quickview btn-tooltip" aria-label="Quick view" href="#ModalQuickview" data-bs-toggle="modal"></a>
                    </div>
                    <div class="image-box">
                        <a href="shop-single-product.html?slug=${product.slug}">
                            <img src="${imageUrl}" alt="${product.name}">
                        </a>
                    </div>
                    <div class="info-right">
                        <a class="color-brand-3 font-sm-bold" href="shop-single-product.html?slug=${product.slug}">${product.name}</a>
                        <div class="price-info">
                            <strong class="font-lg-bold color-brand-3 price-main">$${parseFloat(displayPrice).toFixed(2)}</strong>
                        </div>
                        <div class="mt-10 box-btn-cart">
                            <a class="btn btn-cart" href="#" onclick="addToCart(${product.id}); return false;">Add To Cart</a>
                        </div>
                    </div>
                </div>
            </div>`;
        });

        // If slick is initialized, destroy it first
        if ($(container).hasClass('slick-initialized')) {
            $(container).slick('unslick');
        }
        container.innerHTML = html;
    }

    // Generate or get session ID for cart
    function getSessionId() {
        let sessionId = localStorage.getItem('cart_session_id');
        if (!sessionId) {
            sessionId = 'sess_' + Math.random().toString(36).substring(2) + Date.now().toString(36);
            localStorage.setItem('cart_session_id', sessionId);
        }
        return sessionId;
    }

    // Add to cart function for single product page
    window.addToCartSingle = function() {
        const productId = window.currentProductId;
        if (!productId) {
            alert('Product not loaded');
            return;
        }

        const quantity = document.querySelector('.input-quantity input')?.value || 1;
        const token = localStorage.getItem('kalses_token');
        const sessionId = getSessionId();

        const headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-Session-ID': sessionId
        };

        if (token) {
            headers['Authorization'] = 'Bearer ' + token;
        }

        fetch(API_BASE + '/cart/add', {
            method: 'POST',
            headers: headers,
            body: JSON.stringify({
                product_id: productId,
                quantity: parseInt(quantity),
                session_id: sessionId
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Product added to cart!');
                updateCartCount();
            } else {
                alert(data.message || 'Error adding to cart');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error adding to cart');
        });
    };

    // Add to cart function for related products
    window.addToCart = function(productId) {
        const token = localStorage.getItem('kalses_token');
        const sessionId = getSessionId();

        const headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-Session-ID': sessionId
        };

        if (token) {
            headers['Authorization'] = 'Bearer ' + token;
        }

        fetch(API_BASE + '/cart/add', {
            method: 'POST',
            headers: headers,
            body: JSON.stringify({
                product_id: productId,
                quantity: 1,
                session_id: sessionId
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Product added to cart!');
                updateCartCount();
            } else {
                alert(data.message || 'Error adding to cart');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error adding to cart');
        });
    };

    function updateCartCount() {
        const token = localStorage.getItem('kalses_token');
        const sessionId = getSessionId();

        const headers = {
            'X-Session-ID': sessionId
        };

        if (token) {
            headers['Authorization'] = 'Bearer ' + token;
        }

        fetch(API_BASE + '/cart', {
            headers: headers
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                const count = data.data.items?.length || 0;
                document.querySelectorAll('.icon-cart .number-item').forEach(el => {
                    el.textContent = count;
                });
            }
        });
    }
})();
