/**
 * Kalses E-Commerce Frontend Integration
 * Handles UI interactions and API calls for the storefront
 */

document.addEventListener('DOMContentLoaded', function() {
    initHeader();
    initProducts();
    initCart();
    initWishlist();
    initCompare();
    initAuth();
    initSearch();
    initCheckout();
});

// ==================== HEADER ====================
function initHeader() {
    updateCartCount();
    updateWishlistCount();
    updateAuthUI();
}

async function updateCartCount() {
    try {
        const response = await KalsesAPI.cart.get();
        if (response.success) {
            const count = response.data.total_items || 0;
            document.querySelectorAll('.cart-count, .pro-count').forEach(el => {
                el.textContent = count;
            });
        }
    } catch (error) {
        console.error('Failed to update cart count:', error);
    }
}

async function updateWishlistCount() {
    if (!KalsesAPI.isLoggedIn()) return;

    try {
        const response = await KalsesAPI.wishlist.get();
        if (response.success) {
            const count = response.data.length || 0;
            document.querySelectorAll('.wishlist-count').forEach(el => {
                el.textContent = count;
            });
        }
    } catch (error) {
        console.error('Failed to update wishlist count:', error);
    }
}

function updateAuthUI() {
    const isLoggedIn = KalsesAPI.isLoggedIn();
    const customer = KalsesAPI.auth.getCustomer();

    document.querySelectorAll('.auth-logged-in').forEach(el => {
        el.style.display = isLoggedIn ? '' : 'none';
    });

    document.querySelectorAll('.auth-logged-out').forEach(el => {
        el.style.display = isLoggedIn ? 'none' : '';
    });

    if (isLoggedIn && customer) {
        document.querySelectorAll('.customer-name').forEach(el => {
            el.textContent = customer.name;
        });
    }
}

// ==================== PRODUCTS ====================
function initProducts() {
    // Add to cart buttons
    $(document).on('click', '[data-add-to-cart]', async function(e) {
        e.preventDefault();
        const productId = $(this).data('add-to-cart');
        const quantity = parseInt($(this).data('quantity') || 1);
        await addToCart(productId, quantity);
    });

    // Add to wishlist buttons
    $(document).on('click', '[data-add-to-wishlist]', async function(e) {
        e.preventDefault();
        const productId = $(this).data('add-to-wishlist');
        await addToWishlist(productId);
    });

    // Add to compare buttons
    $(document).on('click', '[data-add-to-compare]', async function(e) {
        e.preventDefault();
        const productId = $(this).data('add-to-compare');
        await addToCompare(productId);
    });

    // Load dynamic content
    loadFeaturedProducts();
    loadNewArrivals();
    loadBestSellers();
    loadCategories();
}

async function addToCart(productId, quantity = 1) {
    try {
        showLoading();
        const response = await KalsesAPI.cart.add(productId, quantity);
        if (response.success) {
            showToast('Product added to cart!', 'success');
            updateCartCount();
            updateMiniCart();
        } else {
            showToast(response.message || 'Failed to add to cart', 'error');
        }
    } catch (error) {
        showToast(error.data?.message || 'Failed to add to cart', 'error');
    } finally {
        hideLoading();
    }
}

async function addToWishlist(productId) {
    if (!KalsesAPI.isLoggedIn()) {
        showToast('Please login to add to wishlist', 'warning');
        window.location.href = 'page-login.html';
        return;
    }

    try {
        showLoading();
        const response = await KalsesAPI.wishlist.add(productId);
        if (response.success) {
            showToast('Added to wishlist!', 'success');
            updateWishlistCount();
        } else {
            showToast(response.message || 'Failed to add to wishlist', 'error');
        }
    } catch (error) {
        showToast(error.data?.message || 'Failed to add to wishlist', 'error');
    } finally {
        hideLoading();
    }
}

async function addToCompare(productId) {
    try {
        showLoading();
        const response = await KalsesAPI.compare.add(productId);
        if (response.success) {
            showToast('Added to compare!', 'success');
        } else {
            showToast(response.message || 'Maximum 4 products can be compared', 'error');
        }
    } catch (error) {
        showToast(error.data?.message || 'Failed to add to compare', 'error');
    } finally {
        hideLoading();
    }
}

async function loadFeaturedProducts() {
    const container = document.querySelector('[data-featured-products]');
    if (!container) return;

    try {
        const response = await KalsesAPI.products.getFeatured(8);
        if (response.success && response.data.length > 0) {
            renderProducts(container, response.data);
        }
    } catch (error) {
        console.error('Failed to load featured products:', error);
    }
}

async function loadNewArrivals() {
    const container = document.querySelector('[data-new-arrivals]');
    if (!container) return;

    try {
        const response = await KalsesAPI.products.getNewArrivals(8);
        if (response.success && response.data.length > 0) {
            renderProducts(container, response.data);
        }
    } catch (error) {
        console.error('Failed to load new arrivals:', error);
    }
}

async function loadBestSellers() {
    const container = document.querySelector('[data-best-sellers]');
    if (!container) return;

    try {
        const response = await KalsesAPI.products.getBestSellers(8);
        if (response.success && response.data.length > 0) {
            renderProducts(container, response.data);
        }
    } catch (error) {
        console.error('Failed to load best sellers:', error);
    }
}

async function loadCategories() {
    const container = document.querySelector('[data-categories]');
    if (!container) return;

    try {
        const response = await KalsesAPI.categories.getAll();
        if (response.success) {
            renderCategories(container, response.data);
        }
    } catch (error) {
        console.error('Failed to load categories:', error);
    }
}

function renderProducts(container, products) {
    const html = products.map(product => {
        const imageUrl = product.primary_image?.url || 'assets/imgs/shop/product-placeholder.png';
        const price = product.sale_price || product.price;
        const hasDiscount = product.sale_price && product.sale_price < product.price;

        return `
            <div class="product-cart-wrap">
                <div class="product-img-action-wrap">
                    <div class="product-img product-img-zoom">
                        <a href="shop-single-product.html?slug=${product.slug}">
                            <img class="default-img" src="${imageUrl}" alt="${product.name}">
                        </a>
                    </div>
                    <div class="product-action-1">
                        <a aria-label="Add To Wishlist" class="action-btn" href="#" data-add-to-wishlist="${product.id}">
                            <i class="fi-rs-heart"></i>
                        </a>
                        <a aria-label="Compare" class="action-btn" href="#" data-add-to-compare="${product.id}">
                            <i class="fi-rs-shuffle"></i>
                        </a>
                    </div>
                    ${hasDiscount ? `<div class="product-badges product-badges-position product-badges-mrg"><span class="sale">Sale</span></div>` : ''}
                </div>
                <div class="product-content-wrap">
                    <div class="product-category">
                        <a href="shop-grid.html?category=${product.category?.slug || ''}">${product.category?.name || ''}</a>
                    </div>
                    <h2><a href="shop-single-product.html?slug=${product.slug}">${product.name}</a></h2>
                    <div class="product-price">
                        <span>$${parseFloat(price).toFixed(2)}</span>
                        ${hasDiscount ? `<span class="old-price">$${parseFloat(product.price).toFixed(2)}</span>` : ''}
                    </div>
                    <div class="add-cart">
                        <a class="add" href="#" data-add-to-cart="${product.id}">
                            <i class="fi-rs-shopping-cart mr-5"></i>Add
                        </a>
                    </div>
                </div>
            </div>
        `;
    }).join('');

    container.innerHTML = html;
}

function renderCategories(container, categories) {
    const html = categories.map(category => `
        <li>
            <a href="shop-grid.html?category=${category.slug}">
                ${category.icon ? `<img src="${category.icon}" alt="">` : ''}
                ${category.name}
            </a>
            ${category.children && category.children.length > 0 ? `
                <ul class="sub-menu">
                    ${category.children.map(child => `
                        <li><a href="shop-grid.html?category=${child.slug}">${child.name}</a></li>
                    `).join('')}
                </ul>
            ` : ''}
        </li>
    `).join('');

    container.innerHTML = html;
}

// ==================== CART ====================
function initCart() {
    // Update quantity
    $(document).on('click', '.qty-up, .qty-down', async function() {
        const $row = $(this).closest('tr');
        const itemId = $row.data('item-id');
        if (!itemId) return;

        const $input = $row.find('.qty-val');
        let quantity = parseInt($input.val()) || 1;

        if ($(this).hasClass('qty-up')) {
            quantity++;
        } else if (quantity > 1) {
            quantity--;
        }

        $input.val(quantity);
        await updateCartItem(itemId, quantity);
    });

    // Remove item
    $(document).on('click', '[data-cart-remove]', async function(e) {
        e.preventDefault();
        const itemId = $(this).data('cart-remove');
        await removeCartItem(itemId);
    });

    // Apply coupon
    $(document).on('click', '[data-apply-coupon]', async function(e) {
        e.preventDefault();
        const code = $('[data-coupon-code]').val();
        if (code) {
            await applyCoupon(code);
        }
    });

    // Load cart page
    if ($('[data-cart-page]').length) {
        loadCartPage();
    }
}

async function loadCartPage() {
    try {
        showLoading();
        const response = await KalsesAPI.cart.get();
        if (response.success) {
            renderCartPage(response.data);
        }
    } catch (error) {
        console.error('Failed to load cart:', error);
    } finally {
        hideLoading();
    }
}

async function updateCartItem(itemId, quantity) {
    try {
        const response = await KalsesAPI.cart.update(itemId, quantity);
        if (response.success) {
            updateCartTotals(response.data);
            updateCartCount();
        }
    } catch (error) {
        showToast(error.data?.message || 'Failed to update cart', 'error');
    }
}

async function removeCartItem(itemId) {
    try {
        const response = await KalsesAPI.cart.remove(itemId);
        if (response.success) {
            showToast('Item removed', 'success');
            loadCartPage();
            updateCartCount();
        }
    } catch (error) {
        showToast('Failed to remove item', 'error');
    }
}

async function applyCoupon(code) {
    try {
        const response = await KalsesAPI.cart.applyCoupon(code);
        if (response.success) {
            showToast('Coupon applied!', 'success');
            sessionStorage.setItem('applied_coupon', code);
            updateCartTotals(response.data);
        }
    } catch (error) {
        showToast(error.data?.message || 'Invalid coupon', 'error');
    }
}

function renderCartPage(data) {
    const items = data.cart?.items || [];
    const container = $('[data-cart-items]');

    if (items.length === 0) {
        container.html('<tr><td colspan="6" class="text-center py-5">Your cart is empty. <a href="shop-grid.html">Continue shopping</a></td></tr>');
        return;
    }

    const html = items.map(item => {
        const product = item.product || {};
        const imageUrl = product.primary_image?.url || 'assets/imgs/shop/product-placeholder.png';

        return `
            <tr data-item-id="${item.id}">
                <td class="image product-thumbnail">
                    <img src="${imageUrl}" alt="${product.name}">
                </td>
                <td class="product-des product-name">
                    <h6><a href="shop-single-product.html?slug=${product.slug}">${product.name}</a></h6>
                </td>
                <td class="price" data-title="Price">
                    <span>$${parseFloat(item.price).toFixed(2)}</span>
                </td>
                <td class="text-center detail-info" data-title="Qty">
                    <div class="detail-extralink mr-15">
                        <div class="detail-qty border radius">
                            <a href="#" class="qty-down"><i class="fi-rs-angle-small-down"></i></a>
                            <input type="text" class="qty-val" value="${item.quantity}" min="1">
                            <a href="#" class="qty-up"><i class="fi-rs-angle-small-up"></i></a>
                        </div>
                    </div>
                </td>
                <td class="price" data-title="Total">
                    <span>$${(item.price * item.quantity).toFixed(2)}</span>
                </td>
                <td class="action text-center" data-title="Remove">
                    <a href="#" class="text-body" data-cart-remove="${item.id}"><i class="fi-rs-trash"></i></a>
                </td>
            </tr>
        `;
    }).join('');

    container.html(html);
    updateCartTotals(data);
}

function updateCartTotals(data) {
    $('[data-cart-subtotal]').text('$' + parseFloat(data.subtotal || 0).toFixed(2));
    $('[data-cart-total]').text('$' + parseFloat(data.total || data.subtotal || 0).toFixed(2));

    if (data.discount && data.discount > 0) {
        $('[data-cart-discount-row]').show();
        $('[data-cart-discount]').text('-$' + parseFloat(data.discount).toFixed(2));
    } else {
        $('[data-cart-discount-row]').hide();
    }
}

async function updateMiniCart() {
    // Update mini cart dropdown in header
    try {
        const response = await KalsesAPI.cart.get();
        if (response.success) {
            const items = response.data.cart?.items || [];
            const $miniCart = $('.dropdown-cart ul');

            if ($miniCart.length && items.length > 0) {
                const html = items.slice(0, 3).map(item => {
                    const product = item.product || {};
                    return `
                        <li>
                            <div class="shopping-cart-img">
                                <a href="shop-single-product.html?slug=${product.slug}">
                                    <img src="${product.primary_image?.url || 'assets/imgs/shop/product-placeholder.png'}" alt="">
                                </a>
                            </div>
                            <div class="shopping-cart-title">
                                <h4><a href="shop-single-product.html?slug=${product.slug}">${product.name}</a></h4>
                                <h4><span>${item.quantity} x </span>$${parseFloat(item.price).toFixed(2)}</h4>
                            </div>
                        </li>
                    `;
                }).join('');

                $miniCart.html(html);
            }
        }
    } catch (error) {
        console.error('Failed to update mini cart:', error);
    }
}

// ==================== WISHLIST ====================
function initWishlist() {
    if ($('[data-wishlist-page]').length) {
        loadWishlistPage();
    }

    $(document).on('click', '[data-wishlist-remove]', async function(e) {
        e.preventDefault();
        const productId = $(this).data('wishlist-remove');
        await removeFromWishlist(productId);
    });
}

async function loadWishlistPage() {
    if (!KalsesAPI.isLoggedIn()) {
        window.location.href = 'page-login.html';
        return;
    }

    try {
        showLoading();
        const response = await KalsesAPI.wishlist.get();
        if (response.success) {
            renderWishlistPage(response.data);
        }
    } catch (error) {
        console.error('Failed to load wishlist:', error);
    } finally {
        hideLoading();
    }
}

async function removeFromWishlist(productId) {
    try {
        await KalsesAPI.wishlist.remove(productId);
        showToast('Removed from wishlist', 'success');
        loadWishlistPage();
        updateWishlistCount();
    } catch (error) {
        showToast('Failed to remove', 'error');
    }
}

function renderWishlistPage(items) {
    const container = $('[data-wishlist-items]');

    if (items.length === 0) {
        container.html('<tr><td colspan="5" class="text-center py-5">Your wishlist is empty. <a href="shop-grid.html">Browse products</a></td></tr>');
        return;
    }

    const html = items.map(item => {
        const product = item.product || {};
        const imageUrl = product.primary_image?.url || 'assets/imgs/shop/product-placeholder.png';
        const price = product.sale_price || product.price;

        return `
            <tr>
                <td class="image product-thumbnail">
                    <img src="${imageUrl}" alt="${product.name}">
                </td>
                <td class="product-des product-name">
                    <h6><a href="shop-single-product.html?slug=${product.slug}">${product.name}</a></h6>
                </td>
                <td class="price" data-title="Price">
                    <span>$${parseFloat(price).toFixed(2)}</span>
                </td>
                <td class="text-center" data-title="Stock">
                    <span class="${product.stock > 0 ? 'text-success' : 'text-danger'}">
                        ${product.stock > 0 ? 'In Stock' : 'Out of Stock'}
                    </span>
                </td>
                <td class="action text-right" data-title="Action">
                    <a href="#" class="btn btn-sm" data-add-to-cart="${product.id}"><i class="fi-rs-shopping-cart mr-5"></i>Add</a>
                    <a href="#" class="text-danger ml-10" data-wishlist-remove="${product.id}"><i class="fi-rs-trash"></i></a>
                </td>
            </tr>
        `;
    }).join('');

    container.html(html);
}

// ==================== COMPARE ====================
function initCompare() {
    if ($('[data-compare-page]').length) {
        loadComparePage();
    }

    $(document).on('click', '[data-compare-remove]', async function(e) {
        e.preventDefault();
        const productId = $(this).data('compare-remove');
        await removeFromCompare(productId);
    });
}

async function loadComparePage() {
    try {
        showLoading();
        const response = await KalsesAPI.compare.get();
        if (response.success) {
            renderComparePage(response.data);
        }
    } catch (error) {
        console.error('Failed to load compare:', error);
    } finally {
        hideLoading();
    }
}

async function removeFromCompare(productId) {
    try {
        await KalsesAPI.compare.remove(productId);
        showToast('Removed from compare', 'success');
        loadComparePage();
    } catch (error) {
        showToast('Failed to remove', 'error');
    }
}

function renderComparePage(items) {
    const container = $('[data-compare-table]');

    if (items.length === 0) {
        container.html('<p class="text-center py-5">No products to compare. <a href="shop-grid.html">Browse products</a></p>');
        return;
    }

    // Build compare table dynamically
    let html = '<table class="table table-bordered">';

    // Product images
    html += '<tr><th>Product</th>';
    items.forEach(item => {
        const product = item.product || {};
        html += `<td class="text-center">
            <img src="${product.primary_image?.url || 'assets/imgs/shop/product-placeholder.png'}" alt="" style="max-width: 150px;">
            <h6 class="mt-2">${product.name}</h6>
        </td>`;
    });
    html += '</tr>';

    // Price
    html += '<tr><th>Price</th>';
    items.forEach(item => {
        const product = item.product || {};
        const price = product.sale_price || product.price;
        html += `<td class="text-center">$${parseFloat(price).toFixed(2)}</td>`;
    });
    html += '</tr>';

    // Stock
    html += '<tr><th>Availability</th>';
    items.forEach(item => {
        const product = item.product || {};
        html += `<td class="text-center ${product.stock > 0 ? 'text-success' : 'text-danger'}">
            ${product.stock > 0 ? 'In Stock' : 'Out of Stock'}
        </td>`;
    });
    html += '</tr>';

    // Actions
    html += '<tr><th>Action</th>';
    items.forEach(item => {
        const product = item.product || {};
        html += `<td class="text-center">
            <a href="#" class="btn btn-sm btn-primary" data-add-to-cart="${product.id}">Add to Cart</a>
            <a href="#" class="btn btn-sm btn-outline-danger ml-2" data-compare-remove="${product.id}">Remove</a>
        </td>`;
    });
    html += '</tr>';

    html += '</table>';
    container.html(html);
}

// ==================== AUTH ====================
function initAuth() {
    // Login form
    $('[data-login-form]').on('submit', async function(e) {
        e.preventDefault();
        const email = $(this).find('[name="email"]').val();
        const password = $(this).find('[name="password"]').val();

        try {
            showLoading();
            const response = await KalsesAPI.auth.login(email, password);
            if (response.success) {
                showToast('Login successful!', 'success');
                setTimeout(() => {
                    window.location.href = 'page-account.html';
                }, 1000);
            }
        } catch (error) {
            showToast(error.data?.message || 'Login failed', 'error');
        } finally {
            hideLoading();
        }
    });

    // Register form
    $('[data-register-form]').on('submit', async function(e) {
        e.preventDefault();
        const data = {
            name: $(this).find('[name="name"]').val(),
            email: $(this).find('[name="email"]').val(),
            password: $(this).find('[name="password"]').val(),
            password_confirmation: $(this).find('[name="password_confirmation"]').val(),
            phone: $(this).find('[name="phone"]').val(),
        };

        try {
            showLoading();
            const response = await KalsesAPI.auth.register(data);
            if (response.success) {
                showToast('Registration successful!', 'success');
                setTimeout(() => {
                    window.location.href = 'page-account.html';
                }, 1000);
            }
        } catch (error) {
            const errors = error.data?.errors;
            if (errors) {
                const messages = Object.values(errors).flat().join('<br>');
                showToast(messages, 'error');
            } else {
                showToast(error.data?.message || 'Registration failed', 'error');
            }
        } finally {
            hideLoading();
        }
    });

    // Logout
    $(document).on('click', '[data-logout]', async function(e) {
        e.preventDefault();
        try {
            await KalsesAPI.auth.logout();
            showToast('Logged out', 'success');
        } finally {
            window.location.href = 'index.html';
        }
    });

    // Account page
    if ($('[data-account-page]').length) {
        loadAccountPage();
    }
}

async function loadAccountPage() {
    if (!KalsesAPI.isLoggedIn()) {
        window.location.href = 'page-login.html';
        return;
    }

    try {
        const response = await KalsesAPI.auth.getProfile();
        if (response.success) {
            renderAccountPage(response.data);
        }
    } catch (error) {
        console.error('Failed to load account:', error);
    }
}

function renderAccountPage(customer) {
    $('[data-customer-name]').text(customer.name);
    $('[data-customer-email]').text(customer.email);
    $('[data-customer-phone]').text(customer.phone || 'Not set');

    // Render recent orders
    if (customer.orders && customer.orders.length > 0) {
        const ordersHtml = customer.orders.map(order => `
            <tr>
                <td>${order.order_number}</td>
                <td>${new Date(order.created_at).toLocaleDateString()}</td>
                <td><span class="badge bg-${getStatusColor(order.status)}">${order.status}</span></td>
                <td>$${parseFloat(order.total).toFixed(2)}</td>
                <td><a href="page-order.html?id=${order.id}">View</a></td>
            </tr>
        `).join('');
        $('[data-recent-orders]').html(ordersHtml);
    }
}

function getStatusColor(status) {
    const colors = {
        pending: 'warning',
        processing: 'info',
        shipped: 'primary',
        delivered: 'success',
        cancelled: 'danger',
    };
    return colors[status] || 'secondary';
}

// ==================== SEARCH ====================
function initSearch() {
    $('[data-search-form]').on('submit', function(e) {
        e.preventDefault();
        const query = $(this).find('[name="search"]').val();
        if (query) {
            window.location.href = 'shop-grid.html?search=' + encodeURIComponent(query);
        }
    });

    // Parse search params on shop page
    if ($('[data-products-grid]').length) {
        loadProductsGrid();
    }
}

async function loadProductsGrid() {
    const urlParams = new URLSearchParams(window.location.search);
    const params = {};

    if (urlParams.get('search')) params.search = urlParams.get('search');
    if (urlParams.get('category')) params.category = urlParams.get('category');
    if (urlParams.get('min_price')) params.min_price = urlParams.get('min_price');
    if (urlParams.get('max_price')) params.max_price = urlParams.get('max_price');
    if (urlParams.get('sort')) params.sort_by = urlParams.get('sort');

    try {
        showLoading();
        const response = await KalsesAPI.products.getAll(params);
        if (response.success) {
            renderProducts($('[data-products-grid]')[0], response.data.data || response.data);
        }
    } catch (error) {
        console.error('Failed to load products:', error);
    } finally {
        hideLoading();
    }
}

// ==================== CHECKOUT ====================
function initCheckout() {
    if ($('[data-checkout-page]').length) {
        loadCheckoutPage();
    }

    $('[data-checkout-form]').on('submit', async function(e) {
        e.preventDefault();
        await processCheckout();
    });
}

async function loadCheckoutPage() {
    try {
        showLoading();
        const couponCode = sessionStorage.getItem('applied_coupon');
        const response = await KalsesAPI.checkout.getSummary(couponCode);
        if (response.success) {
            renderCheckoutSummary(response.data);
        }
    } catch (error) {
        console.error('Failed to load checkout:', error);
    } finally {
        hideLoading();
    }
}

function renderCheckoutSummary(data) {
    $('[data-checkout-subtotal]').text('$' + parseFloat(data.subtotal).toFixed(2));
    $('[data-checkout-shipping]').text('$' + parseFloat(data.shipping).toFixed(2));
    $('[data-checkout-tax]').text('$' + parseFloat(data.tax).toFixed(2));
    $('[data-checkout-total]').text('$' + parseFloat(data.total).toFixed(2));

    if (data.discount > 0) {
        $('[data-checkout-discount-row]').show();
        $('[data-checkout-discount]').text('-$' + parseFloat(data.discount).toFixed(2));
    }
}

async function processCheckout() {
    const orderData = {
        shipping_address: {
            name: $('[name="shipping_name"]').val(),
            phone: $('[name="shipping_phone"]').val(),
            email: $('[name="shipping_email"]').val(),
            address_line_1: $('[name="shipping_address"]').val(),
            city: $('[name="shipping_city"]').val(),
            state: $('[name="shipping_state"]').val(),
            postal_code: $('[name="shipping_zip"]').val(),
            country: $('[name="shipping_country"]').val() || 'USA',
        },
        payment_method: $('[name="payment_method"]:checked').val() || 'cod',
        coupon_code: sessionStorage.getItem('applied_coupon'),
        notes: $('[name="order_notes"]').val(),
    };

    try {
        showLoading();
        const response = await KalsesAPI.checkout.process(orderData);
        if (response.success) {
            sessionStorage.removeItem('applied_coupon');

            if (response.data.checkout_url) {
                // Redirect to Stripe
                window.location.href = response.data.checkout_url;
            } else {
                // Order placed successfully
                showToast('Order placed successfully!', 'success');
                setTimeout(() => {
                    window.location.href = 'page-order-success.html?order=' + response.data.order.order_number;
                }, 1500);
            }
        }
    } catch (error) {
        showToast(error.data?.message || 'Failed to process order', 'error');
    } finally {
        hideLoading();
    }
}

// ==================== UTILITIES ====================
function showToast(message, type = 'info') {
    const toastHtml = `
        <div class="kalses-toast kalses-toast-${type}">
            <span>${message}</span>
            <button class="kalses-toast-close">&times;</button>
        </div>
    `;

    // Add styles if needed
    if (!$('#kalses-toast-styles').length) {
        $('head').append(`
            <style id="kalses-toast-styles">
                .kalses-toast { position: fixed; top: 20px; right: 20px; padding: 15px 20px; border-radius: 5px; z-index: 10000; animation: slideInRight 0.3s ease; display: flex; align-items: center; gap: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); }
                .kalses-toast-success { background: #28a745; color: white; }
                .kalses-toast-error { background: #dc3545; color: white; }
                .kalses-toast-warning { background: #ffc107; color: #333; }
                .kalses-toast-info { background: #17a2b8; color: white; }
                .kalses-toast-close { background: none; border: none; color: inherit; font-size: 20px; cursor: pointer; opacity: 0.7; }
                .kalses-toast-close:hover { opacity: 1; }
                @keyframes slideInRight { from { transform: translateX(100%); opacity: 0; } to { transform: translateX(0); opacity: 1; } }
            </style>
        `);
    }

    const $toast = $(toastHtml).appendTo('body');
    $toast.find('.kalses-toast-close').on('click', () => $toast.remove());
    setTimeout(() => $toast.remove(), 5000);
}

function showLoading() {
    if ($('.kalses-loader').length) return;

    $('body').append(`
        <div class="kalses-loader">
            <div class="kalses-spinner"></div>
        </div>
    `);

    if (!$('#kalses-loader-styles').length) {
        $('head').append(`
            <style id="kalses-loader-styles">
                .kalses-loader { position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(255,255,255,0.8); display: flex; align-items: center; justify-content: center; z-index: 10001; }
                .kalses-spinner { width: 40px; height: 40px; border: 4px solid #f3f3f3; border-top: 4px solid #3bb77e; border-radius: 50%; animation: spin 1s linear infinite; }
                @keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }
            </style>
        `);
    }
}

function hideLoading() {
    $('.kalses-loader').remove();
}

// Export for use in other scripts
window.KalsesShop = {
    addToCart,
    addToWishlist,
    addToCompare,
    updateCartCount,
    showToast,
};
