(function() {
    'use strict';

    const API_BASE = window.KALSES_API_BASE || 'http://127.0.0.1:8000/api/v1';

    function getSessionId() {
        let sessionId = localStorage.getItem('kalses_session_id');
        if (!sessionId) {
            sessionId = 'sess_' + Math.random().toString(36).substr(2, 9) + Date.now();
            localStorage.setItem('kalses_session_id', sessionId);
        }
        return sessionId;
    }

    function getHeaders() {
        const token = localStorage.getItem('kalses_token');
        const headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-Session-ID': getSessionId()
        };
        if (token) {
            headers['Authorization'] = 'Bearer ' + token;
        }
        return headers;
    }

    document.addEventListener('DOMContentLoaded', function() {
        loadCompare();
    });

    function loadCompare() {
        fetch(API_BASE + '/compare', {
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                const items = data.data.items || data.data;
                renderCompare(items);
            } else {
                renderEmptyCompare();
            }
        })
        .catch(error => {
            console.error('Error loading compare:', error);
            renderEmptyCompare();
        });
    }

    function renderCompare(items) {
        const container = document.querySelector('.table-compare tbody');
        if (!container) return;

        if (!items || items.length === 0) {
            renderEmptyCompare();
            return;
        }

        // Get products from compare items
        const products = items.map(item => item.product || item);

        // Build comparison table
        let html = '';

        // Product images row
        html += '<tr><th class="text-lg-start text-center">Preview</th>';
        products.forEach(product => {
            const imageUrl = product.primary_image?.image_url || 'assets/imgs/page/product/img-sub.png';
            html += `<td class="text-center">
                <a href="shop-single-product.html?slug=${product.slug}">
                    <img src="${imageUrl}" alt="${product.name}" style="max-width: 150px;">
                </a>
            </td>`;
        });
        html += '</tr>';

        // Product name row
        html += '<tr><th class="text-lg-start text-center">Name</th>';
        products.forEach(product => {
            html += `<td class="text-center">
                <a href="shop-single-product.html?slug=${product.slug}" class="color-brand-3 font-md-bold">${product.name}</a>
            </td>`;
        });
        html += '</tr>';

        // Price row
        html += '<tr><th class="text-lg-start text-center">Price</th>';
        products.forEach(product => {
            const price = product.sale_price || product.price;
            html += `<td class="text-center">
                <span class="font-lg-bold color-brand-3">$${parseFloat(price).toFixed(2)}</span>
                ${product.sale_price && product.price > product.sale_price ?
                    `<br><span class="font-sm color-gray-500 text-decoration-line-through">$${parseFloat(product.price).toFixed(2)}</span>` : ''}
            </td>`;
        });
        html += '</tr>';

        // Rating row
        html += '<tr><th class="text-lg-start text-center">Rating</th>';
        products.forEach(product => {
            html += `<td class="text-center">
                <div class="rating d-inline-block">${generateStars(product.average_rating || 0)}</div>
                <span class="font-xs color-gray-500"> (${product.reviews_count || 0})</span>
            </td>`;
        });
        html += '</tr>';

        // Stock status row
        html += '<tr><th class="text-lg-start text-center">Availability</th>';
        products.forEach(product => {
            const inStock = product.stock_quantity > 0;
            html += `<td class="text-center">
                <span class="${inStock ? 'color-success' : 'color-danger'} font-sm-bold">${inStock ? 'In Stock' : 'Out of Stock'}</span>
            </td>`;
        });
        html += '</tr>';

        // SKU row
        html += '<tr><th class="text-lg-start text-center">SKU</th>';
        products.forEach(product => {
            html += `<td class="text-center"><span class="font-sm color-gray-500">${product.sku || 'N/A'}</span></td>`;
        });
        html += '</tr>';

        // Description row
        html += '<tr><th class="text-lg-start text-center">Description</th>';
        products.forEach(product => {
            const desc = product.short_description || product.description || '';
            html += `<td class="text-center"><p class="font-sm color-gray-500">${desc.substring(0, 150)}${desc.length > 150 ? '...' : ''}</p></td>`;
        });
        html += '</tr>';

        // Add to cart row
        html += '<tr><th class="text-lg-start text-center">Action</th>';
        products.forEach(product => {
            html += `<td class="text-center">
                <a class="btn btn-cart font-sm-bold mb-10" href="#" onclick="addToCartFromCompare(${product.id}); return false;">Add to Cart</a>
                <br>
                <a class="btn btn-gray font-sm" href="#" onclick="removeFromCompare(${product.id}); return false;">Remove</a>
            </td>`;
        });
        html += '</tr>';

        container.innerHTML = html;
    }

    function generateStars(rating) {
        let stars = '';
        for (let i = 0; i < 5; i++) {
            if (i < Math.floor(rating)) {
                stars += '<img src="assets/imgs/template/icons/star.svg" alt="star">';
            } else {
                stars += '<img src="assets/imgs/template/icons/star-gray.svg" alt="star">';
            }
        }
        return stars;
    }

    function renderEmptyCompare() {
        const container = document.querySelector('.table-compare');
        if (container) {
            container.outerHTML = `
                <div class="text-center py-5">
                    <h4 class="color-brand-3 mb-20">No products to compare</h4>
                    <p class="color-gray-500 mb-20">Add products to compare their features and specifications.</p>
                    <a class="btn btn-buy" href="shop-grid.html">Browse Products</a>
                </div>`;
        }
    }

    // Global functions
    window.addToCartFromCompare = function(productId) {
        fetch(API_BASE + '/cart/add', {
            method: 'POST',
            headers: getHeaders(),
            body: JSON.stringify({
                product_id: productId,
                quantity: 1,
                session_id: getSessionId()
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Product added to cart!');
                updateCartCount();
            } else {
                alert(data.message || 'Error adding to cart');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error adding to cart');
        });
    };

    window.removeFromCompare = function(productId) {
        fetch(API_BASE + '/compare/' + productId, {
            method: 'DELETE',
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                loadCompare();
            } else {
                alert(data.message || 'Error removing from compare');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error removing from compare');
        });
    };

    window.clearCompare = function() {
        if (!confirm('Remove all products from compare?')) return;

        fetch(API_BASE + '/compare', {
            method: 'DELETE',
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                loadCompare();
            } else {
                alert(data.message || 'Error clearing compare');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error clearing compare');
        });
    };

    function updateCartCount() {
        fetch(API_BASE + '/cart', {
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                const count = data.data.cart?.items?.length || data.data.items?.length || 0;
                document.querySelectorAll('.icon-cart .number-item').forEach(el => {
                    el.textContent = count;
                });
            }
        });
    }
})();
