(function() {
    'use strict';

    const API_BASE = window.KALSES_API_BASE || 'http://127.0.0.1:8000/api/v1';

    // Generate or get session ID for cart
    function getSessionId() {
        let sessionId = localStorage.getItem('cart_session_id');
        if (!sessionId) {
            sessionId = 'sess_' + Math.random().toString(36).substring(2) + Date.now().toString(36);
            localStorage.setItem('cart_session_id', sessionId);
        }
        return sessionId;
    }

    document.addEventListener('DOMContentLoaded', function() {
        loadCart();
    });

    function loadCart() {
        const token = localStorage.getItem('kalses_token');
        const sessionId = getSessionId();

        const headers = {
            'Accept': 'application/json',
            'X-Session-ID': sessionId
        };

        if (token) {
            headers['Authorization'] = 'Bearer ' + token;
        }

        fetch(API_BASE + '/cart', {
            headers: headers
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                renderCart(data.data);
            } else {
                renderEmptyCart();
            }
        })
        .catch(error => {
            console.error('Error loading cart:', error);
            renderEmptyCart();
        });
    }

    function renderCart(cartData) {
        const cart = cartData.cart || cartData;
        const items = cart.items || [];

        if (items.length === 0) {
            renderEmptyCart();
            return;
        }

        // Render cart items
        const container = document.querySelector('.content-wishlist');
        if (container) {
            let html = '';
            let subtotal = 0;

            items.forEach(item => {
                const product = item.product;
                const imageUrl = product.primary_image?.image_url || product.image_url || 'assets/imgs/page/homepage1/imgsp1.png';
                const price = parseFloat(item.price);
                const itemTotal = price * item.quantity;
                subtotal += itemTotal;

                html += `
                <div class="item-wishlist" data-item-id="${item.id}" data-product-id="${product.id}">
                    <div class="wishlist-cb">
                        <input class="cb-layout cb-select" type="checkbox">
                    </div>
                    <div class="wishlist-product">
                        <div class="product-wishlist">
                            <div class="product-image">
                                <a href="shop-single-product.html?slug=${product.slug}">
                                    <img src="${imageUrl}" alt="${product.name}">
                                </a>
                            </div>
                            <div class="product-info">
                                <a href="shop-single-product.html?slug=${product.slug}">
                                    <h6 class="color-brand-3">${product.name}</h6>
                                </a>
                                <div class="rating">
                                    <img src="assets/imgs/template/icons/star.svg" alt="star">
                                    <img src="assets/imgs/template/icons/star.svg" alt="star">
                                    <img src="assets/imgs/template/icons/star.svg" alt="star">
                                    <img src="assets/imgs/template/icons/star.svg" alt="star">
                                    <img src="assets/imgs/template/icons/star.svg" alt="star">
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="wishlist-price">
                        <h4 class="color-brand-3">$${price.toFixed(2)}</h4>
                    </div>
                    <div class="wishlist-status">
                        <div class="box-quantity">
                            <div class="input-quantity">
                                <input class="font-xl color-brand-3 qty-input" type="text" value="${item.quantity}" data-item-id="${item.id}">
                                <span class="minus-cart" onclick="updateQuantity(${item.id}, ${item.quantity - 1})"></span>
                                <span class="plus-cart" onclick="updateQuantity(${item.id}, ${item.quantity + 1})"></span>
                            </div>
                        </div>
                    </div>
                    <div class="wishlist-action">
                        <h4 class="color-brand-3">$${itemTotal.toFixed(2)}</h4>
                    </div>
                    <div class="wishlist-remove">
                        <a class="btn btn-delete" href="#" onclick="removeFromCart(${item.id}); return false;"></a>
                    </div>
                </div>`;
            });

            container.innerHTML = html;
        }

        // Render cart summary
        renderCartSummary(cartData.subtotal || subtotal, items.length);
    }

    function renderCartSummary(subtotal, itemCount) {
        const summaryContainer = document.querySelector('.summary-cart');
        if (summaryContainer) {
            const shipping = subtotal >= 50 ? 0 : 5.00;
            const total = subtotal + shipping;

            summaryContainer.innerHTML = `
                <div class="border-bottom mb-10">
                    <div class="row">
                        <div class="col-6"><span class="font-md-bold color-gray-500">Subtotal</span></div>
                        <div class="col-6 text-end">
                            <h4 id="cart-subtotal">$${subtotal.toFixed(2)}</h4>
                        </div>
                    </div>
                </div>
                <div class="border-bottom mb-10">
                    <div class="row">
                        <div class="col-6"><span class="font-md-bold color-gray-500">Shipping</span></div>
                        <div class="col-6 text-end">
                            <h4>${shipping === 0 ? 'Free' : '$' + shipping.toFixed(2)}</h4>
                        </div>
                    </div>
                </div>
                <div class="border-bottom mb-10">
                    <div class="row">
                        <div class="col-6"><span class="font-md-bold color-gray-500">Items</span></div>
                        <div class="col-6 text-end">
                            <h6 id="cart-items-count">${itemCount} item(s)</h6>
                        </div>
                    </div>
                </div>
                <div class="mb-10">
                    <div class="row">
                        <div class="col-6"><span class="font-md-bold color-gray-500">Total</span></div>
                        <div class="col-6 text-end">
                            <h4 id="cart-total">$${total.toFixed(2)}</h4>
                        </div>
                    </div>
                </div>
                <div class="box-button">
                    <a class="btn btn-buy" href="shop-checkout.html">Proceed To CheckOut</a>
                </div>`;
        }
    }

    function renderEmptyCart() {
        const container = document.querySelector('.content-wishlist');
        if (container) {
            container.innerHTML = `
                <div class="text-center py-5">
                    <h4 class="color-brand-3 mb-20">Your cart is empty</h4>
                    <p class="color-gray-500 mb-20">Looks like you haven't added any products to your cart yet.</p>
                    <a class="btn btn-buy" href="shop-grid.html">Continue Shopping</a>
                </div>`;
        }

        renderCartSummary(0, 0);
    }

    // Update quantity
    window.updateQuantity = function(itemId, newQuantity) {
        if (newQuantity < 1) {
            removeFromCart(itemId);
            return;
        }

        const token = localStorage.getItem('kalses_token');
        const sessionId = getSessionId();

        const headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-Session-ID': sessionId
        };

        if (token) {
            headers['Authorization'] = 'Bearer ' + token;
        }

        fetch(API_BASE + '/cart/item/' + itemId, {
            method: 'PUT',
            headers: headers,
            body: JSON.stringify({
                quantity: newQuantity
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                loadCart();
                updateCartCount();
            } else {
                alert(data.message || 'Error updating cart');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error updating cart');
        });
    };

    // Remove from cart
    window.removeFromCart = function(itemId) {
        const token = localStorage.getItem('kalses_token');
        const sessionId = getSessionId();

        const headers = {
            'Accept': 'application/json',
            'X-Session-ID': sessionId
        };

        if (token) {
            headers['Authorization'] = 'Bearer ' + token;
        }

        fetch(API_BASE + '/cart/item/' + itemId, {
            method: 'DELETE',
            headers: headers
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                loadCart();
                updateCartCount();
            } else {
                alert(data.message || 'Error removing item');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error removing item');
        });
    };

    function updateCartCount() {
        const token = localStorage.getItem('kalses_token');
        const sessionId = getSessionId();

        const headers = {
            'X-Session-ID': sessionId
        };

        if (token) {
            headers['Authorization'] = 'Bearer ' + token;
        }

        fetch(API_BASE + '/cart', {
            headers: headers
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                const count = data.data.cart?.items?.length || data.data.items?.length || 0;
                document.querySelectorAll('.icon-cart .number-item').forEach(el => {
                    el.textContent = count;
                });
            }
        });
    }

    // Make loadCart available globally
    window.cartApi = {
        loadCart: loadCart,
        updateQuantity: window.updateQuantity,
        removeFromCart: window.removeFromCart
    };
})();
