(function() {
    'use strict';

    const API_BASE = window.KALSES_API_BASE || 'http://127.0.0.1:8000/api/v1';

    // Check if user is already logged in
    function isLoggedIn() {
        return !!localStorage.getItem('kalses_token');
    }

    // Get current token
    function getToken() {
        return localStorage.getItem('kalses_token');
    }

    // Handle Registration
    function setupRegistrationForm() {
        const form = document.getElementById('register-form');
        if (!form) return;

        form.addEventListener('submit', function(e) {
            e.preventDefault();

            const name = document.getElementById('register-name').value.trim();
            const email = document.getElementById('register-email').value.trim();
            const password = document.getElementById('register-password').value;
            const passwordConfirm = document.getElementById('register-password-confirm').value;
            const termsAgree = document.getElementById('register-terms');

            // Clear previous errors
            clearErrors();

            // Validation
            let hasError = false;

            if (!name) {
                showError('register-name', 'Full name is required');
                hasError = true;
            }

            if (!email) {
                showError('register-email', 'Email is required');
                hasError = true;
            } else if (!isValidEmail(email)) {
                showError('register-email', 'Please enter a valid email address');
                hasError = true;
            }

            if (!password) {
                showError('register-password', 'Password is required');
                hasError = true;
            } else if (password.length < 8) {
                showError('register-password', 'Password must be at least 8 characters');
                hasError = true;
            }

            if (password !== passwordConfirm) {
                showError('register-password-confirm', 'Passwords do not match');
                hasError = true;
            }

            if (termsAgree && !termsAgree.checked) {
                showError('register-terms', 'You must agree to the terms and conditions');
                hasError = true;
            }

            if (hasError) return;

            // Disable submit button
            const submitBtn = form.querySelector('input[type="submit"], button[type="submit"]');
            const originalText = submitBtn.value || submitBtn.textContent;
            submitBtn.disabled = true;
            if (submitBtn.tagName === 'INPUT') {
                submitBtn.value = 'Creating Account...';
            } else {
                submitBtn.textContent = 'Creating Account...';
            }

            // Make API call
            fetch(API_BASE + '/auth/register', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                body: JSON.stringify({
                    name: name,
                    email: email,
                    password: password,
                    password_confirmation: passwordConfirm
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success && data.data && data.data.token) {
                    // Store the token
                    localStorage.setItem('kalses_token', data.data.token);

                    // Store customer info if provided
                    if (data.data.customer) {
                        localStorage.setItem('kalses_customer', JSON.stringify(data.data.customer));
                    }

                    // Show success message
                    alert('Account created successfully! Welcome to Kalses.');

                    // Check if there's a redirect URL
                    const urlParams = new URLSearchParams(window.location.search);
                    const redirect = urlParams.get('redirect');

                    if (redirect) {
                        window.location.href = redirect;
                    } else {
                        // Default redirect to wholesale or account page
                        window.location.href = 'page-wholesale.html';
                    }
                } else {
                    // Show error message
                    const errorMsg = data.message || data.error || 'Registration failed. Please try again.';

                    // Check for validation errors
                    if (data.errors) {
                        Object.keys(data.errors).forEach(field => {
                            const fieldId = 'register-' + field.replace('_', '-');
                            showError(fieldId, data.errors[field][0]);
                        });
                    } else {
                        alert(errorMsg);
                    }

                    // Re-enable button
                    submitBtn.disabled = false;
                    if (submitBtn.tagName === 'INPUT') {
                        submitBtn.value = originalText;
                    } else {
                        submitBtn.textContent = originalText;
                    }
                }
            })
            .catch(error => {
                console.error('Registration error:', error);
                alert('An error occurred during registration. Please try again.');

                // Re-enable button
                submitBtn.disabled = false;
                if (submitBtn.tagName === 'INPUT') {
                    submitBtn.value = originalText;
                } else {
                    submitBtn.textContent = originalText;
                }
            });
        });
    }

    // Handle Login
    function setupLoginForm() {
        const form = document.getElementById('login-form');
        if (!form) return;

        form.addEventListener('submit', function(e) {
            e.preventDefault();

            const email = document.getElementById('login-email').value.trim();
            const password = document.getElementById('login-password').value;
            const rememberMe = document.getElementById('login-remember');

            // Clear previous errors
            clearErrors();

            // Validation
            let hasError = false;

            if (!email) {
                showError('login-email', 'Email is required');
                hasError = true;
            }

            if (!password) {
                showError('login-password', 'Password is required');
                hasError = true;
            }

            if (hasError) return;

            // Disable submit button
            const submitBtn = form.querySelector('input[type="submit"], button[type="submit"]');
            const originalText = submitBtn.value || submitBtn.textContent;
            submitBtn.disabled = true;
            if (submitBtn.tagName === 'INPUT') {
                submitBtn.value = 'Signing In...';
            } else {
                submitBtn.textContent = 'Signing In...';
            }

            // Make API call
            fetch(API_BASE + '/auth/login', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                body: JSON.stringify({
                    email: email,
                    password: password
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success && data.data && data.data.token) {
                    // Store the token
                    localStorage.setItem('kalses_token', data.data.token);

                    // Store customer info if provided
                    if (data.data.customer) {
                        localStorage.setItem('kalses_customer', JSON.stringify(data.data.customer));
                    }

                    // Check if there's a redirect URL
                    const urlParams = new URLSearchParams(window.location.search);
                    const redirect = urlParams.get('redirect');

                    if (redirect) {
                        window.location.href = redirect;
                    } else {
                        // Default redirect to account page
                        window.location.href = 'page-account.html';
                    }
                } else {
                    // Show error message
                    const errorMsg = data.message || data.error || 'Invalid email or password.';
                    alert(errorMsg);

                    // Re-enable button
                    submitBtn.disabled = false;
                    if (submitBtn.tagName === 'INPUT') {
                        submitBtn.value = originalText;
                    } else {
                        submitBtn.textContent = originalText;
                    }
                }
            })
            .catch(error => {
                console.error('Login error:', error);
                alert('An error occurred during login. Please try again.');

                // Re-enable button
                submitBtn.disabled = false;
                if (submitBtn.tagName === 'INPUT') {
                    submitBtn.value = originalText;
                } else {
                    submitBtn.textContent = originalText;
                }
            });
        });
    }

    // Handle Logout
    function logout() {
        const token = getToken();

        if (token) {
            // Call logout API to invalidate token on server
            fetch(API_BASE + '/auth/logout', {
                method: 'POST',
                headers: {
                    'Authorization': 'Bearer ' + token,
                    'Accept': 'application/json'
                }
            }).catch(() => {
                // Ignore logout API errors
            });
        }

        // Clear local storage
        localStorage.removeItem('kalses_token');
        localStorage.removeItem('kalses_customer');

        // Redirect to login page
        window.location.href = 'page-login.html';
    }

    // Helper: Validate email
    function isValidEmail(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }

    // Helper: Show error on field
    function showError(fieldId, message) {
        const field = document.getElementById(fieldId);
        if (!field) return;

        field.classList.add('is-invalid');

        // Create or update error message
        let errorDiv = field.parentNode.querySelector('.invalid-feedback');
        if (!errorDiv) {
            errorDiv = document.createElement('div');
            errorDiv.className = 'invalid-feedback';
            field.parentNode.appendChild(errorDiv);
        }
        errorDiv.textContent = message;
        errorDiv.style.display = 'block';
    }

    // Helper: Clear all errors
    function clearErrors() {
        document.querySelectorAll('.is-invalid').forEach(el => {
            el.classList.remove('is-invalid');
        });
        document.querySelectorAll('.invalid-feedback').forEach(el => {
            el.style.display = 'none';
        });
    }

    // Initialize on DOM ready
    document.addEventListener('DOMContentLoaded', function() {
        setupRegistrationForm();
        setupLoginForm();

        // Setup logout links
        document.querySelectorAll('.logout-link, [data-action="logout"]').forEach(link => {
            link.addEventListener('click', function(e) {
                e.preventDefault();
                logout();
            });
        });
    });

    // Expose functions globally
    window.kalsesAuth = {
        isLoggedIn: isLoggedIn,
        getToken: getToken,
        logout: logout
    };
})();
