/**
 * Kalses E-Commerce API Integration
 * Connects frontend HTML templates to Laravel API
 */

const KalsesAPI = (function() {
    // Configuration - Uses global config from config.js
    const API_BASE_URL = window.KALSES_API_BASE || 'http://127.0.0.1:8000/api/v1';
    const SESSION_KEY = 'kalses_session_id';
    const TOKEN_KEY = 'kalses_token';
    const CUSTOMER_KEY = 'kalses_customer';

    // Utility Functions
    function getSessionId() {
        let sessionId = localStorage.getItem(SESSION_KEY);
        if (!sessionId) {
            sessionId = 'sess_' + Math.random().toString(36).substr(2, 9) + Date.now();
            localStorage.setItem(SESSION_KEY, sessionId);
        }
        return sessionId;
    }

    function getAuthToken() {
        return localStorage.getItem(TOKEN_KEY);
    }

    function setAuthToken(token) {
        localStorage.setItem(TOKEN_KEY, token);
    }

    function removeAuthToken() {
        localStorage.removeItem(TOKEN_KEY);
        localStorage.removeItem(CUSTOMER_KEY);
    }

    function getCustomer() {
        const customer = localStorage.getItem(CUSTOMER_KEY);
        return customer ? JSON.parse(customer) : null;
    }

    function setCustomer(customer) {
        localStorage.setItem(CUSTOMER_KEY, JSON.stringify(customer));
    }

    function isLoggedIn() {
        return !!getAuthToken();
    }

    // API Request Helper
    async function apiRequest(endpoint, options = {}) {
        const url = API_BASE_URL + endpoint;
        const headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-Session-ID': getSessionId(),
        };

        const token = getAuthToken();
        if (token) {
            headers['Authorization'] = 'Bearer ' + token;
        }

        const config = {
            ...options,
            headers: {
                ...headers,
                ...options.headers,
            },
        };

        if (options.body && typeof options.body === 'object') {
            config.body = JSON.stringify(options.body);
        }

        try {
            const response = await fetch(url, config);
            const data = await response.json();

            if (!response.ok) {
                throw { response, data };
            }

            return data;
        } catch (error) {
            console.error('API Error:', error);
            throw error;
        }
    }

    // ==================== AUTH API ====================
    const auth = {
        async register(userData) {
            userData.session_id = getSessionId();
            const response = await apiRequest('/auth/register', {
                method: 'POST',
                body: userData,
            });
            if (response.success) {
                setAuthToken(response.data.token);
                setCustomer(response.data.customer);
            }
            return response;
        },

        async login(email, password) {
            const response = await apiRequest('/auth/login', {
                method: 'POST',
                body: { email, password, session_id: getSessionId() },
            });
            if (response.success) {
                setAuthToken(response.data.token);
                setCustomer(response.data.customer);
            }
            return response;
        },

        async logout() {
            try {
                await apiRequest('/auth/logout', { method: 'POST' });
            } finally {
                removeAuthToken();
            }
        },

        async getProfile() {
            return apiRequest('/profile');
        },

        async updateProfile(data) {
            return apiRequest('/profile', {
                method: 'PUT',
                body: data,
            });
        },

        async updatePassword(currentPassword, newPassword, confirmPassword) {
            return apiRequest('/profile/password', {
                method: 'PUT',
                body: {
                    current_password: currentPassword,
                    password: newPassword,
                    password_confirmation: confirmPassword,
                },
            });
        },

        isLoggedIn,
        getCustomer,
    };

    // ==================== PRODUCTS API ====================
    const products = {
        async getAll(params = {}) {
            const queryString = new URLSearchParams(params).toString();
            return apiRequest('/products' + (queryString ? '?' + queryString : ''));
        },

        async getOne(slug) {
            return apiRequest('/products/' + slug);
        },

        async getFeatured(limit = 8) {
            return apiRequest('/products/featured?limit=' + limit);
        },

        async getNewArrivals(limit = 8) {
            return apiRequest('/products/new-arrivals?limit=' + limit);
        },

        async getBestSellers(limit = 8) {
            return apiRequest('/products/best-sellers?limit=' + limit);
        },
    };

    // ==================== CATEGORIES API ====================
    const categories = {
        async getAll() {
            return apiRequest('/categories');
        },

        async getOne(slug) {
            return apiRequest('/categories/' + slug);
        },
    };

    // ==================== VENDORS API ====================
    const vendors = {
        async getAll() {
            return apiRequest('/vendors');
        },

        async getOne(slug) {
            return apiRequest('/vendors/' + slug);
        },
    };

    // ==================== CART API ====================
    const cart = {
        async get() {
            return apiRequest('/cart');
        },

        async add(productId, quantity = 1) {
            return apiRequest('/cart/add', {
                method: 'POST',
                body: { product_id: productId, quantity },
            });
        },

        async update(itemId, quantity) {
            return apiRequest('/cart/item/' + itemId, {
                method: 'PUT',
                body: { quantity },
            });
        },

        async remove(itemId) {
            return apiRequest('/cart/item/' + itemId, {
                method: 'DELETE',
            });
        },

        async clear() {
            return apiRequest('/cart/clear', {
                method: 'DELETE',
            });
        },

        async applyCoupon(code) {
            return apiRequest('/cart/coupon', {
                method: 'POST',
                body: { code },
            });
        },
    };

    // ==================== CHECKOUT API ====================
    const checkout = {
        async getSummary(couponCode = null) {
            const params = couponCode ? '?coupon_code=' + couponCode : '';
            return apiRequest('/checkout/summary' + params);
        },

        async process(orderData) {
            return apiRequest('/checkout/process', {
                method: 'POST',
                body: orderData,
            });
        },

        async verifyPayment(sessionId) {
            return apiRequest('/checkout/verify-payment?session_id=' + sessionId);
        },
    };

    // ==================== WISHLIST API ====================
    const wishlist = {
        async get() {
            return apiRequest('/wishlist');
        },

        async add(productId) {
            return apiRequest('/wishlist/add', {
                method: 'POST',
                body: { product_id: productId },
            });
        },

        async remove(productId) {
            return apiRequest('/wishlist/' + productId, {
                method: 'DELETE',
            });
        },

        async check(productId) {
            return apiRequest('/wishlist/check/' + productId);
        },
    };

    // ==================== COMPARE API ====================
    const compare = {
        async get() {
            return apiRequest('/compare');
        },

        async add(productId) {
            return apiRequest('/compare/add', {
                method: 'POST',
                body: { product_id: productId },
            });
        },

        async remove(productId) {
            return apiRequest('/compare/' + productId, {
                method: 'DELETE',
            });
        },

        async clear() {
            return apiRequest('/compare', {
                method: 'DELETE',
            });
        },
    };

    // ==================== ADDRESSES API ====================
    const addresses = {
        async getAll() {
            return apiRequest('/addresses');
        },

        async add(addressData) {
            return apiRequest('/addresses', {
                method: 'POST',
                body: addressData,
            });
        },

        async update(id, addressData) {
            return apiRequest('/addresses/' + id, {
                method: 'PUT',
                body: addressData,
            });
        },

        async remove(id) {
            return apiRequest('/addresses/' + id, {
                method: 'DELETE',
            });
        },
    };

    // ==================== ORDERS API ====================
    const orders = {
        async getAll() {
            return apiRequest('/orders');
        },

        async getOne(id) {
            return apiRequest('/orders/' + id);
        },
    };

    // ==================== CONTENT API ====================
    const content = {
        async getSliders(location = 'homepage') {
            return apiRequest('/sliders/' + location);
        },

        async getBanners(location) {
            return apiRequest('/banners/' + location);
        },

        async getPromoBlocks(location = null) {
            return apiRequest('/promo-blocks' + (location ? '/' + location : ''));
        },

        async getPosts(params = {}) {
            const queryString = new URLSearchParams(params).toString();
            return apiRequest('/posts' + (queryString ? '?' + queryString : ''));
        },

        async getPost(slug) {
            return apiRequest('/posts/' + slug);
        },

        async getPage(slug) {
            return apiRequest('/pages/' + slug);
        },

        async getSettings() {
            return apiRequest('/settings');
        },
    };

    // ==================== REVIEWS API ====================
    const reviews = {
        async submit(productId, rating, title = null, comment = null) {
            return apiRequest('/reviews', {
                method: 'POST',
                body: {
                    product_id: productId,
                    rating,
                    title,
                    comment,
                    customer_id: getCustomer()?.id,
                },
            });
        },
    };

    // Get base URL helper
    function getBaseUrl() {
        return API_BASE_URL;
    }

    // Public API
    return {
        auth,
        products,
        categories,
        vendors,
        cart,
        checkout,
        wishlist,
        compare,
        addresses,
        orders,
        content,
        reviews,
        getSessionId,
        isLoggedIn,
        getBaseUrl,
    };
})();

// Make it globally available
window.KalsesAPI = KalsesAPI;
