(function() {
    'use strict';

    const API_BASE = window.KALSES_API_BASE || 'http://127.0.0.1:8000/api/v1';

    function getSessionId() {
        let sessionId = localStorage.getItem('kalses_session_id');
        if (!sessionId) {
            sessionId = 'sess_' + Math.random().toString(36).substr(2, 9) + Date.now();
            localStorage.setItem('kalses_session_id', sessionId);
        }
        return sessionId;
    }

    function getHeaders() {
        const token = localStorage.getItem('kalses_token');
        const headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-Session-ID': getSessionId()
        };
        if (token) {
            headers['Authorization'] = 'Bearer ' + token;
        }
        return headers;
    }

    document.addEventListener('DOMContentLoaded', function() {
        loadWishlist();
    });

    function loadWishlist() {
        fetch(API_BASE + '/wishlist', {
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                renderWishlist(data.data.items || data.data);
            } else {
                renderEmptyWishlist();
            }
        })
        .catch(error => {
            console.error('Error loading wishlist:', error);
            renderEmptyWishlist();
        });
    }

    function renderWishlist(items) {
        const container = document.querySelector('.content-wishlist');
        if (!container) return;

        if (!items || items.length === 0) {
            renderEmptyWishlist();
            return;
        }

        let html = '';
        items.forEach(item => {
            const product = item.product || item;
            const imageUrl = product.primary_image?.image_url || 'assets/imgs/page/product/img-sub.png';
            const price = product.sale_price || product.price;
            const inStock = product.stock_quantity > 0;

            html += `
            <div class="item-wishlist" data-product-id="${product.id}">
                <div class="wishlist-cb">
                    <input class="cb-layout cb-select" type="checkbox" data-product-id="${product.id}">
                </div>
                <div class="wishlist-product">
                    <div class="product-wishlist">
                        <div class="product-image">
                            <a href="shop-single-product.html?slug=${product.slug}">
                                <img src="${imageUrl}" alt="${product.name}">
                            </a>
                        </div>
                        <div class="product-info">
                            <a href="shop-single-product.html?slug=${product.slug}">
                                <h6 class="color-brand-3">${product.name}</h6>
                            </a>
                            <div class="rating">
                                ${generateStars(product.average_rating || 5)}
                                <span class="font-xs color-gray-500"> (${product.reviews_count || 0})</span>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="wishlist-price">
                    <h4 class="color-brand-3">$${parseFloat(price).toFixed(2)}</h4>
                </div>
                <div class="wishlist-status">
                    <span class="btn btn-gray font-md-bold color-brand-3">${inStock ? 'In Stock' : 'Out of Stock'}</span>
                </div>
                <div class="wishlist-action">
                    <a class="btn btn-cart font-sm-bold" href="#" onclick="addToCartFromWishlist(${product.id}); return false;">Add to Cart</a>
                </div>
                <div class="wishlist-remove">
                    <a class="btn btn-delete" href="#" onclick="removeFromWishlist(${product.id}); return false;"></a>
                </div>
            </div>`;
        });

        container.innerHTML = html;
        updateWishlistCount(items.length);
    }

    function generateStars(rating) {
        let stars = '';
        for (let i = 0; i < 5; i++) {
            if (i < Math.floor(rating)) {
                stars += '<img src="assets/imgs/template/icons/star.svg" alt="star">';
            } else {
                stars += '<img src="assets/imgs/template/icons/star-gray.svg" alt="star">';
            }
        }
        return stars;
    }

    function renderEmptyWishlist() {
        const container = document.querySelector('.content-wishlist');
        if (container) {
            container.innerHTML = `
                <div class="text-center py-5">
                    <h4 class="color-brand-3 mb-20">Your wishlist is empty</h4>
                    <p class="color-gray-500 mb-20">Add items to your wishlist to save them for later.</p>
                    <a class="btn btn-buy" href="shop-grid.html">Browse Products</a>
                </div>`;
        }
        updateWishlistCount(0);
    }

    function updateWishlistCount(count) {
        document.querySelectorAll('.icon-wishlist .number-item').forEach(el => {
            el.textContent = count;
        });
    }

    // Global functions
    window.addToCartFromWishlist = function(productId) {
        fetch(API_BASE + '/cart/add', {
            method: 'POST',
            headers: getHeaders(),
            body: JSON.stringify({
                product_id: productId,
                quantity: 1,
                session_id: getSessionId()
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Product added to cart!');
                updateCartCount();
            } else {
                alert(data.message || 'Error adding to cart');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error adding to cart');
        });
    };

    window.removeFromWishlist = function(productId) {
        if (!confirm('Remove this item from wishlist?')) return;

        fetch(API_BASE + '/wishlist/' + productId, {
            method: 'DELETE',
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                loadWishlist();
            } else {
                alert(data.message || 'Error removing from wishlist');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error removing from wishlist');
        });
    };

    window.addAllToCart = function() {
        const checkboxes = document.querySelectorAll('.cb-select:checked');
        if (checkboxes.length === 0) {
            alert('Please select items to add to cart');
            return;
        }

        const productIds = Array.from(checkboxes).map(cb => cb.dataset.productId);
        let completed = 0;

        productIds.forEach(productId => {
            fetch(API_BASE + '/cart/add', {
                method: 'POST',
                headers: getHeaders(),
                body: JSON.stringify({
                    product_id: parseInt(productId),
                    quantity: 1,
                    session_id: getSessionId()
                })
            })
            .then(response => response.json())
            .then(data => {
                completed++;
                if (completed === productIds.length) {
                    alert('Selected items added to cart!');
                    updateCartCount();
                }
            })
            .catch(error => {
                console.error('Error:', error);
                completed++;
            });
        });
    };

    function updateCartCount() {
        fetch(API_BASE + '/cart', {
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                const count = data.data.cart?.items?.length || data.data.items?.length || 0;
                document.querySelectorAll('.icon-cart .number-item').forEach(el => {
                    el.textContent = count;
                });
            }
        });
    }
})();
