(function() {
    'use strict';

    const API_BASE = window.KALSES_API_BASE || 'http://127.0.0.1:8000/api/v1';
    let cartData = null;
    let appliedCoupon = null;

    function getSessionId() {
        let sessionId = localStorage.getItem('kalses_session_id');
        if (!sessionId) {
            sessionId = 'sess_' + Math.random().toString(36).substr(2, 9) + Date.now();
            localStorage.setItem('kalses_session_id', sessionId);
        }
        return sessionId;
    }

    function getHeaders() {
        const token = localStorage.getItem('kalses_token');
        const headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-Session-ID': getSessionId()
        };
        if (token) {
            headers['Authorization'] = 'Bearer ' + token;
        }
        return headers;
    }

    document.addEventListener('DOMContentLoaded', function() {
        loadCart();
        setupCheckoutForm();
        setupCouponForm();
        prefillCustomerData();
    });

    // Load cart items
    function loadCart() {
        fetch(API_BASE + '/cart', {
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                cartData = data.data.cart || data.data;
                renderCartItems(cartData);
            } else {
                renderEmptyCart();
            }
        })
        .catch(error => {
            console.error('Error loading cart:', error);
            renderEmptyCart();
        });
    }

    function renderCartItems(cart) {
        const items = cart.items || [];
        const container = document.querySelector('.listCheckout');

        if (!container) return;

        if (items.length === 0) {
            renderEmptyCart();
            return;
        }

        let html = '';
        let subtotal = 0;

        items.forEach(item => {
            const product = item.product;
            const imageUrl = product.primary_image?.image_url || product.image_url || 'assets/imgs/page/product/img-sub.png';
            const price = parseFloat(item.price);
            const itemTotal = price * item.quantity;
            subtotal += itemTotal;

            html += `
            <div class="item-wishlist">
                <div class="wishlist-product">
                    <div class="product-wishlist">
                        <div class="product-image">
                            <a href="shop-single-product.html?slug=${product.slug}">
                                <img src="${imageUrl}" alt="${product.name}">
                            </a>
                        </div>
                        <div class="product-info">
                            <a href="shop-single-product.html?slug=${product.slug}">
                                <h6 class="color-brand-3">${product.name}</h6>
                            </a>
                            <div class="rating">
                                ${generateStars(product.average_rating || 5)}
                                <span class="font-xs color-gray-500"> (${product.reviews_count || 0})</span>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="wishlist-status">
                    <h5 class="color-gray-500">x${item.quantity}</h5>
                </div>
                <div class="wishlist-price">
                    <h4 class="color-brand-3 font-lg-bold">$${itemTotal.toFixed(2)}</h4>
                </div>
            </div>`;
        });

        container.innerHTML = html;
        updateOrderSummary(subtotal);
    }

    function generateStars(rating) {
        let stars = '';
        for (let i = 0; i < 5; i++) {
            if (i < Math.floor(rating)) {
                stars += '<img src="assets/imgs/template/icons/star.svg" alt="star">';
            } else {
                stars += '<img src="assets/imgs/template/icons/star-gray.svg" alt="star">';
            }
        }
        return stars;
    }

    function updateOrderSummary(subtotal) {
        const shipping = subtotal >= 50 ? 0 : 5.00;
        let discount = 0;

        if (appliedCoupon) {
            if (appliedCoupon.discount_type === 'percentage') {
                discount = subtotal * (appliedCoupon.discount_value / 100);
            } else {
                discount = appliedCoupon.discount_value;
            }
        }

        const total = subtotal + shipping - discount;

        // Update summary display
        const summaryContainer = document.querySelector('.form-group.mb-0');
        if (summaryContainer) {
            let summaryHtml = `
                <div class="row mb-10">
                    <div class="col-lg-6 col-6"><span class="font-md-bold color-brand-3">Subtotal</span></div>
                    <div class="col-lg-6 col-6 text-end"><span class="font-lg-bold color-brand-3" id="checkout-subtotal">$${subtotal.toFixed(2)}</span></div>
                </div>`;

            if (appliedCoupon) {
                summaryHtml += `
                <div class="row mb-10">
                    <div class="col-lg-6 col-6"><span class="font-md-bold color-success">Discount (${appliedCoupon.code})</span></div>
                    <div class="col-lg-6 col-6 text-end"><span class="font-lg-bold color-success">-$${discount.toFixed(2)}</span></div>
                </div>`;
            }

            summaryHtml += `
                <div class="border-bottom mb-10 pb-5">
                    <div class="row">
                        <div class="col-lg-6 col-6"><span class="font-md-bold color-brand-3">Shipping</span></div>
                        <div class="col-lg-6 col-6 text-end"><span class="font-lg-bold color-brand-3" id="checkout-shipping">${shipping === 0 ? 'Free' : '$' + shipping.toFixed(2)}</span></div>
                    </div>
                </div>
                <div class="row">
                    <div class="col-lg-6 col-6"><span class="font-md-bold color-brand-3">Total</span></div>
                    <div class="col-lg-6 col-6 text-end"><span class="font-lg-bold color-brand-3" id="checkout-total">$${total.toFixed(2)}</span></div>
                </div>`;

            summaryContainer.innerHTML = summaryHtml;
        }
    }

    function renderEmptyCart() {
        const container = document.querySelector('.listCheckout');
        if (container) {
            container.innerHTML = `
                <div class="text-center py-4">
                    <h5 class="color-brand-3 mb-15">Your cart is empty</h5>
                    <a class="btn btn-buy" href="shop-grid.html">Continue Shopping</a>
                </div>`;
        }
        updateOrderSummary(0);
    }

    // Prefill customer data if logged in
    function prefillCustomerData() {
        const customer = JSON.parse(localStorage.getItem('kalses_customer') || 'null');
        if (!customer) return;

        const emailInput = document.querySelector('input[placeholder*="Email"]');
        if (emailInput && customer.email) {
            emailInput.value = customer.email;
        }

        // Load saved addresses
        fetch(API_BASE + '/addresses', {
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                const addresses = data.data.addresses || data.data;
                const defaultAddress = addresses.find(a => a.is_default) || addresses[0];
                if (defaultAddress) {
                    prefillAddress(defaultAddress);
                }
            }
        })
        .catch(error => console.error('Error loading addresses:', error));
    }

    function prefillAddress(address) {
        const firstNameInput = document.querySelector('input[placeholder*="First name"]');
        const lastNameInput = document.querySelector('input[placeholder*="Last name"]');
        const address1Input = document.querySelector('input[placeholder*="Address 1"]');
        const address2Input = document.querySelector('input[placeholder*="Address 2"]');
        const cityInput = document.querySelector('input[placeholder*="City"]');
        const zipInput = document.querySelector('input[placeholder*="PostCode"]');
        const companyInput = document.querySelector('input[placeholder*="Company"]');
        const phoneInput = document.querySelector('input[placeholder*="Phone"]');

        if (firstNameInput) firstNameInput.value = address.first_name || '';
        if (lastNameInput) lastNameInput.value = address.last_name || '';
        if (address1Input) address1Input.value = address.address_1 || '';
        if (address2Input) address2Input.value = address.address_2 || '';
        if (cityInput) cityInput.value = address.city || '';
        if (zipInput) zipInput.value = address.zip_code || '';
        if (companyInput) companyInput.value = address.company || '';
        if (phoneInput) phoneInput.value = address.phone || '';
    }

    // Setup coupon form
    function setupCouponForm() {
        const couponInput = document.querySelector('input[placeholder*="Coupon"]');
        const applyBtn = couponInput?.nextElementSibling;

        if (applyBtn) {
            applyBtn.addEventListener('click', function(e) {
                e.preventDefault();
                const code = couponInput.value.trim();
                if (code) {
                    applyCoupon(code);
                }
            });
        }
    }

    function applyCoupon(code) {
        fetch(API_BASE + '/cart/coupon', {
            method: 'POST',
            headers: getHeaders(),
            body: JSON.stringify({ code: code })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                appliedCoupon = data.data.coupon || { code: code, discount_type: 'fixed', discount_value: data.data.discount || 0 };
                alert('Coupon applied successfully!');
                loadCart();
            } else {
                alert(data.message || 'Invalid coupon code');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error applying coupon');
        });
    }

    // Setup checkout form
    function setupCheckoutForm() {
        const placeOrderBtn = document.querySelector('#place-order-btn') || document.querySelector('.btn-buy.arrow-next');
        if (placeOrderBtn) {
            placeOrderBtn.href = '#';
            placeOrderBtn.addEventListener('click', function(e) {
                e.preventDefault();
                processCheckout();
            });
        }
    }

    function processCheckout() {
        if (!cartData || !cartData.items || cartData.items.length === 0) {
            alert('Your cart is empty');
            return;
        }

        // Gather form data
        const emailInput = document.querySelector('input[placeholder*="Email"]');
        const firstNameInput = document.querySelector('input[placeholder*="First name"]');
        const lastNameInput = document.querySelector('input[placeholder*="Last name"]');
        const address1Input = document.querySelector('input[placeholder*="Address 1"]');
        const address2Input = document.querySelector('input[placeholder*="Address 2"]');
        const cityInput = document.querySelector('input[placeholder*="City"]');
        const zipInput = document.querySelector('input[placeholder*="PostCode"]');
        const companyInput = document.querySelector('input[placeholder*="Company"]');
        const phoneInput = document.querySelector('input[placeholder*="Phone"]');
        const notesInput = document.querySelector('textarea[placeholder*="Additional"]');
        const countrySelect = document.querySelector('.select-style1');

        // Get selected payment method
        const paymentMethodRadio = document.querySelector('input[name="payment_method"]:checked');
        const paymentMethod = paymentMethodRadio ? paymentMethodRadio.value : 'cod';

        // Validate required fields
        const email = emailInput?.value?.trim();
        const firstName = firstNameInput?.value?.trim();
        const lastName = lastNameInput?.value?.trim();
        const address1 = address1Input?.value?.trim();
        const city = cityInput?.value?.trim();
        const zip = zipInput?.value?.trim();
        const phone = phoneInput?.value?.trim();

        if (!email || !firstName || !lastName || !address1 || !city || !zip || !phone) {
            alert('Please fill in all required fields');
            return;
        }

        // Email validation
        if (!email.match(/^[^\s@]+@[^\s@]+\.[^\s@]+$/)) {
            alert('Please enter a valid email address');
            return;
        }

        const orderData = {
            email: email,
            shipping_address: {
                first_name: firstName,
                last_name: lastName,
                address_1: address1,
                address_2: address2Input?.value?.trim() || '',
                city: city,
                state: countrySelect?.value || '',
                zip_code: zip,
                country: 'USA',
                phone: phone,
                company: companyInput?.value?.trim() || ''
            },
            billing_address: {
                first_name: firstName,
                last_name: lastName,
                address_1: address1,
                address_2: address2Input?.value?.trim() || '',
                city: city,
                state: countrySelect?.value || '',
                zip_code: zip,
                country: 'USA',
                phone: phone
            },
            notes: notesInput?.value?.trim() || '',
            payment_method: paymentMethod,
            coupon_code: appliedCoupon?.code || null,
            session_id: getSessionId()
        };

        // Show loading state
        const placeOrderBtn = document.querySelector('#place-order-btn') || document.querySelector('.btn-buy.arrow-next');
        const originalText = placeOrderBtn.textContent;
        placeOrderBtn.textContent = 'Processing...';
        placeOrderBtn.style.pointerEvents = 'none';

        fetch(API_BASE + '/checkout/process', {
            method: 'POST',
            headers: getHeaders(),
            body: JSON.stringify(orderData)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Clear cart from local storage
                localStorage.removeItem('kalses_cart');

                // Check if this is a Stripe payment that needs redirect
                if (paymentMethod === 'stripe' && data.data.checkout_url) {
                    // Redirect to Stripe Checkout
                    window.location.href = data.data.checkout_url;
                } else {
                    // COD or direct success
                    const orderNumber = data.data.order_number || data.data.order?.order_number || 'N/A';

                    // Show success message
                    showOrderSuccess(orderNumber, paymentMethod);
                }
            } else {
                alert(data.message || 'Error processing order. Please try again.');
                placeOrderBtn.textContent = originalText;
                placeOrderBtn.style.pointerEvents = 'auto';
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error processing order. Please try again.');
            placeOrderBtn.textContent = originalText;
            placeOrderBtn.style.pointerEvents = 'auto';
        });
    }

    // Show order success message
    function showOrderSuccess(orderNumber, paymentMethod) {
        const message = paymentMethod === 'cod'
            ? `Order placed successfully!\n\nOrder #${orderNumber}\n\nYou will pay when you receive your order.`
            : `Order placed successfully!\n\nOrder #${orderNumber}`;

        alert(message);
        window.location.href = 'page-account.html';
    }

    // Check for payment verification on page load (returning from Stripe)
    function checkPaymentVerification() {
        const urlParams = new URLSearchParams(window.location.search);
        const sessionId = urlParams.get('session_id');
        const paymentStatus = urlParams.get('payment');

        // Handle cancelled payment
        if (paymentStatus === 'cancelled') {
            alert('Payment was cancelled. Your order has not been placed.');
            window.history.replaceState({}, document.title, window.location.pathname);
            return;
        }

        // Handle successful payment verification
        if (sessionId && paymentStatus === 'success') {
            // Show loading indicator
            const container = document.querySelector('.shop-template .container');
            if (container) {
                container.innerHTML = `
                    <div class="text-center py-5">
                        <div class="spinner-border text-success mb-3" role="status">
                            <span class="visually-hidden">Verifying payment...</span>
                        </div>
                        <h4>Verifying your payment...</h4>
                        <p class="text-muted">Please wait while we confirm your order.</p>
                    </div>
                `;
            }

            // Verify the payment
            fetch(API_BASE + '/checkout/verify-payment?session_id=' + sessionId, {
                headers: getHeaders()
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const orderNumber = data.data.order?.order_number || data.data.order_number || 'N/A';

                    // Show success message
                    if (container) {
                        container.innerHTML = `
                            <div class="text-center py-5">
                                <div class="mb-3">
                                    <i class="fas fa-check-circle text-success" style="font-size: 64px;"></i>
                                </div>
                                <h3 class="color-brand-3">Payment Successful!</h3>
                                <p class="text-muted mb-4">Thank you for your order.</p>
                                <h5>Order Number: <strong>${orderNumber}</strong></h5>
                                <p class="mt-3">A confirmation email has been sent to your email address.</p>
                                <div class="mt-4">
                                    <a href="page-account.html" class="btn btn-buy">View My Orders</a>
                                    <a href="shop-grid.html" class="btn btn-outline-primary ms-2">Continue Shopping</a>
                                </div>
                            </div>
                        `;
                    } else {
                        alert('Payment successful! Order #' + orderNumber);
                        window.location.href = 'page-account.html';
                    }
                } else {
                    if (container) {
                        container.innerHTML = `
                            <div class="text-center py-5">
                                <div class="mb-3">
                                    <i class="fas fa-exclamation-triangle text-warning" style="font-size: 64px;"></i>
                                </div>
                                <h3 class="color-brand-3">Payment Verification Issue</h3>
                                <p class="text-muted">We could not verify your payment. Please contact support if you were charged.</p>
                                <div class="mt-4">
                                    <a href="page-contact.html" class="btn btn-buy">Contact Support</a>
                                </div>
                            </div>
                        `;
                    } else {
                        alert('Payment verification failed. Please contact support.');
                    }
                }
            })
            .catch(error => {
                console.error('Payment verification error:', error);
                alert('Error verifying payment. Please contact support.');
            });
        }
    }

    // Run payment verification check on load
    checkPaymentVerification();
})();
