(function() {
    'use strict';

    const API_BASE = window.KALSES_API_BASE || 'http://127.0.0.1:8000/api/v1';
    const STORAGE_URL = window.KALSES_STORAGE_URL || 'http://127.0.0.1:8000/storage';

    // Check authentication on page load
    document.addEventListener('DOMContentLoaded', function() {
        const token = localStorage.getItem('kalses_token');
        if (!token) {
            window.location.href = 'page-login.html?redirect=page-account.html';
            return;
        }

        // Clear demo content and show loading states
        clearAllDemoContent();

        // Load account data
        loadProfile();
        loadOrders();
        loadWishlist();
        setupForms();
        setupLogout();
    });

    function getHeaders() {
        const token = localStorage.getItem('kalses_token');
        const sessionId = localStorage.getItem('kalses_session_id') || 'sess_' + Math.random().toString(36).substr(2, 9) + Date.now();

        return {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'Authorization': 'Bearer ' + token,
            'X-Session-ID': sessionId
        };
    }

    // Clear all demo content
    function clearAllDemoContent() {
        const loadingHTML = '<div class="text-center py-5"><div class="spinner-border text-primary" role="status"></div><p class="mt-3 color-gray-500">Loading...</p></div>';

        // Clear orders tab
        const ordersTab = document.getElementById('tab-orders');
        if (ordersTab) ordersTab.innerHTML = loadingHTML;

        // Clear wishlist content (keep header)
        const wishlistContent = document.querySelector('#tab-wishlist .content-wishlist');
        if (wishlistContent) wishlistContent.innerHTML = loadingHTML;

        // Clear notifications
        const notificationTab = document.getElementById('tab-notification');
        if (notificationTab) {
            const listNotifications = notificationTab.querySelector('.list-notifications');
            if (listNotifications) {
                listNotifications.innerHTML = loadingHTML;
            }
        }
    }

    // Load customer profile
    function loadProfile() {
        fetch(API_BASE + '/profile', {
            headers: getHeaders()
        })
        .then(response => {
            if (response.status === 401) {
                localStorage.removeItem('kalses_token');
                localStorage.removeItem('kalses_customer');
                window.location.href = 'page-login.html?redirect=page-account.html';
                return null;
            }
            return response.json();
        })
        .then(data => {
            if (data && data.success && data.data) {
                const customer = data.data.customer || data.data;
                renderProfile(customer);
                localStorage.setItem('kalses_customer', JSON.stringify(customer));
            }
        })
        .catch(error => {
            console.error('Error loading profile:', error);
        });
    }

    function renderProfile(customer) {
        // Update welcome message in sidebar
        const welcomeHeadings = document.querySelectorAll('h3');
        welcomeHeadings.forEach(el => {
            if (el.textContent.includes('Hello')) {
                el.innerHTML = 'Hello <span class="color-brand-3">' + (customer.name || 'Customer') + '</span>';
            }
        });

        // Update profile form fields
        const nameInput = document.getElementById('profile-name');
        if (nameInput) nameInput.value = customer.name || '';

        const emailInput = document.getElementById('profile-email');
        if (emailInput) emailInput.value = customer.email || '';

        const phoneInput = document.getElementById('profile-phone');
        if (phoneInput) phoneInput.value = customer.phone || '';

        // Update customer info display in right column
        updateCustomerInfoDisplay(customer);
    }

    function updateCustomerInfoDisplay(customer) {
        const settingsTab = document.getElementById('tab-setting');
        if (!settingsTab) return;

        const infoColumn = settingsTab.querySelector('.col-lg-5');
        if (infoColumn) {
            infoColumn.innerHTML = `
                <div class="box-info-address">
                    <h4 class="font-md-bold color-brand-3 mb-20">Account Information</h4>
                    <div class="mb-15">
                        <p class="font-sm color-gray-700 mb-5"><strong>Name:</strong></p>
                        <p class="font-sm color-gray-500">${customer.name || 'Not set'}</p>
                    </div>
                    <div class="mb-15">
                        <p class="font-sm color-gray-700 mb-5"><strong>Email:</strong></p>
                        <p class="font-sm color-gray-500">${customer.email || 'Not set'}</p>
                    </div>
                    <div class="mb-15">
                        <p class="font-sm color-gray-700 mb-5"><strong>Phone:</strong></p>
                        <p class="font-sm color-gray-500">${customer.phone || 'Not set'}</p>
                    </div>
                    <div class="mb-15">
                        <p class="font-sm color-gray-700 mb-5"><strong>Member Since:</strong></p>
                        <p class="font-sm color-gray-500">${new Date(customer.created_at).toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' })}</p>
                    </div>
                </div>
            `;
        }
    }

    // Load orders
    function loadOrders() {
        fetch(API_BASE + '/orders', {
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                const orders = data.data.orders || data.data;
                renderOrders(orders);
                renderNotifications(orders);
            } else {
                renderOrders([]);
                renderNotifications([]);
            }
        })
        .catch(error => {
            console.error('Error loading orders:', error);
            renderOrders([]);
            renderNotifications([]);
        });
    }

    function renderOrders(orders) {
        const ordersTab = document.getElementById('tab-orders');
        if (!ordersTab) return;

        if (!orders || orders.length === 0) {
            ordersTab.innerHTML = `
                <div class="text-center py-5">
                    <i class="fas fa-shopping-bag fa-3x color-gray-400 mb-20"></i>
                    <h4 class="color-brand-3 mb-15">No orders yet</h4>
                    <p class="color-gray-500 mb-20">You haven't placed any orders yet.</p>
                    <a class="btn btn-buy" href="shop-grid.html">Start Shopping</a>
                </div>
            `;
            return;
        }

        let html = '';
        orders.forEach(order => {
            const statusLabel = getStatusLabel(order.status);
            const orderDate = new Date(order.created_at).toLocaleDateString('en-US', {
                day: 'numeric',
                month: 'long',
                year: 'numeric'
            });

            html += `
            <div class="box-orders mb-20">
                <div class="head-orders">
                    <div class="head-left">
                        <h5 class="mr-20">Order ID: #${order.order_number || order.id}</h5>
                        <span class="font-md color-brand-3 mr-20">Date: ${orderDate}</span>
                        <span class="label-delivery ${statusLabel.class}">${statusLabel.text}</span>
                    </div>
                    <div class="head-right">
                        <a class="btn btn-buy font-sm-bold w-auto" href="#" onclick="viewOrderDetails(${order.id}); return false;">View Order</a>
                    </div>
                </div>
                <div class="body-orders">
                    <div class="list-orders">`;

            if (order.items && order.items.length > 0) {
                order.items.slice(0, 3).forEach(item => {
                    const imageUrl = getProductImage(item);
                    html += `
                        <div class="item-orders">
                            <div class="image-orders"><img src="${imageUrl}" alt="${item.product?.name || 'Product'}"></div>
                            <div class="info-orders">
                                <h5>${item.product?.name || item.product_name || 'Product'}</h5>
                            </div>
                            <div class="quantity-orders">
                                <h5>Quantity: ${item.quantity}</h5>
                            </div>
                            <div class="price-orders">
                                <h3>$${parseFloat(item.price).toFixed(2)}</h3>
                            </div>
                        </div>`;
                });

                if (order.items.length > 3) {
                    html += `<p class="text-muted font-sm mt-10">+ ${order.items.length - 3} more items</p>`;
                }
            }

            html += `
                    </div>
                    <div class="border-top pt-15 mt-15">
                        <div class="row">
                            <div class="col-6"><strong>Total:</strong></div>
                            <div class="col-6 text-end"><strong class="color-brand-3">$${parseFloat(order.total).toFixed(2)}</strong></div>
                        </div>
                    </div>
                </div>
            </div>`;
        });

        ordersTab.innerHTML = html;
    }

    function getProductImage(item) {
        if (item.product?.primary_image_url) return item.product.primary_image_url;
        if (item.product?.images?.[0]?.image_path) return STORAGE_URL + '/' + item.product.images[0].image_path;
        return 'assets/imgs/page/account/img-1.png';
    }

    function getStatusLabel(status) {
        const statusMap = {
            'pending': { text: 'Pending', class: 'label-pending' },
            'processing': { text: 'Processing', class: 'label-processing' },
            'shipped': { text: 'Shipped', class: 'label-shipped' },
            'delivered': { text: 'Delivered', class: 'label-delivered' },
            'cancelled': { text: 'Cancelled', class: 'label-cancel' },
            'refunded': { text: 'Refunded', class: 'label-cancel' }
        };
        return statusMap[status] || { text: status || 'Unknown', class: '' };
    }

    function renderNotifications(orders) {
        const notificationTab = document.getElementById('tab-notification');
        if (!notificationTab) return;

        let listDiv = notificationTab.querySelector('.list-notifications');
        if (!listDiv) {
            notificationTab.innerHTML = '<div class="list-notifications"></div>';
            listDiv = notificationTab.querySelector('.list-notifications');
        }

        if (!orders || orders.length === 0) {
            listDiv.innerHTML = `
                <div class="text-center py-5">
                    <i class="fas fa-bell fa-3x color-gray-400 mb-20"></i>
                    <h4 class="color-brand-3 mb-15">No notifications</h4>
                    <p class="color-gray-500">You'll see order updates and notifications here.</p>
                </div>
            `;
            return;
        }

        let html = '';
        orders.slice(0, 5).forEach(order => {
            const imageUrl = order.items?.[0] ? getProductImage(order.items[0]) : 'assets/imgs/page/account/img-1.png';
            const statusText = getNotificationText(order.status);

            html += `
            <div class="item-notification">
                <div class="image-notification"><img src="${imageUrl}" alt="Order"></div>
                <div class="info-notification">
                    <h5 class="mb-5">${statusText}</h5>
                    <p class="font-md color-brand-3">Order <span class="font-md-bold">#${order.order_number || order.id}</span> ${getStatusDescription(order.status)}</p>
                    <small class="color-gray-500">${new Date(order.updated_at || order.created_at).toLocaleDateString()}</small>
                </div>
                <div class="button-notification">
                    <a class="btn btn-buy w-auto" href="#" onclick="viewOrderDetails(${order.id}); return false;">View Details</a>
                </div>
            </div>`;
        });

        listDiv.innerHTML = html;
    }

    function getNotificationText(status) {
        const texts = {
            'pending': 'Order Placed',
            'processing': 'Order Processing',
            'shipped': 'Order Shipped',
            'delivered': 'Order Delivered',
            'cancelled': 'Order Cancelled'
        };
        return texts[status] || 'Order Update';
    }

    function getStatusDescription(status) {
        const descriptions = {
            'pending': 'is being reviewed.',
            'processing': 'is being prepared.',
            'shipped': 'has been shipped!',
            'delivered': 'has been delivered!',
            'cancelled': 'has been cancelled.'
        };
        return descriptions[status] || 'status updated.';
    }

    // Load wishlist
    function loadWishlist() {
        fetch(API_BASE + '/wishlist', {
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                renderWishlist(data.data.items || data.data);
            } else {
                renderWishlist([]);
            }
        })
        .catch(error => {
            console.error('Error loading wishlist:', error);
            renderWishlist([]);
        });
    }

    function renderWishlist(items) {
        const contentDiv = document.querySelector('#tab-wishlist .content-wishlist');
        if (!contentDiv) return;

        if (!items || items.length === 0) {
            contentDiv.innerHTML = `
                <div class="text-center py-5">
                    <i class="fas fa-heart fa-3x color-gray-400 mb-20"></i>
                    <h4 class="color-brand-3 mb-15">Your wishlist is empty</h4>
                    <p class="color-gray-500 mb-20">Add items to your wishlist to save them for later.</p>
                    <a class="btn btn-buy" href="shop-grid.html">Browse Products</a>
                </div>
            `;
            return;
        }

        let html = '';
        items.forEach(item => {
            const product = item.product || item;
            const imageUrl = product.primary_image_url || (product.images?.[0]?.image_path ? STORAGE_URL + '/' + product.images[0].image_path : 'assets/imgs/page/product/img-sub.png');
            const price = product.sale_price || product.price;
            const inStock = product.stock_quantity > 0;

            html += `
            <div class="item-wishlist" data-product-id="${product.id}">
                <div class="wishlist-cb">
                    <input class="cb-layout cb-select" type="checkbox">
                </div>
                <div class="wishlist-product">
                    <div class="product-wishlist">
                        <div class="product-image">
                            <a href="shop-single-product.html?slug=${product.slug}">
                                <img src="${imageUrl}" alt="${product.name}">
                            </a>
                        </div>
                        <div class="product-info">
                            <a href="shop-single-product.html?slug=${product.slug}">
                                <h6 class="color-brand-3">${product.name}</h6>
                            </a>
                            <div class="rating">
                                ${generateStars(product.average_rating || 0)}
                                <span class="font-xs color-gray-500"> (${product.reviews_count || 0})</span>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="wishlist-price">
                    <h4 class="color-brand-3">$${parseFloat(price).toFixed(2)}</h4>
                </div>
                <div class="wishlist-status">
                    <span class="btn ${inStock ? 'btn-gray' : 'btn-danger'} font-md-bold">${inStock ? 'In Stock' : 'Out of Stock'}</span>
                </div>
                <div class="wishlist-action">
                    <a class="btn btn-cart font-sm-bold" href="#" onclick="addToCartFromWishlist(${product.id}); return false;">Add to Cart</a>
                </div>
                <div class="wishlist-remove">
                    <a class="btn btn-delete" href="#" onclick="removeFromWishlist(${product.id}); return false;"></a>
                </div>
            </div>`;
        });

        contentDiv.innerHTML = html;
    }

    function generateStars(rating) {
        let stars = '';
        for (let i = 0; i < 5; i++) {
            if (i < Math.floor(rating)) {
                stars += '<img src="assets/imgs/template/icons/star.svg" alt="star">';
            } else {
                stars += '<img src="assets/imgs/template/icons/star-gray.svg" alt="star">';
            }
        }
        return stars;
    }

    // Setup form handlers
    function setupForms() {
        // Profile form
        const profileForm = document.getElementById('profile-form');
        if (profileForm) {
            profileForm.addEventListener('submit', function(e) {
                e.preventDefault();
                saveProfile();
            });
        }

        // Order tracking form
        const trackingForm = document.getElementById('order-tracking-form');
        if (trackingForm) {
            trackingForm.addEventListener('submit', function(e) {
                e.preventDefault();
                const input = document.getElementById('tracking-order-number');
                if (input && input.value.trim()) {
                    trackOrder(input.value.trim());
                }
            });
        }
    }

    function saveProfile() {
        const name = document.getElementById('profile-name')?.value;
        const phone = document.getElementById('profile-phone')?.value;

        fetch(API_BASE + '/profile', {
            method: 'PUT',
            headers: getHeaders(),
            body: JSON.stringify({ name, phone })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Profile updated successfully!');
                loadProfile();
            } else {
                alert(data.message || 'Error updating profile');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error updating profile');
        });
    }

    function trackOrder(orderNumber) {
        const resultDiv = document.getElementById('tracking-result');
        if (resultDiv) {
            resultDiv.innerHTML = '<div class="text-center py-4"><div class="spinner-border text-primary"></div><p class="mt-2">Searching for order...</p></div>';
        }

        fetch(API_BASE + '/orders/track/' + encodeURIComponent(orderNumber), {
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                displayOrderTracking(data.data);
            } else {
                displayTrackingError('Order not found. Please check the order number and try again.');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            displayTrackingError('Error tracking order. Please try again.');
        });
    }

    function displayTrackingError(message) {
        const resultDiv = document.getElementById('tracking-result');
        if (resultDiv) {
            resultDiv.innerHTML = `
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle mr-2"></i> ${message}
                </div>
            `;
        }
    }

    function displayOrderTracking(order) {
        const resultDiv = document.getElementById('tracking-result');
        if (!resultDiv) return;

        const statusLabel = getStatusLabel(order.status);
        const orderDate = new Date(order.created_at).toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' });

        // Determine which steps are active based on status
        const statusOrder = ['pending', 'processing', 'shipped', 'delivered'];
        const currentStep = statusOrder.indexOf(order.status);

        resultDiv.innerHTML = `
            <div class="border-bottom mb-20 pb-20"></div>
            <h3 class="mb-10">Order Status: <span class="color-success">${statusLabel.text}</span></h3>
            <h6 class="color-gray-500 mb-20">Order #${order.order_number || order.id} - Placed on ${orderDate}</h6>

            <div class="table-responsive">
                <div class="list-steps">
                    <div class="item-step">
                        <div class="rounded-step">
                            <div class="icon-step step-1 ${currentStep >= 0 ? 'active' : ''}"></div>
                            <h6 class="mb-5">Order Placed</h6>
                            <p class="font-md color-gray-500">${orderDate}</p>
                        </div>
                    </div>
                    <div class="item-step">
                        <div class="rounded-step">
                            <div class="icon-step step-2 ${currentStep >= 1 ? 'active' : ''}"></div>
                            <h6 class="mb-5">Processing</h6>
                            <p class="font-md color-gray-500">${currentStep >= 1 ? 'Confirmed' : 'Pending'}</p>
                        </div>
                    </div>
                    <div class="item-step">
                        <div class="rounded-step">
                            <div class="icon-step step-3 ${currentStep >= 2 ? 'active' : ''}"></div>
                            <h6 class="mb-5">Shipped</h6>
                            <p class="font-md color-gray-500">${currentStep >= 2 ? 'In Transit' : 'Pending'}</p>
                        </div>
                    </div>
                    <div class="item-step">
                        <div class="rounded-step">
                            <div class="icon-step step-4 ${currentStep >= 3 ? 'active' : ''}"></div>
                            <h6 class="mb-5">Delivered</h6>
                            <p class="font-md color-gray-500">${currentStep >= 3 ? 'Completed' : 'Pending'}</p>
                        </div>
                    </div>
                </div>
            </div>

            <div class="mt-30 p-20 bg-light rounded">
                <h5 class="mb-15">Order Details</h5>
                <div class="row">
                    <div class="col-md-6">
                        <p class="mb-10"><strong>Order ID:</strong> ${order.order_number || order.id}</p>
                        <p class="mb-10"><strong>Status:</strong> <span class="badge ${statusLabel.class === 'label-delivered' ? 'bg-success' : 'bg-primary'}">${statusLabel.text}</span></p>
                        <p class="mb-10"><strong>Payment:</strong> ${order.payment_method || 'N/A'}</p>
                    </div>
                    <div class="col-md-6">
                        <p class="mb-10"><strong>Subtotal:</strong> $${parseFloat(order.subtotal || order.total).toFixed(2)}</p>
                        <p class="mb-10"><strong>Shipping:</strong> $${parseFloat(order.shipping_cost || 0).toFixed(2)}</p>
                        <p class="mb-10"><strong>Total:</strong> <strong class="color-brand-3">$${parseFloat(order.total).toFixed(2)}</strong></p>
                    </div>
                </div>
            </div>

            ${order.items && order.items.length > 0 ? `
            <div class="mt-20">
                <h5 class="mb-15">Items in this Order</h5>
                ${order.items.map(item => `
                    <div class="d-flex align-items-center mb-10 p-10 border rounded">
                        <img src="${getProductImage(item)}" alt="${item.product?.name || 'Product'}" style="width: 60px; height: 60px; object-fit: cover;" class="mr-15 rounded">
                        <div class="flex-grow-1">
                            <h6 class="mb-5">${item.product?.name || item.product_name || 'Product'}</h6>
                            <span class="color-gray-500">Qty: ${item.quantity} x $${parseFloat(item.price).toFixed(2)}</span>
                        </div>
                        <strong>$${(item.quantity * parseFloat(item.price)).toFixed(2)}</strong>
                    </div>
                `).join('')}
            </div>
            ` : ''}
        `;
    }

    // Setup logout
    function setupLogout() {
        document.querySelectorAll('a[href="#"], a').forEach(link => {
            if (link.textContent.trim().toLowerCase() === 'logout' ||
                link.textContent.trim().toLowerCase() === 'log out') {
                link.addEventListener('click', function(e) {
                    e.preventDefault();
                    logout();
                });
            }
        });
    }

    // Global functions
    window.viewOrderDetails = function(orderId) {
        fetch(API_BASE + '/orders/' + orderId, {
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                const order = data.data.order || data.data;
                let itemsList = '';
                if (order.items) {
                    order.items.forEach(item => {
                        itemsList += `\n- ${item.product?.name || item.product_name || 'Product'} x${item.quantity}: $${parseFloat(item.price).toFixed(2)}`;
                    });
                }
                alert(`Order #${order.order_number || orderId}\n\nStatus: ${order.status}\nTotal: $${parseFloat(order.total).toFixed(2)}\nDate: ${new Date(order.created_at).toLocaleDateString()}\n\nItems:${itemsList}`);
            }
        })
        .catch(error => {
            console.error('Error:', error);
        });
    };

    window.addToCartFromWishlist = function(productId) {
        fetch(API_BASE + '/cart/add', {
            method: 'POST',
            headers: getHeaders(),
            body: JSON.stringify({
                product_id: productId,
                quantity: 1
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Product added to cart!');
                updateCartCount();
            } else {
                alert(data.message || 'Error adding to cart');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error adding to cart');
        });
    };

    window.removeFromWishlist = function(productId) {
        if (!confirm('Remove this item from wishlist?')) return;

        fetch(API_BASE + '/wishlist/' + productId, {
            method: 'DELETE',
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                loadWishlist();
            } else {
                alert(data.message || 'Error removing from wishlist');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error removing from wishlist');
        });
    };

    window.logout = function() {
        fetch(API_BASE + '/auth/logout', {
            method: 'POST',
            headers: getHeaders()
        })
        .finally(() => {
            localStorage.removeItem('kalses_token');
            localStorage.removeItem('kalses_customer');
            window.location.href = 'page-login.html';
        });
    };

    function updateCartCount() {
        fetch(API_BASE + '/cart', {
            headers: getHeaders()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                const count = data.data.cart?.items?.length || data.data.items?.length || 0;
                document.querySelectorAll('.icon-cart .number-item').forEach(el => {
                    el.textContent = count;
                });
            }
        });
    }
})();
