<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Review;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::with(['category', 'vendor', 'primaryImage', 'images'])
            ->where('is_active', true);

        if ($request->search) {
            $q = $request->search;
            $query->where(function ($q2) use ($q) {
                $q2->where('name', 'like', "%$q%")
                   ->orWhere('description', 'like', "%$q%")
                   ->orWhere('brand', 'like', "%$q%");
            });
        }
        if ($request->category) {
            // Accept either a slug (string) or a category ID (numeric)
            if (is_numeric($request->category)) {
                $query->where('category_id', $request->category);
            } else {
                $query->whereHas('category', fn($q) => $q->where('slug', $request->category));
            }
        }
        if ($request->vendor || $request->vendor_id) {
            $query->where('vendor_id', $request->vendor ?? $request->vendor_id);
        }
        if ($request->featured) {
            $query->where('is_featured', true);
        }
        if ($request->deal) {
            $query->where('is_deal', true);
        }
        if ($request->is_new) {
            $query->where('is_new', true);
        }
        if ($request->best_seller) {
            $query->where('is_best_seller', true);
        }
        if ($request->min_price) {
            $query->where('price', '>=', (float)$request->min_price);
        }
        if ($request->max_price) {
            $query->where('price', '<=', (float)$request->max_price);
        }

        $sort = $request->sort ?? 'newest';
        match ($sort) {
            'price_asc'  => $query->orderBy('price'),
            'price_desc' => $query->orderByDesc('price'),
            'rating'     => $query->orderByDesc('rating'),
            'popular'    => $query->orderByDesc('sales_count'),
            'oldest'     => $query->orderBy('created_at'),
            'name_asc'   => $query->orderBy('name'),
            'name_desc'  => $query->orderByDesc('name'),
            default      => $query->orderByDesc('created_at'),
        };

        $perPage = min((int)($request->per_page ?? 12), 48);
        return response()->json($query->paginate($perPage));
    }

    public function show($slug)
    {
        $product = Product::with(['category', 'vendor', 'images', 'primaryImage', 'reviews.user'])
            ->where('slug', $slug)
            ->where('is_active', true)
            ->firstOrFail();

        $related = Product::with(['primaryImage', 'category', 'vendor'])
            ->where('category_id', $product->category_id)
            ->where('id', '!=', $product->id)
            ->where('is_active', true)
            ->limit(8)
            ->get();

        return response()->json(['product' => $product, 'related' => $related]);
    }

    public function reviews($id)
    {
        $reviews = Review::with('user')
            ->where('product_id', $id)
            ->where('is_approved', true)
            ->orderByDesc('created_at')
            ->paginate(10);
        return response()->json($reviews);
    }

    public function addReview(Request $request, $id)
    {
        $data = $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'title'  => 'nullable|string|max:255',
            'body'   => 'required|string',
        ]);
        $product = Product::findOrFail($id);
        $review  = Review::create([
            'product_id' => $id,
            'user_id'    => $request->user()->id,
            'rating'     => $data['rating'],
            'title'      => $data['title'] ?? null,
            'body'       => $data['body'],
        ]);
        $avg   = Review::where('product_id', $id)->where('is_approved', true)->avg('rating');
        $count = Review::where('product_id', $id)->where('is_approved', true)->count();
        $product->update(['rating' => round($avg, 2), 'reviews_count' => $count]);
        return response()->json($review->load('user'), 201);
    }
}
