<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\CartItem;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Stripe\Stripe;
use Stripe\Checkout\Session as StripeSession;

class PaymentController extends Controller
{
    public function __construct()
    {
        Stripe::setApiKey(config("services.stripe.secret"));
    }

    public function createStripeSession(Request $request)
    {
        $data = $request->validate([
            "shipping_name"    => "required|string",
            "shipping_email"   => "required|email",
            "shipping_phone"   => "nullable|string",
            "shipping_address" => "required|string",
            "shipping_city"    => "required|string",
            "shipping_state"   => "nullable|string",
            "shipping_zip"     => "required|string",
            "shipping_country" => "required|string",
            "notes"            => "nullable|string",
            "coupon_code"      => "nullable|string",
        ]);

        $user      = $request->user();
        $cartItems = CartItem::with("product")->where("user_id", $user->id)->get();

        if ($cartItems->isEmpty()) {
            return response()->json(["message" => "Cart is empty"], 422);
        }

        $lineItems = [];
        foreach ($cartItems as $item) {
            $product = $item->product;
            if (!$product || !$product->is_active) continue;
            $price = (float)($product->sale_price ?? $product->price);
            $lineItems[] = [
                "price_data" => [
                    "currency"     => "usd",
                    "unit_amount"  => (int)round($price * 100),
                    "product_data" => [
                        "name"   => $product->name,
                        "images" => $product->primaryImage ? [$product->primaryImage->url] : [],
                    ],
                ],
                "quantity" => $item->quantity,
            ];
        }

        $frontendUrl = config("app.frontend_url", "http://localhost:5173");

        $session = StripeSession::create([
            "payment_method_types" => ["card"],
            "line_items"           => $lineItems,
            "mode"                 => "payment",
            "customer_email"       => $data["shipping_email"],
            "success_url"          => $frontendUrl . "/checkout/success?session_id={CHECKOUT_SESSION_ID}",
            "cancel_url"           => $frontendUrl . "/checkout",
            "metadata"             => [
                "user_id"          => $user->id,
                "shipping_name"    => $data["shipping_name"],
                "shipping_email"   => $data["shipping_email"],
                "shipping_phone"   => $data["shipping_phone"] ?? "",
                "shipping_address" => $data["shipping_address"],
                "shipping_city"    => $data["shipping_city"],
                "shipping_state"   => $data["shipping_state"] ?? "",
                "shipping_zip"     => $data["shipping_zip"],
                "shipping_country" => $data["shipping_country"],
                "notes"            => $data["notes"] ?? "",
                "coupon_code"      => $data["coupon_code"] ?? "",
            ],
        ]);

        return response()->json(["url" => $session->url, "session_id" => $session->id]);
    }

    public function verifyStripeSession(Request $request)
    {
        $data = $request->validate(["session_id" => "required|string"]);

        try {
            $session = StripeSession::retrieve($data["session_id"]);
        } catch (\Exception $e) {
            return response()->json(["message" => "Invalid session"], 422);
        }

        if ($session->payment_status !== "paid") {
            return response()->json(["message" => "Payment not completed"], 422);
        }

        $existing = Order::where("payment_reference", $session->id)->first();
        if ($existing) {
            return response()->json($existing->load("items"));
        }

        $user      = $request->user();
        $meta      = $session->metadata;
        $cartItems = CartItem::with("product")->where("user_id", $user->id)->get();

        $subtotal   = 0;
        $orderItems = [];
        foreach ($cartItems as $item) {
            $product = $item->product;
            if (!$product) continue;
            $price          = (float)$product->price;
            $salePrice      = $product->sale_price ? (float)$product->sale_price : null;
            $effectivePrice = $salePrice ?? $price;
            $itemSubtotal   = $effectivePrice * $item->quantity;
            $subtotal      += $itemSubtotal;
            $orderItems[]   = [
                "product_id"    => $product->id,
                "product_name"  => $product->name,
                "product_image" => $product->primaryImage?->url,
                "price"         => $price,
                "sale_price"    => $salePrice,
                "quantity"      => $item->quantity,
                "subtotal"      => $itemSubtotal,
            ];
        }

        $total    = $session->amount_total / 100;
        $tax      = round($subtotal * 0.08, 2);
        $discount = 0;
        $shipping = 0;

        $order = Order::create([
            "user_id"           => $user->id,
            "order_number"      => "ORD-" . strtoupper(Str::random(10)),
            "status"            => "processing",
            "subtotal"          => $subtotal,
            "discount"          => $discount,
            "shipping"          => $shipping,
            "tax"               => $tax,
            "total"             => $total,
            "payment_method"    => "stripe",
            "payment_status"    => "paid",
            "payment_reference" => $session->id,
            "shipping_name"     => $meta["shipping_name"] ?? $user->name,
            "shipping_email"    => $meta["shipping_email"] ?? $user->email,
            "shipping_phone"    => $meta["shipping_phone"] ?? null,
            "shipping_address"  => $meta["shipping_address"] ?? "",
            "shipping_city"     => $meta["shipping_city"] ?? "",
            "shipping_state"    => $meta["shipping_state"] ?? null,
            "shipping_zip"      => $meta["shipping_zip"] ?? "",
            "shipping_country"  => $meta["shipping_country"] ?? "US",
            "notes"             => $meta["notes"] ?? null,
            "is_wholesale"      => $user->is_wholesale,
        ]);

        foreach ($orderItems as $orderItem) {
            OrderItem::create(array_merge($orderItem, ["order_id" => $order->id]));
            Product::where("id", $orderItem["product_id"])->decrement("stock", $orderItem["quantity"]);
            Product::where("id", $orderItem["product_id"])->increment("sales_count", $orderItem["quantity"]);
        }

        CartItem::where("user_id", $user->id)->delete();

        try {
            \Mail::to($order->shipping_email)->send(new \App\Mail\OrderConfirmationMail($order));
        } catch (\Exception $e) {
            \Log::error("Order confirmation email failed: " . $e->getMessage());
        }

        return response()->json($order->load("items"), 201);
    }
}
