<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\CartItem;
use App\Models\Coupon;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class OrderController extends Controller
{
    public function index(Request $request)
    {
        $orders = Order::with('items')
            ->where('user_id', $request->user()->id)
            ->orderByDesc('created_at')
            ->paginate(10);
        return response()->json($orders);
    }

    public function show(Request $request, $id)
    {
        $order = Order::with('items')
            ->where('user_id', $request->user()->id)
            ->findOrFail($id);
        return response()->json($order);
    }

    public function checkout(Request $request)
    {
        $data = $request->validate([
            'shipping_name'    => 'required|string',
            'shipping_email'   => 'required|email',
            'shipping_phone'   => 'nullable|string',
            'shipping_address' => 'required|string',
            'shipping_city'    => 'required|string',
            'shipping_state'   => 'nullable|string',
            'shipping_zip'     => 'required|string',
            'shipping_country' => 'required|string',
            'payment_method'   => 'required|in:cod,stripe,bank_transfer',
            'notes'            => 'nullable|string',
            'coupon_code'      => 'nullable|string',
        ]);

        $user      = $request->user();
        $cartItems = CartItem::with('product')->where('user_id', $user->id)->get();

        if ($cartItems->isEmpty()) {
            return response()->json(['message' => 'Cart is empty'], 422);
        }

        $subtotal   = 0;
        $orderItems = [];
        foreach ($cartItems as $item) {
            $product = $item->product;
            if (!$product || !$product->is_active) continue;
            $price          = (float) $product->price;
            $salePrice      = $product->sale_price ? (float) $product->sale_price : null;
            $effectivePrice = $salePrice ?? $price;
            $itemSubtotal   = $effectivePrice * $item->quantity;
            $subtotal      += $itemSubtotal;
            $orderItems[]   = [
                'product_id'    => $product->id,
                'product_name'  => $product->name,
                'product_image' => $product->primaryImage?->url,
                'price'         => $price,
                'sale_price'    => $salePrice,
                'quantity'      => $item->quantity,
                'subtotal'      => $itemSubtotal,
            ];
        }

        $discount   = 0;
        $couponCode = null;
        if (!empty($data['coupon_code'])) {
            $coupon = Coupon::where('code', strtoupper($data['coupon_code']))->first();
            if ($coupon && $coupon->isValid($subtotal)) {
                $discount   = $coupon->calculateDiscount($subtotal);
                $couponCode = $coupon->code;
                $coupon->increment('used_count');
            }
        }

        // Wholesale discount
        if ($user->is_wholesale) {
            $wa = $user->wholesaleApplication;
            if ($wa && $wa->status === 'approved' && $wa->discount_rate > 0) {
                $wsDiscount = $subtotal * ($wa->discount_rate / 100);
                $discount   = max($discount, $wsDiscount);
            }
        }

        $shipping = 0;
        $tax      = round(($subtotal - $discount) * 0.08, 2);
        $total    = $subtotal - $discount + $shipping + $tax;

        $order = Order::create([
            'user_id'          => $user->id,
            'order_number'     => 'ORD-' . strtoupper(Str::random(10)),
            'status'           => 'pending',
            'subtotal'         => $subtotal,
            'discount'         => $discount,
            'shipping'         => $shipping,
            'tax'              => $tax,
            'total'            => $total,
            'coupon_code'      => $couponCode,
            'payment_method'   => $data['payment_method'],
            'payment_status'   => 'pending',
            'shipping_name'    => $data['shipping_name'],
            'shipping_email'   => $data['shipping_email'],
            'shipping_phone'   => $data['shipping_phone'] ?? null,
            'shipping_address' => $data['shipping_address'],
            'shipping_city'    => $data['shipping_city'],
            'shipping_state'   => $data['shipping_state'] ?? null,
            'shipping_zip'     => $data['shipping_zip'],
            'shipping_country' => $data['shipping_country'],
            'notes'            => $data['notes'] ?? null,
            'is_wholesale'     => $user->is_wholesale,
        ]);

        foreach ($orderItems as $orderItem) {
            OrderItem::create(array_merge($orderItem, ['order_id' => $order->id]));
            Product::where('id', $orderItem['product_id'])->decrement('stock', $orderItem['quantity']);
            Product::where('id', $orderItem['product_id'])->increment('sales_count', $orderItem['quantity']);
        }


        CartItem::where('user_id', $user->id)->delete();

        // Send order confirmation email
        try {
            \Mail::to($order->shipping_email)->send(new \App\Mail\OrderConfirmationMail($order));
        } catch (\Exception $e) {
            \Log::error('Order confirmation email failed: ' . $e->getMessage());
        }

        return response()->json($order->load('items'), 201);
    }

    public function cancel(Request $request, $id)
    {
        $order = Order::where('user_id', $request->user()->id)->findOrFail($id);
        if (!in_array($order->status, ['pending', 'processing'])) {
            return response()->json(['message' => 'Cannot cancel this order'], 422);
        }
        $order->update(['status' => 'cancelled']);
        return response()->json($order);
    }
}
