<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\CartItem;
use App\Models\Product;
use Illuminate\Http\Request;

class CartController extends Controller
{
    public function index(Request $request)
    {
        $items    = CartItem::with(['product.primaryImage', 'product.images', 'product.category'])
            ->where('user_id', $request->user()->id)
            ->get();
        $subtotal = $items->sum(fn($i) => ($i->product->sale_price ?? $i->product->price) * $i->quantity);
        return response()->json(['items' => $items, 'subtotal' => $subtotal, 'total' => $subtotal]);
    }

    public function store(Request $request)
    {
        $data    = $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity'   => 'sometimes|integer|min:1|max:100',
        ]);
        $product = Product::findOrFail($data['product_id']);
        $qty     = $data['quantity'] ?? 1;

        $item = CartItem::where('user_id', $request->user()->id)
            ->where('product_id', $data['product_id'])
            ->first();

        if ($item) {
            $item->update(['quantity' => min($item->quantity + $qty, max($product->stock, 999))]);
        } else {
            $item = CartItem::create([
                'user_id'    => $request->user()->id,
                'product_id' => $data['product_id'],
                'quantity'   => $qty,
            ]);
        }
        return response()->json($item->load(['product.primaryImage', 'product.images']), 201);
    }

    public function update(Request $request, $id)
    {
        $data = $request->validate(['quantity' => 'required|integer|min:1|max:100']);
        $item = CartItem::where('id', $id)->where('user_id', $request->user()->id)->firstOrFail();
        $item->update(['quantity' => $data['quantity']]);
        return response()->json($item->load(['product.primaryImage']));
    }

    public function destroy(Request $request, $id)
    {
        CartItem::where('id', $id)->where('user_id', $request->user()->id)->delete();
        return response()->json(['message' => 'Removed']);
    }

    public function clear(Request $request)
    {
        CartItem::where('user_id', $request->user()->id)->delete();
        return response()->json(['message' => 'Cart cleared']);
    }
}
