<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductImage;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ProductAdminController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::with(['category:id,name', 'primaryImage', 'images'])
            ->withCount('orderItems as sales_count');

        if ($s = $request->search) {
            $query->where(function ($q) use ($s) {
                $q->where('name', 'like', "%$s%")
                  ->orWhere('sku', 'like', "%$s%")
                  ->orWhere('brand', 'like', "%$s%");
            });
        }

        if ($cat = $request->category_id) {
            $query->where('category_id', $cat);
        }

        if ($request->status === 'active') {
            $query->where('is_active', true);
        } elseif ($request->status === 'inactive') {
            $query->where('is_active', false);
        }

        return response()->json($query->orderByDesc('created_at')->paginate(20));
    }

    public function show($id)
    {
        return response()->json(
            Product::with(['category:id,name', 'vendor:id,store_name', 'images'])->findOrFail($id)
        );
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name'              => 'required|string|max:255',
            'category_id'       => 'nullable|exists:categories,id',
            'vendor_id'         => 'nullable|exists:vendors,id',
            'price'             => 'required|numeric|min:0',
            'sale_price'        => 'nullable|numeric|min:0',
            'stock'             => 'integer|min:0',
            'sku'               => 'nullable|string|max:100',
            'brand'             => 'nullable|string|max:100',
            'short_description' => 'nullable|string',
            'description'       => 'nullable|string',
            'is_featured'       => 'boolean',
            'is_deal'           => 'boolean',
            'is_new'            => 'boolean',
            'is_best_seller'    => 'boolean',
            'is_active'         => 'boolean',
            'images'            => 'array',
            'images.*'          => 'url',
        ]);

        $images = $data['images'] ?? [];
        unset($data['images']);

        $data['slug'] = Str::slug($data['name']) . '-' . Str::random(6);

        $product = Product::create($data);

        foreach ($images as $i => $url) {
            ProductImage::create([
                'product_id' => $product->id,
                'url'        => $url,
                'is_primary' => $i === 0,
                'sort_order' => $i,
            ]);
        }

        return response()->json($product->load('images'), 201);
    }

    public function update(Request $request, $id)
    {
        $product = Product::findOrFail($id);

        $data = $request->validate([
            'name'              => 'sometimes|required|string|max:255',
            'category_id'       => 'nullable|exists:categories,id',
            'vendor_id'         => 'nullable|exists:vendors,id',
            'price'             => 'sometimes|required|numeric|min:0',
            'sale_price'        => 'nullable|numeric|min:0',
            'stock'             => 'integer|min:0',
            'sku'               => 'nullable|string|max:100',
            'brand'             => 'nullable|string|max:100',
            'short_description' => 'nullable|string',
            'description'       => 'nullable|string',
            'is_featured'       => 'boolean',
            'is_deal'           => 'boolean',
            'is_new'            => 'boolean',
            'is_best_seller'    => 'boolean',
            'is_active'         => 'boolean',
            'images'            => 'array',
            'images.*'          => 'url',
        ]);

        $images = $data['images'] ?? null;
        unset($data['images']);

        $product->update($data);

        if ($images !== null) {
            $product->images()->delete();
            foreach ($images as $i => $url) {
                ProductImage::create([
                    'product_id' => $product->id,
                    'url'        => $url,
                    'is_primary' => $i === 0,
                    'sort_order' => $i,
                ]);
            }
        }

        return response()->json($product->load('images'));
    }

    public function destroy($id)
    {
        $product = Product::findOrFail($id);
        $product->images()->delete();
        $product->delete();

        return response()->json(['message' => 'Product deleted']);
    }
}
