<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\Request;

class OrderAdminController extends Controller
{
    public function index(Request $request)
    {
        $query = Order::with('user:id,name,email')
            ->withCount('items');

        if ($s = $request->search) {
            $query->where(function ($q) use ($s) {
                $q->where('order_number', 'like', "%$s%")
                  ->orWhere('shipping_name', 'like', "%$s%")
                  ->orWhere('shipping_email', 'like', "%$s%")
                  ->orWhereHas('user', fn($u) => $u->where('email', 'like', "%$s%")->orWhere('name', 'like', "%$s%"));
            });
        }

        if ($status = $request->status) {
            $query->where('status', $status);
        }

        return response()->json($query->orderByDesc('created_at')->paginate(20));
    }

    public function show($id)
    {
        return response()->json(
            Order::with([
                'user:id,name,email,phone',
                'items.product.images',
                'items.product.primaryImage',
            ])->findOrFail($id)
        );
    }

    public function updateStatus(Request $request, $id)
    {
        $order = Order::findOrFail($id);

        $data = $request->validate([
            'status'         => 'sometimes|in:pending,processing,shipped,delivered,cancelled,refunded',
            'payment_status' => 'sometimes|in:pending,paid,failed,refunded',
        ]);


        $order->update($data);

        // Send status update email
        if (isset($data['status'])) {
            try {
                \Mail::to($order->shipping_email)->send(new \App\Mail\OrderStatusMail($order, $data['status']));
            } catch (\Exception $e) {
                \Log::error('Order status email failed: ' . $e->getMessage());
            }
        }

        return response()->json($order);
    }
}
