<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CategoryAdminController extends Controller
{
    public function index()
    {
        return response()->json(
            Category::with('parent:id,name')->withCount('products')->orderBy('name')->get()
        );
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name'        => 'required|string|max:255',
            'slug'        => 'nullable|string|max:255|unique:categories,slug',
            'parent_id'   => 'nullable|exists:categories,id',
            'description' => 'nullable|string',
            'image'       => 'nullable|string',
            'is_active'   => 'boolean',
        ]);

        $data['slug'] = $data['slug'] ?: Str::slug($data['name']);

        return response()->json(Category::create($data), 201);
    }

    public function update(Request $request, $id)
    {
        $cat  = Category::findOrFail($id);
        $data = $request->validate([
            'name'        => 'sometimes|required|string|max:255',
            'slug'        => 'nullable|string|max:255|unique:categories,slug,' . $id,
            'parent_id'   => 'nullable|exists:categories,id',
            'description' => 'nullable|string',
            'image'       => 'nullable|string',
            'is_active'   => 'boolean',
        ]);

        if (isset($data['name']) && empty($data['slug'])) {
            $data['slug'] = Str::slug($data['name']);
        }

        $cat->update($data);

        return response()->json($cat);
    }

    public function destroy($id)
    {
        $cat = Category::findOrFail($id);
        // Move children to top-level
        Category::where('parent_id', $id)->update(['parent_id' => null]);
        $cat->delete();

        return response()->json(['message' => 'Category deleted']);
    }
}
