import { useState, useEffect, useCallback } from 'react'
import toast from 'react-hot-toast'
import { adminGetWholesale, adminUpdateWholesale } from '../services/api'

const Wholesale = () => {
  const [apps, setApps] = useState({ data: [], total: 0, last_page: 1 })
  const [filters, setFilters] = useState({ status: '', page: 1 })
  const [loading, setLoading] = useState(true)
  const [acting, setActing] = useState(null)
  const [detail, setDetail] = useState(null)

  const load = useCallback(async () => {
    setLoading(true)
    try {
      const r = await adminGetWholesale(filters)
      setApps(r.data)
    } catch { toast.error('Failed to load applications') }
    finally { setLoading(false) }
  }, [filters])

  useEffect(() => { load() }, [load])

  const handleAction = async (id, status) => {
    setActing(id)
    try {
      await adminUpdateWholesale(id, { status })
      toast.success(`Application ${status}!`)
      setDetail(null)
      load()
    } catch { toast.error('Action failed') }
    finally { setActing(null) }
  }

  return (
    <div>
      {/* Detail Modal */}
      {detail && (
        <div className="modal d-block" style={{ background: 'rgba(0,0,0,.5)' }}>
          <div className="modal-dialog modal-dialog-centered modal-lg">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Wholesale Application — {detail.business_name}</h5>
                <button className="btn-close" onClick={() => setDetail(null)}></button>
              </div>
              <div className="modal-body">
                <div className="row g-3">
                  <div className="col-md-6">
                    <div className="mb-3">
                      <div className="fw-semibold text-muted mb-1" style={{ fontSize: '.8rem' }}>APPLICANT</div>
                      <div>{detail.user?.name || '—'}</div>
                      <div className="text-muted">{detail.user?.email || '—'}</div>
                    </div>
                    <div className="mb-3">
                      <div className="fw-semibold text-muted mb-1" style={{ fontSize: '.8rem' }}>BUSINESS NAME</div>
                      <div>{detail.business_name}</div>
                    </div>
                    <div className="mb-3">
                      <div className="fw-semibold text-muted mb-1" style={{ fontSize: '.8rem' }}>BUSINESS TYPE</div>
                      <div>{detail.business_type || '—'}</div>
                    </div>
                    <div className="mb-3">
                      <div className="fw-semibold text-muted mb-1" style={{ fontSize: '.8rem' }}>TAX ID / REG #</div>
                      <div>{detail.tax_id || '—'}</div>
                    </div>
                  </div>
                  <div className="col-md-6">
                    <div className="mb-3">
                      <div className="fw-semibold text-muted mb-1" style={{ fontSize: '.8rem' }}>PHONE</div>
                      <div>{detail.phone || '—'}</div>
                    </div>
                    <div className="mb-3">
                      <div className="fw-semibold text-muted mb-1" style={{ fontSize: '.8rem' }}>ADDRESS</div>
                      <div>{detail.address || '—'}</div>
                    </div>
                    <div className="mb-3">
                      <div className="fw-semibold text-muted mb-1" style={{ fontSize: '.8rem' }}>MONTHLY ORDER VALUE</div>
                      <div>{detail.monthly_order_value || '—'}</div>
                    </div>
                    <div className="mb-3">
                      <div className="fw-semibold text-muted mb-1" style={{ fontSize: '.8rem' }}>SUBMITTED</div>
                      <div>{new Date(detail.created_at).toLocaleString()}</div>
                    </div>
                  </div>
                  {detail.notes && (
                    <div className="col-12">
                      <div className="fw-semibold text-muted mb-1" style={{ fontSize: '.8rem' }}>NOTES / MESSAGE</div>
                      <div className="p-3 bg-light rounded">{detail.notes}</div>
                    </div>
                  )}
                </div>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setDetail(null)}>Close</button>
                {detail.status === 'pending' && (
                  <>
                    <button className="btn btn-danger" onClick={() => handleAction(detail.id, 'rejected')} disabled={acting === detail.id}>
                      {acting === detail.id ? <span className="spinner-border spinner-border-sm me-1"></span> : <i className="bi bi-x-circle me-1"></i>}
                      Reject
                    </button>
                    <button className="btn btn-success" onClick={() => handleAction(detail.id, 'approved')} disabled={acting === detail.id}>
                      {acting === detail.id ? <span className="spinner-border spinner-border-sm me-1"></span> : <i className="bi bi-check-circle me-1"></i>}
                      Approve
                    </button>
                  </>
                )}
              </div>
            </div>
          </div>
        </div>
      )}

      <div className="d-flex align-items-center justify-content-between mb-4">
        <div>
          <h4 className="fw-bold mb-1">Wholesale Applications</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>{apps.total || 0} applications</p>
        </div>
      </div>

      <div className="page-card mb-4 p-3">
        <div className="row g-2">
          <div className="col-md-4">
            <select className="form-select" value={filters.status} onChange={e => setFilters(f => ({ ...f, status: e.target.value, page: 1 }))}>
              <option value="">All Status</option>
              <option value="pending">Pending</option>
              <option value="approved">Approved</option>
              <option value="rejected">Rejected</option>
            </select>
          </div>
          <div className="col-md-2">
            <button className="btn btn-outline-secondary w-100" onClick={() => setFilters({ status: '', page: 1 })}>Reset</button>
          </div>
        </div>
      </div>

      <div className="page-card">
        {loading ? (
          <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>
        ) : (apps.data || []).length === 0 ? (
          <div className="empty-state">
            <i className="bi bi-building d-block mb-2" style={{ fontSize: '2rem', color: '#adb5bd' }}></i>
            <p>No wholesale applications</p>
          </div>
        ) : (
          <div className="table-responsive">
            <table className="admin-table">
              <thead>
                <tr><th>#</th><th>Applicant</th><th>Business</th><th>Type</th><th>Status</th><th>Date</th><th>Actions</th></tr>
              </thead>
              <tbody>
                {(apps.data || []).map(a => (
                  <tr key={a.id}>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{a.id}</td>
                    <td>
                      <div style={{ fontWeight: 600, fontSize: '.875rem' }}>{a.user?.name || '—'}</div>
                      <div className="text-muted" style={{ fontSize: '.75rem' }}>{a.user?.email || '—'}</div>
                    </td>
                    <td style={{ fontWeight: 600, fontSize: '.875rem' }}>{a.business_name}</td>
                    <td className="text-muted">{a.business_type || '—'}</td>
                    <td>
                      <span className={`badge-status badge-${a.status === 'approved' ? 'active' : a.status === 'rejected' ? 'inactive' : 'pending'}`}>
                        {a.status}
                      </span>
                    </td>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{new Date(a.created_at).toLocaleDateString()}</td>
                    <td>
                      <div className="action-btns">
                        <button className="btn btn-sm btn-outline-primary" onClick={() => setDetail(a)} title="View">
                          <i className="bi bi-eye"></i>
                        </button>
                        {a.status === 'pending' && (
                          <>
                            <button className="btn btn-sm btn-outline-success" onClick={() => handleAction(a.id, 'approved')} disabled={acting === a.id} title="Approve">
                              {acting === a.id ? <span className="spinner-border spinner-border-sm"></span> : <i className="bi bi-check-lg"></i>}
                            </button>
                            <button className="btn btn-sm btn-outline-danger" onClick={() => handleAction(a.id, 'rejected')} disabled={acting === a.id} title="Reject">
                              <i className="bi bi-x-lg"></i>
                            </button>
                          </>
                        )}
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
        {apps.last_page > 1 && (
          <div className="d-flex justify-content-center py-3">
            <div className="admin-pagination">
              <button disabled={filters.page <= 1} onClick={() => setFilters(f => ({ ...f, page: f.page - 1 }))}><i className="bi bi-chevron-left"></i></button>
              {Array.from({ length: apps.last_page }, (_, i) => i + 1).map(p => (
                <button key={p} className={filters.page === p ? 'active' : ''} onClick={() => setFilters(f => ({ ...f, page: p }))}>{p}</button>
              ))}
              <button disabled={filters.page >= apps.last_page} onClick={() => setFilters(f => ({ ...f, page: f.page + 1 }))}><i className="bi bi-chevron-right"></i></button>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}
export default Wholesale
