import { useState, useEffect, useCallback } from 'react'
import toast from 'react-hot-toast'
import { adminGetVendors, adminCreateVendor, adminUpdateVendor, adminDeleteVendor } from '../services/api'
import Confirm from '../components/ui/Confirm'
import ImageUpload from '../components/ui/ImageUpload'

const EMPTY = { store_name: '', email: '', phone: '', description: '', address: '', logo: '', banner: '', commission_rate: '10', is_active: true }

const Vendors = () => {
  const [vendors, setVendors] = useState({ data: [], total: 0, last_page: 1 })
  const [filters, setFilters] = useState({ search: '', page: 1 })
  const [loading, setLoading] = useState(true)
  const [modal, setModal] = useState(false)
  const [form, setForm] = useState(EMPTY)
  const [editId, setEditId] = useState(null)
  const [saving, setSaving] = useState(false)
  const [confirm, setConfirm] = useState(null)

  const load = useCallback(async () => {
    setLoading(true)
    try {
      const r = await adminGetVendors(filters)
      setVendors(r.data)
    } catch { toast.error('Failed to load vendors') }
    finally { setLoading(false) }
  }, [filters])

  useEffect(() => { load() }, [load])

  const openCreate = () => { setForm(EMPTY); setEditId(null); setModal(true) }
  const openEdit = (v) => {
    setForm({
      store_name: v.store_name, email: v.email || '', phone: v.phone || '',
      description: v.description || '', address: v.address || '',
      logo: v.logo || '', banner: v.banner || '',
      commission_rate: v.commission_rate || '10', is_active: !!v.is_active
    })
    setEditId(v.id)
    setModal(true)
  }

  const set = (k, v) => setForm(f => ({ ...f, [k]: v }))

  const handleSubmit = async (e) => {
    e.preventDefault()
    setSaving(true)
    try {
      if (editId) await adminUpdateVendor(editId, form)
      else await adminCreateVendor(form)
      toast.success(editId ? 'Vendor updated!' : 'Vendor created!')
      setModal(false)
      load()
    } catch (err) {
      toast.error(err.response?.data?.message || 'Save failed')
    } finally { setSaving(false) }
  }

  const handleDelete = async (id) => {
    try {
      await adminDeleteVendor(id)
      toast.success('Vendor deleted')
      load()
    } catch { toast.error('Delete failed') }
    setConfirm(null)
  }

  return (
    <div>
      <Confirm show={!!confirm} message="Delete this vendor? All their products will be unlinked." onConfirm={() => handleDelete(confirm)} onCancel={() => setConfirm(null)} />

      {modal && (
        <div className="modal d-block" style={{ background: 'rgba(0,0,0,.5)' }}>
          <div className="modal-dialog modal-lg modal-dialog-centered modal-dialog-scrollable">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">{editId ? 'Edit Vendor' : 'New Vendor'}</h5>
                <button className="btn-close" onClick={() => setModal(false)}></button>
              </div>
              <form onSubmit={handleSubmit}>
                <div className="modal-body">
                  <div className="row g-3">
                    <div className="col-md-6">
                      <label className="form-label fw-semibold">Store Name *</label>
                      <input className="form-control" value={form.store_name} onChange={e => set('store_name', e.target.value)} required />
                    </div>
                    <div className="col-md-6">
                      <label className="form-label fw-semibold">Email</label>
                      <input className="form-control" type="email" value={form.email} onChange={e => set('email', e.target.value)} />
                    </div>
                    <div className="col-md-6">
                      <label className="form-label fw-semibold">Phone</label>
                      <input className="form-control" value={form.phone} onChange={e => set('phone', e.target.value)} />
                    </div>
                    <div className="col-md-6">
                      <label className="form-label fw-semibold">Commission Rate (%)</label>
                      <div className="input-group">
                        <input className="form-control" type="number" min="0" max="100" step="0.1" value={form.commission_rate} onChange={e => set('commission_rate', e.target.value)} />
                        <span className="input-group-text">%</span>
                      </div>
                    </div>
                    <div className="col-12">
                      <label className="form-label fw-semibold">Address</label>
                      <input className="form-control" value={form.address} onChange={e => set('address', e.target.value)} />
                    </div>
                    <div className="col-12">
                      <label className="form-label fw-semibold">Description</label>
                      <textarea className="form-control" rows={3} value={form.description} onChange={e => set('description', e.target.value)} />
                    </div>
                    <div className="col-md-6">
                      <ImageUpload value={form.logo} onChange={v => set('logo', v)} label="Logo" height={110} />
                    </div>
                    <div className="col-md-6">
                      <ImageUpload value={form.banner} onChange={v => set('banner', v)} label="Banner" height={110} />
                    </div>
                    <div className="col-12">
                      <div className="form-check form-switch">
                        <input className="form-check-input" type="checkbox" checked={form.is_active} onChange={e => set('is_active', e.target.checked)} id="v_active" />
                        <label className="form-check-label" htmlFor="v_active">Active</label>
                      </div>
                    </div>
                  </div>
                </div>
                <div className="modal-footer">
                  <button type="button" className="btn btn-secondary" onClick={() => setModal(false)}>Cancel</button>
                  <button type="submit" className="btn btn-primary" disabled={saving}>
                    {saving ? <span className="spinner-border spinner-border-sm me-1"></span> : null}
                    {editId ? 'Update' : 'Create'}
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      )}

      <div className="d-flex align-items-center justify-content-between mb-4">
        <div>
          <h4 className="fw-bold mb-1">Vendors</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>{vendors.total || 0} vendors</p>
        </div>
        <button className="btn btn-primary" onClick={openCreate}>
          <i className="bi bi-plus-circle me-2"></i>Add Vendor
        </button>
      </div>

      <div className="page-card mb-4 p-3">
        <div className="row g-2">
          <div className="col-md-6">
            <div style={{ position: 'relative' }}>
              <i className="bi bi-search" style={{ position: 'absolute', left: 12, top: '50%', transform: 'translateY(-50%)', color: '#adb5bd' }}></i>
              <input className="form-control" style={{ paddingLeft: 36 }} placeholder="Search vendors..."
                value={filters.search} onChange={e => setFilters(f => ({ ...f, search: e.target.value, page: 1 }))} />
            </div>
          </div>
          <div className="col-md-2">
            <button className="btn btn-outline-secondary w-100" onClick={() => setFilters({ search: '', page: 1 })}>Reset</button>
          </div>
        </div>
      </div>

      <div className="page-card">
        {loading ? (
          <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>
        ) : (vendors.data || []).length === 0 ? (
          <div className="empty-state">
            <i className="bi bi-shop d-block mb-2" style={{ fontSize: '2rem', color: '#adb5bd' }}></i>
            <p>No vendors yet</p>
            <button className="btn btn-primary btn-sm" onClick={openCreate}>Add First Vendor</button>
          </div>
        ) : (
          <div className="table-responsive">
            <table className="admin-table">
              <thead>
                <tr><th>#</th><th>Vendor</th><th>Contact</th><th>Commission</th><th>Status</th><th>Actions</th></tr>
              </thead>
              <tbody>
                {(vendors.data || []).map(v => (
                  <tr key={v.id}>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{v.id}</td>
                    <td>
                      <div className="d-flex align-items-center gap-3">
                        {v.logo && <img src={v.logo} style={{ width: 36, height: 36, objectFit: 'cover', borderRadius: 6 }} onError={e => { e.target.style.display = 'none' }} alt="" />}
                        <div>
                          <div style={{ fontWeight: 600, fontSize: '.875rem' }}>{v.store_name}</div>
                          {v.address && <div className="text-muted" style={{ fontSize: '.75rem' }}>{v.address}</div>}
                        </div>
                      </div>
                    </td>
                    <td>
                      <div style={{ fontSize: '.875rem' }}>{v.email || '—'}</div>
                      <div className="text-muted" style={{ fontSize: '.75rem' }}>{v.phone || ''}</div>
                    </td>
                    <td><span className="badge bg-light text-dark">{v.commission_rate || 0}%</span></td>
                    <td><span className={`badge-status badge-${v.is_active ? 'active' : 'inactive'}`}>{v.is_active ? 'Active' : 'Inactive'}</span></td>
                    <td>
                      <div className="action-btns">
                        <button className="btn btn-sm btn-outline-primary" onClick={() => openEdit(v)}><i className="bi bi-pencil"></i></button>
                        <button className="btn btn-sm btn-outline-danger" onClick={() => setConfirm(v.id)}><i className="bi bi-trash"></i></button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>
    </div>
  )
}
export default Vendors
