import { useState, useEffect, useCallback } from 'react'
import toast from 'react-hot-toast'
import { adminGetUsers, adminToggleAdmin } from '../services/api'

const Users = () => {
  const [users, setUsers] = useState({ data: [], total: 0, current_page: 1, last_page: 1 })
  const [filters, setFilters] = useState({ search: '', page: 1 })
  const [loading, setLoading] = useState(true)
  const [toggling, setToggling] = useState(null)

  const load = useCallback(async () => {
    setLoading(true)
    try {
      const r = await adminGetUsers(filters)
      setUsers(r.data)
    } catch { toast.error('Failed to load users') }
    finally { setLoading(false) }
  }, [filters])

  useEffect(() => { load() }, [load])

  const handleToggleAdmin = async (id) => {
    setToggling(id)
    try {
      await adminToggleAdmin(id)
      toast.success('Admin status updated')
      load()
    } catch { toast.error('Update failed') }
    finally { setToggling(null) }
  }

  return (
    <div>
      <div className="d-flex align-items-center justify-content-between mb-4">
        <div>
          <h4 className="fw-bold mb-1">Users</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>{users.total} registered users</p>
        </div>
      </div>

      <div className="page-card mb-4 p-3">
        <div className="row g-2">
          <div className="col-md-6">
            <div style={{ position: 'relative' }}>
              <i className="bi bi-search" style={{ position: 'absolute', left: 12, top: '50%', transform: 'translateY(-50%)', color: '#adb5bd' }}></i>
              <input className="form-control" style={{ paddingLeft: 36 }} placeholder="Search by name or email..."
                value={filters.search} onChange={e => setFilters(f => ({ ...f, search: e.target.value, page: 1 }))} />
            </div>
          </div>
          <div className="col-md-2">
            <button className="btn btn-outline-secondary w-100" onClick={() => setFilters({ search: '', page: 1 })}>Reset</button>
          </div>
        </div>
      </div>

      <div className="page-card">
        {loading ? (
          <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>
        ) : users.data.length === 0 ? (
          <div className="empty-state">
            <i className="bi bi-people d-block mb-2" style={{ fontSize: '2rem', color: '#adb5bd' }}></i>
            <p>No users found</p>
          </div>
        ) : (
          <div className="table-responsive">
            <table className="admin-table">
              <thead>
                <tr><th>#</th><th>User</th><th>Phone</th><th>Roles</th><th>Joined</th><th>Actions</th></tr>
              </thead>
              <tbody>
                {users.data.map(u => (
                  <tr key={u.id}>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{u.id}</td>
                    <td>
                      <div>
                        <div style={{ fontWeight: 600, fontSize: '.875rem' }}>{u.name}</div>
                        <div className="text-muted" style={{ fontSize: '.75rem' }}>{u.email}</div>
                      </div>
                    </td>
                    <td className="text-muted">{u.phone || '—'}</td>
                    <td>
                      <div className="d-flex gap-1 flex-wrap">
                        {u.is_admin && <span className="badge bg-danger">Admin</span>}
                        {u.is_vendor && <span className="badge bg-info text-dark">Vendor</span>}
                        {u.is_wholesale && <span className="badge bg-warning text-dark">Wholesale</span>}
                        {!u.is_admin && !u.is_vendor && !u.is_wholesale && <span className="badge bg-light text-dark">Customer</span>}
                      </div>
                    </td>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{new Date(u.created_at).toLocaleDateString()}</td>
                    <td>
                      <button
                        className={`btn btn-sm ${u.is_admin ? 'btn-outline-danger' : 'btn-outline-warning'}`}
                        onClick={() => handleToggleAdmin(u.id)}
                        disabled={toggling === u.id}
                        title={u.is_admin ? 'Remove admin' : 'Make admin'}
                      >
                        {toggling === u.id
                          ? <span className="spinner-border spinner-border-sm"></span>
                          : <><i className={`bi ${u.is_admin ? 'bi-shield-x' : 'bi-shield-check'} me-1`}></i>{u.is_admin ? 'Remove Admin' : 'Make Admin'}</>
                        }
                      </button>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
        {users.last_page > 1 && (
          <div className="d-flex justify-content-center py-3">
            <div className="admin-pagination">
              <button disabled={filters.page <= 1} onClick={() => setFilters(f => ({ ...f, page: f.page - 1 }))}>
                <i className="bi bi-chevron-left"></i>
              </button>
              {Array.from({ length: users.last_page }, (_, i) => i + 1).map(p => (
                <button key={p} className={filters.page === p ? 'active' : ''} onClick={() => setFilters(f => ({ ...f, page: p }))}>{p}</button>
              ))}
              <button disabled={filters.page >= users.last_page} onClick={() => setFilters(f => ({ ...f, page: f.page + 1 }))}>
                <i className="bi bi-chevron-right"></i>
              </button>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}
export default Users
