import { useState, useEffect, useCallback } from 'react'
import toast from 'react-hot-toast'
import { adminGetSliders, adminCreateSlider, adminUpdateSlider, adminDeleteSlider } from '../services/api'
import Confirm from '../components/ui/Confirm'
import ImageUpload from '../components/ui/ImageUpload'

const EMPTY = { title: '', subtitle: '', image: '', link: '', button_text: '', sort_order: 0, is_active: true }

const Sliders = () => {
  const [sliders, setSliders] = useState([])
  const [loading, setLoading] = useState(true)
  const [modal, setModal] = useState(false)
  const [form, setForm] = useState(EMPTY)
  const [editId, setEditId] = useState(null)
  const [saving, setSaving] = useState(false)
  const [confirm, setConfirm] = useState(null)

  const load = useCallback(async () => {
    setLoading(true)
    try {
      const r = await adminGetSliders()
      setSliders(r.data)
    } catch { toast.error('Failed to load sliders') }
    finally { setLoading(false) }
  }, [])

  useEffect(() => { load() }, [load])

  const openCreate = () => { setForm(EMPTY); setEditId(null); setModal(true) }
  const openEdit = (s) => {
    setForm({ title: s.title || '', subtitle: s.subtitle || '', image: s.image || '', link: s.link || '', button_text: s.button_text || '', sort_order: s.sort_order || 0, is_active: !!s.is_active })
    setEditId(s.id)
    setModal(true)
  }

  const set = (k, v) => setForm(f => ({ ...f, [k]: v }))

  const handleSubmit = async (e) => {
    e.preventDefault()
    setSaving(true)
    try {
      if (editId) await adminUpdateSlider(editId, form)
      else await adminCreateSlider(form)
      toast.success(editId ? 'Slider updated!' : 'Slider created!')
      setModal(false)
      load()
    } catch (err) {
      toast.error(err.response?.data?.message || 'Save failed')
    } finally { setSaving(false) }
  }

  const handleDelete = async (id) => {
    try {
      await adminDeleteSlider(id)
      toast.success('Slider deleted')
      load()
    } catch { toast.error('Delete failed') }
    setConfirm(null)
  }

  return (
    <div>
      <Confirm show={!!confirm} message="Delete this slider?" onConfirm={() => handleDelete(confirm)} onCancel={() => setConfirm(null)} />

      {modal && (
        <div className="modal d-block" style={{ background: 'rgba(0,0,0,.5)' }}>
          <div className="modal-dialog modal-dialog-centered">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">{editId ? 'Edit Slider' : 'New Slider'}</h5>
                <button className="btn-close" onClick={() => setModal(false)}></button>
              </div>
              <form onSubmit={handleSubmit}>
                <div className="modal-body">
                  <div className="mb-3">
                    <ImageUpload value={form.image} onChange={v => set('image', v)} label="Image" required height={150} />
                  </div>
                  <div className="mb-3">
                    <label className="form-label fw-semibold">Title</label>
                    <input className="form-control" value={form.title} onChange={e => set('title', e.target.value)} />
                  </div>
                  <div className="mb-3">
                    <label className="form-label fw-semibold">Subtitle</label>
                    <input className="form-control" value={form.subtitle} onChange={e => set('subtitle', e.target.value)} />
                  </div>
                  <div className="row g-3 mb-3">
                    <div className="col-md-8">
                      <label className="form-label fw-semibold">Link URL</label>
                      <input className="form-control" value={form.link} onChange={e => set('link', e.target.value)} placeholder="/shop" />
                    </div>
                    <div className="col-md-4">
                      <label className="form-label fw-semibold">Button Text</label>
                      <input className="form-control" value={form.button_text} onChange={e => set('button_text', e.target.value)} placeholder="Shop Now" />
                    </div>
                  </div>
                  <div className="row g-3 mb-3">
                    <div className="col-md-6">
                      <label className="form-label fw-semibold">Sort Order</label>
                      <input className="form-control" type="number" min="0" value={form.sort_order} onChange={e => set('sort_order', parseInt(e.target.value) || 0)} />
                    </div>
                  </div>
                  <div className="form-check form-switch">
                    <input className="form-check-input" type="checkbox" checked={form.is_active} onChange={e => set('is_active', e.target.checked)} id="sl_active" />
                    <label className="form-check-label" htmlFor="sl_active">Active</label>
                  </div>
                </div>
                <div className="modal-footer">
                  <button type="button" className="btn btn-secondary" onClick={() => setModal(false)}>Cancel</button>
                  <button type="submit" className="btn btn-primary" disabled={saving}>
                    {saving ? <span className="spinner-border spinner-border-sm me-1"></span> : null}
                    {editId ? 'Update' : 'Create'}
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      )}

      <div className="d-flex align-items-center justify-content-between mb-4">
        <div>
          <h4 className="fw-bold mb-1">Sliders</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>{sliders.length} slider{sliders.length !== 1 ? 's' : ''}</p>
        </div>
        <button className="btn btn-primary" onClick={openCreate}>
          <i className="bi bi-plus-circle me-2"></i>Add Slider
        </button>
      </div>

      <div className="page-card">
        {loading ? (
          <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>
        ) : sliders.length === 0 ? (
          <div className="empty-state">
            <i className="bi bi-images d-block mb-2" style={{ fontSize: '2rem', color: '#adb5bd' }}></i>
            <p>No sliders yet</p>
            <button className="btn btn-primary btn-sm" onClick={openCreate}>Add First Slider</button>
          </div>
        ) : (
          <div className="table-responsive">
            <table className="admin-table">
              <thead>
                <tr><th>#</th><th>Preview</th><th>Title</th><th>Link</th><th>Order</th><th>Status</th><th>Actions</th></tr>
              </thead>
              <tbody>
                {sliders.map(s => (
                  <tr key={s.id}>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{s.id}</td>
                    <td>
                      {s.image && (
                        <img src={s.image} style={{ width: 80, height: 44, objectFit: 'cover', borderRadius: 6 }}
                          onError={e => { e.target.style.display = 'none' }} alt="" />
                      )}
                    </td>
                    <td>
                      <div style={{ fontWeight: 600, fontSize: '.875rem' }}>{s.title || '(no title)'}</div>
                      {s.subtitle && <div className="text-muted" style={{ fontSize: '.75rem' }}>{s.subtitle}</div>}
                    </td>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{s.link || '—'}</td>
                    <td><span className="badge bg-light text-dark">{s.sort_order}</span></td>
                    <td><span className={`badge-status badge-${s.is_active ? 'active' : 'inactive'}`}>{s.is_active ? 'Active' : 'Inactive'}</span></td>
                    <td>
                      <div className="action-btns">
                        <button className="btn btn-sm btn-outline-primary" onClick={() => openEdit(s)}><i className="bi bi-pencil"></i></button>
                        <button className="btn btn-sm btn-outline-danger" onClick={() => setConfirm(s.id)}><i className="bi bi-trash"></i></button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>
    </div>
  )
}
export default Sliders
