import { useState, useEffect } from 'react'
import toast from 'react-hot-toast'
import { adminGetSettings, adminUpdateSettings } from '../services/api'
import ImageUpload from '../components/ui/ImageUpload'

const Settings = () => {
  const [settings, setSettings] = useState({})
  const [loading, setLoading] = useState(true)
  const [saving, setSaving] = useState(false)

  useEffect(() => {
    adminGetSettings()
      .then(r => setSettings(r.data || {}))
      .catch(() => toast.error('Failed to load settings'))
      .finally(() => setLoading(false))
  }, [])

  const set = (k, v) => setSettings(s => ({ ...s, [k]: v }))

  const handleSubmit = async (e) => {
    e.preventDefault()
    setSaving(true)
    try {
      await adminUpdateSettings(settings)
      toast.success('Settings saved!')
    } catch { toast.error('Save failed') }
    finally { setSaving(false) }
  }

  if (loading) return <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>

  return (
    <div>
      <div className="mb-4">
        <h4 className="fw-bold mb-1">Settings</h4>
        <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>Manage your store configuration</p>
      </div>

      <form onSubmit={handleSubmit}>
        <div className="row">
          <div className="col-lg-8">
            {/* Store Info */}
            <div className="form-card mb-4">
              <div className="form-section-title">Store Information</div>
              <div className="row g-3">
                <div className="col-md-6">
                  <label className="form-label fw-semibold">Store Name</label>
                  <input className="form-control" value={settings.store_name || ''} onChange={e => set('store_name', e.target.value)} placeholder="Kalses Store" />
                </div>
                <div className="col-md-6">
                  <label className="form-label fw-semibold">Store Email</label>
                  <input className="form-control" type="email" value={settings.store_email || ''} onChange={e => set('store_email', e.target.value)} placeholder="info@store.com" />
                </div>
                <div className="col-md-6">
                  <label className="form-label fw-semibold">Store Phone</label>
                  <input className="form-control" value={settings.store_phone || ''} onChange={e => set('store_phone', e.target.value)} placeholder="+1 234 567 8900" />
                </div>
                <div className="col-md-6">
                  <label className="form-label fw-semibold">Currency</label>
                  <select className="form-select" value={settings.currency || 'USD'} onChange={e => set('currency', e.target.value)}>
                    <option value="USD">USD ($)</option>
                    <option value="EUR">EUR (€)</option>
                    <option value="GBP">GBP (£)</option>
                    <option value="PHP">PHP (₱)</option>
                    <option value="IDR">IDR (Rp)</option>
                  </select>
                </div>
                <div className="col-12">
                  <label className="form-label fw-semibold">Store Address</label>
                  <textarea className="form-control" rows={2} value={settings.store_address || ''} onChange={e => set('store_address', e.target.value)} />
                </div>
                <div className="col-12">
                  <ImageUpload value={settings.logo_url || ''} onChange={v => set('logo_url', v)} label="Store Logo" height={120} />
                </div>
              </div>
            </div>

            {/* Shipping */}
            <div className="form-card mb-4">
              <div className="form-section-title">Shipping</div>
              <div className="row g-3">
                <div className="col-md-6">
                  <label className="form-label fw-semibold">Default Shipping Cost</label>
                  <div className="input-group">
                    <span className="input-group-text">$</span>
                    <input className="form-control" type="number" step="0.01" min="0" value={settings.shipping_cost || ''} onChange={e => set('shipping_cost', e.target.value)} placeholder="5.99" />
                  </div>
                </div>
                <div className="col-md-6">
                  <label className="form-label fw-semibold">Free Shipping Threshold</label>
                  <div className="input-group">
                    <span className="input-group-text">$</span>
                    <input className="form-control" type="number" step="0.01" min="0" value={settings.free_shipping_min || ''} onChange={e => set('free_shipping_min', e.target.value)} placeholder="50.00" />
                  </div>
                  <small className="text-muted">Leave blank to disable free shipping</small>
                </div>
              </div>
            </div>

            {/* Tax */}
            <div className="form-card mb-4">
              <div className="form-section-title">Tax</div>
              <div className="row g-3">
                <div className="col-md-6">
                  <label className="form-label fw-semibold">Tax Rate (%)</label>
                  <div className="input-group">
                    <input className="form-control" type="number" step="0.01" min="0" max="100" value={settings.tax_rate || ''} onChange={e => set('tax_rate', e.target.value)} placeholder="0" />
                    <span className="input-group-text">%</span>
                  </div>
                </div>
                <div className="col-md-6 d-flex align-items-end">
                  <div className="form-check form-switch">
                    <input className="form-check-input" type="checkbox" checked={!!settings.tax_inclusive} onChange={e => set('tax_inclusive', e.target.checked)} id="tax_inc" />
                    <label className="form-check-label" htmlFor="tax_inc">Prices include tax</label>
                  </div>
                </div>
              </div>
            </div>

            {/* Social */}
            <div className="form-card mb-4">
              <div className="form-section-title">Social Media</div>
              <div className="row g-3">
                {[['facebook_url', 'Facebook URL', 'bi-facebook'], ['instagram_url', 'Instagram URL', 'bi-instagram'], ['twitter_url', 'Twitter/X URL', 'bi-twitter-x'], ['youtube_url', 'YouTube URL', 'bi-youtube']].map(([k, l, icon]) => (
                  <div className="col-md-6" key={k}>
                    <label className="form-label fw-semibold"><i className={`bi ${icon} me-1`}></i>{l}</label>
                    <input className="form-control" value={settings[k] || ''} onChange={e => set(k, e.target.value)} placeholder="https://..." />
                  </div>
                ))}
              </div>
            </div>

            {/* Features */}
            <div className="form-card mb-4">
              <div className="form-section-title">Store Features</div>
              <div className="row g-3">
                {[
                  ['enable_reviews', 'Enable Product Reviews'],
                  ['enable_wishlist', 'Enable Wishlist'],
                  ['enable_compare', 'Enable Compare'],
                  ['enable_wholesale', 'Enable Wholesale'],
                  ['enable_vendor', 'Enable Vendor Marketplace'],
                  ['maintenance_mode', 'Maintenance Mode'],
                ].map(([k, l]) => (
                  <div className="col-md-6" key={k}>
                    <div className="form-check form-switch">
                      <input className="form-check-input" type="checkbox" checked={!!settings[k]} onChange={e => set(k, e.target.checked)} id={k} />
                      <label className="form-check-label" htmlFor={k}>{l}</label>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>

          <div className="col-lg-4">
            <div className="form-card">
              <button className="btn btn-primary w-100" type="submit" disabled={saving}>
                {saving
                  ? <><span className="spinner-border spinner-border-sm me-2"></span>Saving...</>
                  : <><i className="bi bi-check-circle me-2"></i>Save Settings</>}
              </button>
              <p className="text-muted text-center mt-3 mb-0" style={{ fontSize: '.8rem' }}>
                Changes take effect immediately after saving.
              </p>
            </div>
          </div>
        </div>
      </form>
    </div>
  )
}
export default Settings
