import { useState, useEffect, useCallback } from 'react'
import toast from 'react-hot-toast'
import { adminGetReviews, adminApproveReview, adminDeleteReview } from '../services/api'
import Confirm from '../components/ui/Confirm'

const Stars = ({ rating }) => (
  <div className="text-warning">
    {[1, 2, 3, 4, 5].map(i => (
      <i key={i} className={`bi bi-star${i <= rating ? '-fill' : ''}`} style={{ fontSize: '.8rem' }}></i>
    ))}
  </div>
)

const Reviews = () => {
  const [reviews, setReviews] = useState({ data: [], total: 0, last_page: 1 })
  const [filters, setFilters] = useState({ status: '', rating: '', page: 1 })
  const [loading, setLoading] = useState(true)
  const [confirm, setConfirm] = useState(null)
  const [toggling, setToggling] = useState(null)

  const load = useCallback(async () => {
    setLoading(true)
    try {
      const r = await adminGetReviews(filters)
      setReviews(r.data)
    } catch { toast.error('Failed to load reviews') }
    finally { setLoading(false) }
  }, [filters])

  useEffect(() => { load() }, [load])

  const handleApprove = async (id) => {
    setToggling(id)
    try {
      await adminApproveReview(id)
      toast.success('Review status updated')
      load()
    } catch { toast.error('Update failed') }
    finally { setToggling(null) }
  }

  const handleDelete = async (id) => {
    try {
      await adminDeleteReview(id)
      toast.success('Review deleted')
      load()
    } catch { toast.error('Delete failed') }
    setConfirm(null)
  }

  return (
    <div>
      <Confirm show={!!confirm} message="Delete this review permanently?" onConfirm={() => handleDelete(confirm)} onCancel={() => setConfirm(null)} />

      <div className="d-flex align-items-center justify-content-between mb-4">
        <div>
          <h4 className="fw-bold mb-1">Reviews</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>{reviews.total} reviews</p>
        </div>
      </div>

      <div className="page-card mb-4 p-3">
        <div className="row g-2">
          <div className="col-md-4">
            <select className="form-select" value={filters.status} onChange={e => setFilters(f => ({ ...f, status: e.target.value, page: 1 }))}>
              <option value="">All Reviews</option>
              <option value="approved">Approved</option>
              <option value="pending">Pending Approval</option>
            </select>
          </div>
          <div className="col-md-3">
            <select className="form-select" value={filters.rating} onChange={e => setFilters(f => ({ ...f, rating: e.target.value, page: 1 }))}>
              <option value="">All Ratings</option>
              {[5, 4, 3, 2, 1].map(r => <option key={r} value={r}>{r} Stars</option>)}
            </select>
          </div>
          <div className="col-md-2">
            <button className="btn btn-outline-secondary w-100" onClick={() => setFilters({ status: '', rating: '', page: 1 })}>Reset</button>
          </div>
        </div>
      </div>

      <div className="page-card">
        {loading ? (
          <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>
        ) : (reviews.data || []).length === 0 ? (
          <div className="empty-state">
            <i className="bi bi-star d-block mb-2" style={{ fontSize: '2rem', color: '#adb5bd' }}></i>
            <p>No reviews found</p>
          </div>
        ) : (
          <div className="table-responsive">
            <table className="admin-table">
              <thead>
                <tr><th>Product</th><th>Customer</th><th>Rating</th><th>Review</th><th>Status</th><th>Date</th><th>Actions</th></tr>
              </thead>
              <tbody>
                {(reviews.data || []).map(r => (
                  <tr key={r.id}>
                    <td>
                      <div style={{ fontWeight: 600, fontSize: '.875rem', maxWidth: 160 }} className="text-truncate">
                        {r.product?.name || '—'}
                      </div>
                    </td>
                    <td>
                      <div style={{ fontSize: '.875rem' }}>{r.user?.name || 'Guest'}</div>
                      <div className="text-muted" style={{ fontSize: '.75rem' }}>{r.user?.email || '—'}</div>
                    </td>
                    <td><Stars rating={r.rating} /></td>
                    <td style={{ maxWidth: 280 }}>
                      {r.title && <div style={{ fontWeight: 600, fontSize: '.875rem' }}>{r.title}</div>}
                      <div className="text-muted" style={{ fontSize: '.8rem' }}>{r.body ? (r.body.length > 100 ? r.body.slice(0, 100) + '...' : r.body) : '—'}</div>
                    </td>
                    <td>
                      <span className={`badge-status badge-${r.is_approved ? 'active' : 'pending'}`}>
                        {r.is_approved ? 'Approved' : 'Pending'}
                      </span>
                    </td>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{new Date(r.created_at).toLocaleDateString()}</td>
                    <td>
                      <div className="action-btns">
                        <button
                          className={`btn btn-sm ${r.is_approved ? 'btn-outline-warning' : 'btn-outline-success'}`}
                          onClick={() => handleApprove(r.id)}
                          disabled={toggling === r.id}
                          title={r.is_approved ? 'Unapprove' : 'Approve'}
                        >
                          {toggling === r.id
                            ? <span className="spinner-border spinner-border-sm"></span>
                            : <i className={`bi ${r.is_approved ? 'bi-eye-slash' : 'bi-check-lg'}`}></i>}
                        </button>
                        <button className="btn btn-sm btn-outline-danger" onClick={() => setConfirm(r.id)} title="Delete">
                          <i className="bi bi-trash"></i>
                        </button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
        {reviews.last_page > 1 && (
          <div className="d-flex justify-content-center py-3">
            <div className="admin-pagination">
              <button disabled={filters.page <= 1} onClick={() => setFilters(f => ({ ...f, page: f.page - 1 }))}><i className="bi bi-chevron-left"></i></button>
              {Array.from({ length: reviews.last_page }, (_, i) => i + 1).map(p => (
                <button key={p} className={filters.page === p ? 'active' : ''} onClick={() => setFilters(f => ({ ...f, page: p }))}>{p}</button>
              ))}
              <button disabled={filters.page >= reviews.last_page} onClick={() => setFilters(f => ({ ...f, page: f.page + 1 }))}><i className="bi bi-chevron-right"></i></button>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}
export default Reviews
