import { useState, useEffect, useCallback } from 'react'
import { Link, useNavigate } from 'react-router-dom'
import toast from 'react-hot-toast'
import { adminGetProducts, adminDeleteProduct, adminGetCategories } from '../services/api'
import Confirm from '../components/ui/Confirm'

const Products = () => {
  const [products, setProducts] = useState({ data: [], total: 0, current_page: 1, last_page: 1 })
  const [categories, setCategories] = useState([])
  const [filters, setFilters] = useState({ search: '', category_id: '', status: '', page: 1 })
  const [loading, setLoading] = useState(true)
  const [confirm, setConfirm] = useState(null)
  const navigate = useNavigate()

  const load = useCallback(async () => {
    setLoading(true)
    try {
      const r = await adminGetProducts(filters)
      setProducts(r.data)
    } catch { toast.error('Failed to load products') }
    finally { setLoading(false) }
  }, [filters])

  useEffect(() => { load() }, [load])
  useEffect(() => { adminGetCategories().then(r => setCategories(r.data)).catch(() => {}) }, [])

  const handleDelete = async (id) => {
    try {
      await adminDeleteProduct(id)
      toast.success('Product deleted')
      load()
    } catch { toast.error('Delete failed') }
    setConfirm(null)
  }

  const setFilter = (key, val) => setFilters(f => ({ ...f, [key]: val, page: 1 }))

  return (
    <div>
      <Confirm show={!!confirm} message="Delete this product? This cannot be undone." onConfirm={() => handleDelete(confirm)} onCancel={() => setConfirm(null)} />

      <div className="d-flex align-items-center justify-content-between mb-4">
        <div>
          <h4 className="fw-bold mb-1">Products</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>{products.total} total products</p>
        </div>
        <Link to="/products/new" className="btn btn-primary">
          <i className="bi bi-plus-circle me-2"></i>Add Product
        </Link>
      </div>

      <div className="page-card mb-4 p-3">
        <div className="row g-2">
          <div className="col-md-5">
            <div style={{ position: 'relative' }}>
              <i className="bi bi-search" style={{ position: 'absolute', left: 12, top: '50%', transform: 'translateY(-50%)', color: '#adb5bd' }}></i>
              <input className="form-control" style={{ paddingLeft: 36 }} placeholder="Search by name or SKU..."
                value={filters.search} onChange={e => setFilter('search', e.target.value)} />
            </div>
          </div>
          <div className="col-md-3">
            <select className="form-select" value={filters.category_id} onChange={e => setFilter('category_id', e.target.value)}>
              <option value="">All Categories</option>
              {categories.map(c => <option key={c.id} value={c.id}>{c.name}</option>)}
            </select>
          </div>
          <div className="col-md-2">
            <select className="form-select" value={filters.status} onChange={e => setFilter('status', e.target.value)}>
              <option value="">All Status</option>
              <option value="active">Active</option>
              <option value="inactive">Inactive</option>
            </select>
          </div>
          <div className="col-md-2">
            <button className="btn btn-outline-secondary w-100" onClick={() => setFilters({ search: '', category_id: '', status: '', page: 1 })}>
              Reset
            </button>
          </div>
        </div>
      </div>

      <div className="page-card">
        {loading ? (
          <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>
        ) : products.data.length === 0 ? (
          <div className="empty-state">
            <i className="bi bi-box-seam d-block mb-2" style={{ fontSize: '2rem', color: '#adb5bd' }}></i>
            <p>No products found</p>
            <Link to="/products/new" className="btn btn-primary btn-sm">Add First Product</Link>
          </div>
        ) : (
          <div className="table-responsive">
            <table className="admin-table">
              <thead>
                <tr>
                  <th>#</th><th>Product</th><th>Category</th><th>Price</th>
                  <th>Stock</th><th>Status</th><th>Actions</th>
                </tr>
              </thead>
              <tbody>
                {products.data.map(p => (
                  <tr key={p.id}>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{p.id}</td>
                    <td>
                      <div className="d-flex align-items-center gap-3">
                        <img
                          src={p.primary_image?.url || p.images?.[0]?.url || 'https://via.placeholder.com/44x44'}
                          className="product-thumb" alt=""
                          onError={e => { e.target.src = 'https://via.placeholder.com/44x44' }}
                        />
                        <div>
                          <div style={{ fontWeight: 600, fontSize: '.875rem' }}>{p.name}</div>
                          <div className="text-muted" style={{ fontSize: '.75rem' }}>SKU: {p.sku || '—'} · {p.brand || '—'}</div>
                        </div>
                      </div>
                    </td>
                    <td><span className="badge bg-light text-dark">{p.category?.name || '—'}</span></td>
                    <td>
                      <div className="fw-semibold">${parseFloat(p.price).toFixed(2)}</div>
                      {p.sale_price && <div className="text-danger" style={{ fontSize: '.75rem' }}>Sale: ${parseFloat(p.sale_price).toFixed(2)}</div>}
                    </td>
                    <td>
                      <span className={p.stock > 0 ? 'text-success fw-semibold' : 'text-danger fw-semibold'}>
                        {p.stock}
                      </span>
                    </td>
                    <td>
                      <div className="d-flex flex-column gap-1">
                        <span className={`badge-status badge-${p.is_active ? 'active' : 'inactive'}`}>
                          {p.is_active ? 'Active' : 'Inactive'}
                        </span>
                        {p.is_featured && <span className="badge bg-warning text-dark" style={{ fontSize: '.65rem' }}>Featured</span>}
                        {p.is_deal && <span className="badge bg-danger" style={{ fontSize: '.65rem' }}>Deal</span>}
                      </div>
                    </td>
                    <td>
                      <div className="action-btns">
                        <button className="btn btn-sm btn-outline-primary" onClick={() => navigate(`/products/${p.id}/edit`)} title="Edit">
                          <i className="bi bi-pencil"></i>
                        </button>
                        <button className="btn btn-sm btn-outline-danger" onClick={() => setConfirm(p.id)} title="Delete">
                          <i className="bi bi-trash"></i>
                        </button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
        {products.last_page > 1 && (
          <div className="d-flex justify-content-center py-3">
            <div className="admin-pagination">
              <button disabled={filters.page <= 1} onClick={() => setFilters(f => ({ ...f, page: f.page - 1 }))}>
                <i className="bi bi-chevron-left"></i>
              </button>
              {Array.from({ length: products.last_page }, (_, i) => i + 1).map(p => (
                <button key={p} className={filters.page === p ? 'active' : ''} onClick={() => setFilters(f => ({ ...f, page: p }))}>
                  {p}
                </button>
              ))}
              <button disabled={filters.page >= products.last_page} onClick={() => setFilters(f => ({ ...f, page: f.page + 1 }))}>
                <i className="bi bi-chevron-right"></i>
              </button>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}
export default Products
