import { useState, useEffect } from 'react'
import { useNavigate, useParams } from 'react-router-dom'
import toast from 'react-hot-toast'
import { adminGetProduct, adminCreateProduct, adminUpdateProduct, adminGetCategories, adminGetVendors } from '../services/api'
import ImageUpload from '../components/ui/ImageUpload'

const ProductForm = () => {
  const { id } = useParams()
  const navigate = useNavigate()
  const isEdit = !!id

  const [form, setForm] = useState({
    name: '', category_id: '', vendor_id: '', price: '', sale_price: '', stock: '0',
    sku: '', brand: '', short_description: '', description: '',
    is_featured: false, is_deal: false, is_new: true, is_best_seller: false, is_active: true,
    images: [{ key: Date.now(), url: '' }],
  })
  const [categories, setCategories] = useState([])
  const [vendors, setVendors] = useState([])
  const [loading, setLoading] = useState(false)
  const [saving, setSaving] = useState(false)
  const [errors, setErrors] = useState({})

  useEffect(() => {
    adminGetCategories().then(r => setCategories(r.data)).catch(() => {})
    adminGetVendors().then(r => setVendors(r.data.data || r.data)).catch(() => {})
    if (isEdit) {
      setLoading(true)
      adminGetProduct(id).then(r => {
        const p = r.data
        setForm({
          name: p.name || '', category_id: p.category_id || '', vendor_id: p.vendor_id || '',
          price: p.price || '', sale_price: p.sale_price || '', stock: p.stock ?? 0,
          sku: p.sku || '', brand: p.brand || '',
          short_description: p.short_description || '', description: p.description || '',
          is_featured: !!p.is_featured, is_deal: !!p.is_deal, is_new: !!p.is_new,
          is_best_seller: !!p.is_best_seller, is_active: !!p.is_active,
          images: p.images?.length
            ? p.images.map((img, i) => ({ key: Date.now() + i, url: img.url || img }))
            : [{ key: Date.now(), url: '' }],
        })
      }).catch(() => toast.error('Failed to load product')).finally(() => setLoading(false))
    }
  }, [id, isEdit])

  const set = (k, v) => setForm(f => ({ ...f, [k]: v }))
  const setImg = (i, v) => setForm(f => { const imgs = [...f.images]; imgs[i] = { ...imgs[i], url: v }; return { ...f, images: imgs } })
  const addImg = () => setForm(f => ({ ...f, images: [...f.images, { key: Date.now(), url: '' }] }))
  const removeImg = (i) => setForm(f => ({ ...f, images: f.images.filter((_, idx) => idx !== i) }))


  const handleSubmit = async (e) => {
    e.preventDefault()
    setSaving(true)
    setErrors({})
    try {
      const payload = { ...form, images: form.images.map(img => img.url).filter(u => u.trim()) }
      if (isEdit) await adminUpdateProduct(id, payload)
      else await adminCreateProduct(payload)
      toast.success(isEdit ? 'Product updated!' : 'Product created!')
      navigate('/products')
    } catch (err) {
      const errs = err.response?.data?.errors || {}
      setErrors(errs)
      toast.error(err.response?.data?.message || 'Save failed')
    } finally { setSaving(false) }
  }

  if (loading) return <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>

  return (
    <div>
      <div className="d-flex align-items-center gap-3 mb-4">
        <button className="btn btn-outline-secondary btn-sm" onClick={() => navigate('/products')}>
          <i className="bi bi-arrow-left"></i>
        </button>
        <div>
          <h4 className="fw-bold mb-0">{isEdit ? 'Edit Product' : 'New Product'}</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>{isEdit ? 'Update product details' : 'Add a new product to the catalog'}</p>
        </div>
      </div>

      <form onSubmit={handleSubmit}>
        <div className="row">
          <div className="col-lg-8">
            <div className="form-card mb-4">
              <div className="form-section-title">Basic Information</div>
              <div className="mb-3">
                <label className="form-label fw-semibold">Product Name *</label>
                <input className={`form-control ${errors.name ? 'is-invalid' : ''}`} value={form.name}
                  onChange={e => set('name', e.target.value)} placeholder="e.g. Apple iPhone 15 Pro" required />
                {errors.name && <div className="invalid-feedback">{errors.name[0]}</div>}
              </div>
              <div className="row g-3">
                <div className="col-md-6">
                  <label className="form-label fw-semibold">Category</label>
                  <select className="form-select" value={form.category_id} onChange={e => set('category_id', e.target.value)}>
                    <option value="">Select category...</option>
                    {categories.map(c => <option key={c.id} value={c.id}>{c.name}</option>)}
                  </select>
                </div>
                <div className="col-md-6">
                  <label className="form-label fw-semibold">Vendor</label>
                  <select className="form-select" value={form.vendor_id} onChange={e => set('vendor_id', e.target.value)}>
                    <option value="">Select vendor...</option>
                    {vendors.map(v => <option key={v.id} value={v.id}>{v.store_name}</option>)}
                  </select>
                </div>
              </div>
            </div>

            <div className="form-card mb-4">
              <div className="form-section-title">Pricing & Inventory</div>
              <div className="row g-3">
                <div className="col-md-4">
                  <label className="form-label fw-semibold">Price *</label>
                  <div className="input-group">
                    <span className="input-group-text">$</span>
                    <input className="form-control" type="number" step="0.01" min="0" value={form.price}
                      onChange={e => set('price', e.target.value)} required />
                  </div>
                </div>
                <div className="col-md-4">
                  <label className="form-label fw-semibold">Sale Price</label>
                  <div className="input-group">
                    <span className="input-group-text">$</span>
                    <input className="form-control" type="number" step="0.01" min="0" value={form.sale_price}
                      onChange={e => set('sale_price', e.target.value)} placeholder="Optional" />
                  </div>
                </div>
                <div className="col-md-4">
                  <label className="form-label fw-semibold">Stock</label>
                  <input className="form-control" type="number" min="0" value={form.stock}
                    onChange={e => set('stock', e.target.value)} />
                </div>
                <div className="col-md-6">
                  <label className="form-label fw-semibold">SKU</label>
                  <input className="form-control" value={form.sku} onChange={e => set('sku', e.target.value)} placeholder="e.g. APPL-IP15-256" />
                </div>
                <div className="col-md-6">
                  <label className="form-label fw-semibold">Brand</label>
                  <input className="form-control" value={form.brand} onChange={e => set('brand', e.target.value)} placeholder="e.g. Apple" />
                </div>
              </div>
            </div>

            <div className="form-card mb-4">
              <div className="form-section-title">Description</div>
              <div className="mb-3">
                <label className="form-label fw-semibold">Short Description</label>
                <textarea className="form-control" rows={2} value={form.short_description}
                  onChange={e => set('short_description', e.target.value)} placeholder="Brief product summary..." />
              </div>
              <div>
                <label className="form-label fw-semibold">Full Description</label>
                <textarea className="form-control" rows={6} value={form.description}
                  onChange={e => set('description', e.target.value)} placeholder="Full product description..." />
              </div>
            </div>

            <div className="form-card mb-4">
              <div className="d-flex align-items-center justify-content-between mb-3">
                <div className="form-section-title mb-0">Product Images</div>
                <button type="button" className="btn btn-sm btn-outline-primary" onClick={addImg}>
                  <i className="bi bi-plus me-1"></i>Add Image
                </button>
              </div>

              {form.images.map((img, i) => (
                <div key={img.key} className="mb-3 p-3 border rounded" style={{ background: '#fafafa' }}>
                  <div className="d-flex align-items-center gap-2 mb-2">
                    <span className="badge bg-secondary">{i === 0 ? 'Primary' : `Image ${i + 1}`}</span>
                  </div>
                  <ImageUpload
                    value={img.url}
                    onChange={v => setImg(i, v)}
                    onRemove={form.images.length > 1 ? () => removeImg(i) : undefined}
                    height={140}
                  />
                </div>
              ))}

              <small className="text-muted">
                <i className="bi bi-info-circle me-1"></i>
                First image is the primary/thumbnail.
              </small>
            </div>
          </div>

          <div className="col-lg-4">
            <div className="form-card mb-4">
              <div className="form-section-title">Status</div>
              <div className="form-check form-switch mb-3">
                <input className="form-check-input" type="checkbox" checked={form.is_active}
                  onChange={e => set('is_active', e.target.checked)} id="is_active" />
                <label className="form-check-label" htmlFor="is_active">Active (visible on store)</label>
              </div>
              <hr />
              <div className="form-section-title">Labels</div>
              {[['is_featured', 'Featured'], ['is_deal', 'Deal / Sale'], ['is_new', 'New Arrival'], ['is_best_seller', 'Best Seller']].map(([k, l]) => (
                <div key={k} className="form-check form-switch mb-2">
                  <input className="form-check-input" type="checkbox" checked={form[k]}
                    onChange={e => set(k, e.target.checked)} id={k} />
                  <label className="form-check-label" htmlFor={k}>{l}</label>
                </div>
              ))}
            </div>

            <div className="form-card">
              <button className="btn btn-primary w-100" type="submit" disabled={saving}>
                {saving
                  ? <><span className="spinner-border spinner-border-sm me-2"></span>Saving...</>
                  : <><i className={`bi ${isEdit ? 'bi-check-circle' : 'bi-plus-circle'} me-2`}></i>{isEdit ? 'Update Product' : 'Create Product'}</>}
              </button>
              <button type="button" className="btn btn-outline-secondary w-100 mt-2" onClick={() => navigate('/products')}>
                Cancel
              </button>
            </div>
          </div>
        </div>
      </form>
    </div>
  )
}
export default ProductForm
