import { useState, useEffect, useCallback } from 'react'
import { Link, useSearchParams } from 'react-router-dom'
import toast from 'react-hot-toast'
import { adminGetOrders } from '../services/api'

const STATUS_OPTS = ['', 'pending', 'processing', 'shipped', 'delivered', 'cancelled', 'refunded']

const Orders = () => {
  const [searchParams] = useSearchParams()
  const [orders, setOrders] = useState({ data: [], total: 0, current_page: 1, last_page: 1 })
  const [filters, setFilters] = useState({ search: '', status: searchParams.get('status') || '', page: 1 })
  const [loading, setLoading] = useState(true)

  const load = useCallback(async () => {
    setLoading(true)
    try {
      const r = await adminGetOrders(filters)
      setOrders(r.data)
    } catch { toast.error('Failed to load orders') }
    finally { setLoading(false) }
  }, [filters])

  useEffect(() => { load() }, [load])

  const setFilter = (k, v) => setFilters(f => ({ ...f, [k]: v, page: 1 }))

  return (
    <div>
      <div className="d-flex align-items-center justify-content-between mb-4">
        <div>
          <h4 className="fw-bold mb-1">Orders</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>{orders.total} total orders</p>
        </div>
      </div>

      <div className="page-card mb-4 p-3">
        <div className="row g-2">
          <div className="col-md-6">
            <div style={{ position: 'relative' }}>
              <i className="bi bi-search" style={{ position: 'absolute', left: 12, top: '50%', transform: 'translateY(-50%)', color: '#adb5bd' }}></i>
              <input className="form-control" style={{ paddingLeft: 36 }} placeholder="Search by order # or customer..."
                value={filters.search} onChange={e => setFilter('search', e.target.value)} />
            </div>
          </div>
          <div className="col-md-4">
            <select className="form-select" value={filters.status} onChange={e => setFilter('status', e.target.value)}>
              {STATUS_OPTS.map(s => (
                <option key={s} value={s}>{s ? s.charAt(0).toUpperCase() + s.slice(1) : 'All Status'}</option>
              ))}
            </select>
          </div>
          <div className="col-md-2">
            <button className="btn btn-outline-secondary w-100" onClick={() => setFilters({ search: '', status: '', page: 1 })}>
              Reset
            </button>
          </div>
        </div>
      </div>

      <div className="page-card">
        {loading ? (
          <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>
        ) : orders.data.length === 0 ? (
          <div className="empty-state">
            <i className="bi bi-cart-x d-block mb-2" style={{ fontSize: '2rem', color: '#adb5bd' }}></i>
            <p>No orders found</p>
          </div>
        ) : (
          <div className="table-responsive">
            <table className="admin-table">
              <thead>
                <tr><th>Order #</th><th>Customer</th><th>Items</th><th>Total</th><th>Payment</th><th>Status</th><th>Date</th><th></th></tr>
              </thead>
              <tbody>
                {orders.data.map(o => (
                  <tr key={o.id}>
                    <td>
                      <Link to={`/orders/${o.id}`} className="text-primary fw-semibold">{o.order_number}</Link>
                    </td>
                    <td>
                      <div style={{ fontWeight: 600, fontSize: '.875rem' }}>{o.user?.name || o.shipping_name}</div>
                      <div className="text-muted" style={{ fontSize: '.75rem' }}>{o.user?.email || o.shipping_email}</div>
                    </td>
                    <td><span className="badge bg-light text-dark">{o.items_count || o.items?.length || 0}</span></td>
                    <td className="fw-bold">${parseFloat(o.total).toFixed(2)}</td>
                    <td><span className={`badge-status badge-${o.payment_status}`}>{o.payment_status}</span></td>
                    <td><span className={`badge-status badge-${o.status}`}>{o.status}</span></td>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{new Date(o.created_at).toLocaleDateString()}</td>
                    <td>
                      <Link to={`/orders/${o.id}`} className="btn btn-sm btn-outline-primary">
                        <i className="bi bi-eye"></i>
                      </Link>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
        {orders.last_page > 1 && (
          <div className="d-flex justify-content-center py-3">
            <div className="admin-pagination">
              <button disabled={filters.page <= 1} onClick={() => setFilters(f => ({ ...f, page: f.page - 1 }))}>
                <i className="bi bi-chevron-left"></i>
              </button>
              {Array.from({ length: Math.min(orders.last_page, 7) }, (_, i) => i + 1).map(p => (
                <button key={p} className={filters.page === p ? 'active' : ''} onClick={() => setFilters(f => ({ ...f, page: p }))}>
                  {p}
                </button>
              ))}
              <button disabled={filters.page >= orders.last_page} onClick={() => setFilters(f => ({ ...f, page: f.page + 1 }))}>
                <i className="bi bi-chevron-right"></i>
              </button>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}
export default Orders
